/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.processing.bpel.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.TreeSet;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.datamodel.ipc.IpcRecordReader;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.StoreOutputStream;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.bpel.WorkflowStorage;
import org.osgi.service.component.ComponentContext;

/**
 * Helper class to store workflow definitions in objectstore.
 * 
 * @author aweber
 */
public final class ObjectStoreWorkflowStorage implements WorkflowStorage {

  /** max retries for store requests on IOExceptions. */
  private static final int MAX_RETRY_ON_STORE_UNAVAILABLE = 3;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** for JSON->record parsing. */
  private final IpcRecordReader _recordReader = new IpcRecordReader();

  /** for Any->BON/JSON serialization. */
  private final IpcAnyWriter _anyWriter = new IpcAnyWriter();

  /** has store already been successfully prepared? */
  private boolean _isStorePrepared;

  /** store name. */
  private String _storeName;

  /** objectstore reference for storing the workflows. */
  private ObjectStoreService _objectStore;

  /** service reference bind method. */
  public void setObjectStore(final ObjectStoreService objectStore) {
    _objectStore = objectStore;
  }

  /** service reference unbind method. */
  public void unsetObjectStore(final ObjectStoreService objectStore) {
    if (_objectStore == objectStore) {
      _objectStore = null;
    }
  }

  /** for tests. */
  public void setStoreName(final String storeName) {
    _storeName = storeName;
  }

  /** declarative service activation method. */
  protected void activate(final ComponentContext context) {
    final Object storeNameProperty = context.getProperties().get("storeName");
    if (storeNameProperty == null) {
      _log.error("Store name property of WorkflowStorage service not set. Service will be disfunctional.");
    }
    _storeName = storeNameProperty.toString();
  }

  @Override
  public void setWorkflow(final String workflowName, final AnyMap workflow) throws ProcessingException {
    try {
      writeObjectToStore(workflowName, workflow);
    } catch (final ObjectStoreException e) {
      throw new ProcessingException("Error adding workflow '" + workflowName + "' to store", e);
    }
  }

  @Override
  public void deleteWorkflow(final String workflowName) throws ProcessingException {
    try {
      removeObjectFromStore(workflowName);
    } catch (final ObjectStoreException e) {
      throw new ProcessingException("Error deleting workflow '" + workflowName + "' to store", e);
    }
  }

  @Override
  public AnyMap getWorkflow(final String workflowName) throws ProcessingException {
    try {
      if (!existsObjectInStore(workflowName)) {
        return null;
      }
      return readObjectFromStore(workflowName);
    } catch (final ObjectStoreException e) {
      throw new ProcessingException("Error getting workflow '" + workflowName + "' from store", e);
    }
  }

  @Override
  public Collection<String> getWorkflowNames() throws ProcessingException {
    try {
      return readObjectNamesFromStore();
    } catch (final ObjectStoreException e) {
      throw new ProcessingException("Error getting workflow names from store", e);
    }
  }

  /**
   * @param objectId
   *          the object id under which the object is stored
   * @param object
   *          the object to store
   * @throws ObjectStoreException
   *           error writing object to the {@link ObjectStoreService}.
   */
  private void writeObjectToStore(final String objectId, final AnyMap object) throws ObjectStoreException {
    Exception retryableEx = null;
    ensureStore();
    for (int i = 0; i < MAX_RETRY_ON_STORE_UNAVAILABLE; i++) {
      StoreOutputStream storeOutputStream = null;
      try {
        storeOutputStream = _objectStore.writeObject(_storeName, objectId);
        _anyWriter.writeJsonStream(object, storeOutputStream);
        return;
      } catch (final IOException ex) {
        if (storeOutputStream != null) {
          storeOutputStream.abort();
        }
        retryableEx = ex;
        _log.warn("IOException on writing object '" + objectId + "', retrying: " + ex.toString());
      } catch (final ServiceUnavailableException ex) {
        retryableEx = ex;
        _log.warn("ServiceUnavailableException on writing object '" + objectId + "', retrying: " + ex.toString());
      } finally {
        IOUtils.closeQuietly(storeOutputStream);
      }
    }
    throw new ServiceUnavailableException("Finally failed to write object '" + objectId + "' to store", retryableEx);
  }

  /**
   * @return the object names found in the store
   * @throws ObjectStoreException
   *           could not read object information from the {@link ObjectStoreService}.
   */
  private Collection<String> readObjectNamesFromStore() throws ObjectStoreException {
    ensureStore();
    final Collection<String> objectNames = new TreeSet<String>();
    for (final StoreObject info : _objectStore.getStoreObjectInfos(_storeName)) {
      final String objectName = info.getId();
      objectNames.add(objectName);
    }
    return objectNames;
  }

  /**
   * @param objectId
   *          the object id of the object to read from store
   * @return the stored object for the given object id
   * @throws ObjectStoreException
   *           could not read object from the {@link ObjectStoreService}.
   */
  private AnyMap readObjectFromStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    Exception retryableEx = null;
    for (int i = 0; i < MAX_RETRY_ON_STORE_UNAVAILABLE; i++) {
      try {
        InputStream jobJsonStream = null;
        try {
          jobJsonStream = _objectStore.readObject(_storeName, objectId);
          return _recordReader.readJsonStream(jobJsonStream).getMetadata();
        } finally {
          IOUtils.closeQuietly(jobJsonStream);
        }
      } catch (final IOException ex) {
        retryableEx = ex;
        _log.warn("IOException on writing object '" + objectId + "', retrying: " + ex.toString());
      } catch (final ServiceUnavailableException ex) {
        retryableEx = ex;
        _log.warn("ServiceUnavailableException on writing object '" + objectId + "', retrying: " + ex.toString());
      }
    }
    throw new ServiceUnavailableException("Finally failed to read object '" + objectId + "' from store",
      retryableEx);
  }

  /**
   * Removes an object from the store.
   * 
   * @param objectId
   *          the id of the object
   * @throws ObjectStoreException
   *           object could not be removed
   */
  private boolean existsObjectInStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    return ObjectStoreRetryUtil.retryExistsObject(_objectStore, _storeName, objectId);
  }

  /**
   * Removes an object from the store.
   * 
   * @param objectId
   *          the id of the object
   * @throws ObjectStoreException
   *           object could not be removed
   */
  private void removeObjectFromStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    _objectStore.removeObject(_storeName, objectId);
  }

  /**
   * Makes sure the store where the workflow data is stored, does exist and creates it with the store properties
   * retrieved from {@link ClusterConfigService} if it does not exist, yet.
   * 
   * @throws ObjectStoreException
   *           the store could not be created.
   * 
   */
  private void ensureStore() throws ObjectStoreException {
    if (!_isStorePrepared) {
      try {
        synchronized (this) {
          ObjectStoreRetryUtil.retryEnsureStore(_objectStore, _storeName);
          _isStorePrepared = true;
        }
      } catch (final ServiceUnavailableException ex) {
        throw new ServiceUnavailableException("Finally failed to prepare store '" + _storeName + "'", ex);
      }
    }
  }
}
