/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.net.MalformedURLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;

/**
 * Implements the handling of HTTP requests to request the job list or to define new jobs. <br>
 * 
 * URL pattern: <code>smila/jobmanager/jobs/</code> <br>
 * 
 * Methods allowed: <code>GET, POST</code>
 */
public class JobsHandler extends AJobManagerHandler {

  /** The key for the details. */
  public static final String KEY_RETURN_DETAILS = "returnDetails";

  /** job run data of the most current job run. */
  public static final String KEY_LATEST_JOB_RUN = "latestJobRun";

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final DefinitionPersistence defPersistence = getDefinitionPersistence();
    if (isGetRequest(method)) {
      final AnyMap resultMap = DataFactory.DEFAULT.createAnyMap();
      final AnySeq jobList = DataFactory.DEFAULT.createAnySeq();
      final boolean showDetails;
      if (inputRecord != null && inputRecord.getMetadata().containsKey(KEY_RETURN_DETAILS)) {
        showDetails = inputRecord.getMetadata().getBooleanValue(KEY_RETURN_DETAILS).booleanValue();
      } else {
        showDetails = true;
      }
      for (final String jobName : defPersistence.getJobs()) {
        final AnyMap map = DataFactory.DEFAULT.createAnyMap();
        map.put("name", jobName);
        map.put("url", URLCreator.create(getRequestHost(), requestUri, jobName).toString());
        if (showDetails) {
          final AnyMap jobDetails = getJobDetails(jobName, requestUri);
          if (!jobDetails.isEmpty()) {
            map.put(KEY_LATEST_JOB_RUN, jobDetails);
          }
        }
        jobList.add(map);
      }
      resultMap.put(JobDefinition.KEY_JOBS, jobList);
      return resultMap;
    }
    if (isPostRequest(method)) { // POST
      if (inputRecord == null) {
        throw new IllegalArgumentException("Missing input, cannot create jobs from nothing.");
      }
      final AnyMap definition = inputRecord.getMetadata();
      // create timestamp for this workflow and store it with the data
      final Any timestamp = inputRecord.getFactory().createDateTimeValue(new Date());
      definition.put(DefinitionBase.KEY_TIMESTAMP, timestamp);
      // remove readOnly flag if present.
      definition.remove(DefinitionBase.KEY_READ_ONLY);
      final JobDefinition createdJob = JobDefinition.parseJob(definition);
      defPersistence.addJob(createdJob);
      final AnyMap result = DataFactory.DEFAULT.createAnyMap();
      result.put("name", createdJob.getName());
      result.put(JobManagerConstants.TIMESTAMP, timestamp);
      result.put("url", URLCreator.create(getRequestHost(), requestUri, createdJob.getName()).toExternalForm());
      return result;
    }
    return null;
  }

  /**
   * retrieves the job details.
   * <ul>
   * <li>status</li>
   * <li>startTime</li>
   * <li>endTime</li>
   * <li>successfulTaskCount</li>
   * <li>successfulWorkflowRunCount</li>
   * <li>failedAfterRetryTaskCount</li>
   * <li>failedWithoutRetryTaskCount</li>
   * <li>failedWorkflowRunCount</li>
   * </ul>
   * 
   * @param jobName
   *          the name of the job
   * @return a map containing key and value for the above keys and the given job.
   * @throws JobManagerException
   *           error while retrieving job run info.
   * @throws MalformedURLException
   *           cannot build the job run uri.
   */
  private AnyMap getJobDetails(final String jobName, final String requestUri) throws JobManagerException,
    MalformedURLException {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    final JobRunInfo currentRun = getJobRunDataProvider().getJobRunInfo(jobName);
    String jobRunId = null;
    if (currentRun == null) {
      // no current run, get the latest job run id:
      final Collection<String> completedJobRunIds = getJobRunDataProvider().getCompletedJobRunIds(jobName);
      if (!completedJobRunIds.isEmpty()) {
        // get the last one, it is the latest job run id
        final Iterator<String> iter = completedJobRunIds.iterator();
        while (iter.hasNext()) {
          jobRunId = iter.next();
        }
      }
    } else {
      jobRunId = currentRun.getId();
    }

    // did we get a job run id?
    if (jobRunId != null) {
      final AnyMap jobRunData = getJobRunDataProvider().getJobRunData(jobName, jobRunId);
      safeCopyValues(jobRunData, map, Arrays.asList(JobManagerConstants.DATA_JOB_RUN_MODE,
        JobManagerConstants.DATA_JOB_STATE, JobManagerConstants.DATA_JOB_RUN_START_TIME,
        JobManagerConstants.DATA_JOB_RUN_END_TIME));
      if (jobRunData.containsKey(JobManagerConstants.WORKFLOW_RUN_COUNTER)) {
        final AnyMap workflowCounters = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
        safeCopyValues(workflowCounters, map, Arrays.asList(
          JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS,
          JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));
      }
      if (jobRunData.containsKey(JobManagerConstants.TASK_COUNTER)) {
        final AnyMap taskCounters = jobRunData.getMap(JobManagerConstants.TASK_COUNTER);
        safeCopyValues(taskCounters, map, Arrays.asList(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS,
          JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_NOT_RETRIED,
          JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_RETRIED));
      }
      map.put("url", URLCreator.create(getRequestHost(), requestUri, jobName + "/" + jobRunId).toString());
    }
    return map;
  }

  /**
   * Copies a range of values from one map to another (if it existed in the source map in the first place.
   * 
   * @param sourceMap
   *          the source map
   * @param destinationMap
   *          the destination map
   * @param keys
   *          the keys which values are to be copied.
   */
  private void safeCopyValues(final AnyMap sourceMap, final AnyMap destinationMap, final Collection<String> keys) {
    for (final String key : keys) {
      if (sourceMap.containsKey(key)) {
        destinationMap.put(key, sourceMap.get(key));
      }
    }
  }

  /**
   * result status for POST requests should be CREATED.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    if (isPostRequest(method)) {
      return HttpStatus.CREATED;
    }
    return super.getSuccessStatus(method, requestUri);
  }

  /**
   * {@inheritDoc}
   * 
   * GET and POST are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return isGetOrPostRequest(method);
  }

}
