/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.definitions.ParameterDefinition;
import org.eclipse.smila.common.definitions.ValueExpression;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition.Mode;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Input;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.InputOutput;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Output;
import org.eclipse.smila.jobmanager.definitions.WorkflowAction;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.InvalidConfigException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerator;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorProvider;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskworker.output.OutputMode;

/**
 * Helper class for validating asynchronous workflow elements.
 */
public class DefinitionValidator {

  /** prefix of system variables. */
  private static final String VARIABLE_SYSTEM_PREFIX = "_";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** for reading definitions from storage. */
  private final DefinitionAccessor _definitionAccessor;

  /** we need access to worker specific task generators because they can provide parameter validation. */
  private final TaskGeneratorProvider _taskGeneratorProvider;

  /** reference to objectstore service. */
  private final ObjectStoreService _objectStoreService;

  /**
   * @param definitionAccessor
   *          accessor for reading user-submitted or configured definitions.
   * @param taskGeneratorProvider
   *          needed for accessing task generators needed for validation
   * @param objectStoreService
   *          the reference to the object store service.
   */
  public DefinitionValidator(final DefinitionAccessor definitionAccessor,
    final TaskGeneratorProvider taskGeneratorProvider, final ObjectStoreService objectStoreService) {
    _definitionAccessor = definitionAccessor;
    _taskGeneratorProvider = taskGeneratorProvider;
    _objectStoreService = objectStoreService;
  }

  /**
   * check the the workflow referenced by this job.
   * 
   * @param job
   *          job definition
   * @throws InvalidConfigException
   *           validation error.
   * @throws PersistenceException
   *           error reading definition(s).
   */
  public void validateJob(final JobDefinition job) throws InvalidConfigException, PersistenceException {
    // it's not allowed to update a predefined job
    final String jobName = job.getName();
    if (_definitionAccessor.getConfiguredJobs().containsKey(jobName)) {
      throw new InvalidConfigException("Job '" + jobName
        + "' cannot be added, because a preconfigured job definition with the same name already exists.");
    }
    final WorkflowDefinition workflow = _definitionAccessor.getWorkflow(job.getWorkflow());
    if (workflow == null) {
      throw new InvalidConfigException("Workflow '" + job.getWorkflow() + "' of job '" + job.getName()
        + "' is not defined.");
    }
    validateJob(job, workflow, new HashMap<String, BucketDefinition>());
  }

  /**
   * check the the given job and workflow definition definition.
   * 
   * @param job
   *          job definition to check
   * @param workflow
   *          workflow definition to check against
   * @param bucketCache
   *          the bucket cache
   * @throws InvalidConfigException
   *           validation error.
   * @throws PersistenceException
   *           error reading definition(s).
   */
  private void validateJob(final JobDefinition job, final WorkflowDefinition workflow,
    final Map<String, BucketDefinition> bucketCache) throws InvalidConfigException, PersistenceException {

    final List<JobRunMode> jobRunModes = job.getJobRunModes();
    if (jobRunModes != null) {
      final List<JobRunMode> workflowRunModes = workflow.getJobRunModes();
      if (workflowRunModes != null && !workflowRunModes.containsAll(jobRunModes)) {
        throw new InvalidConfigException("Job '" + job.getName() + "' allows job run modes (" + jobRunModes
          + ") that are not allowed by workflow '" + workflow.getName() + "' (" + workflowRunModes + ").");
      }
    }
    // check that all unresolved parameters are resolved in the job
    final Map<String, String> unresolvedParameters = getUnresolvedParameters(workflow, bucketCache);
    for (final Entry<String, String> unresolvedParameter : unresolvedParameters.entrySet()) {
      final List<Any> matchingEntries =
        getMatchingParameterEntries(job.getParameters(), unresolvedParameter.getKey());
      if (matchingEntries == null || matchingEntries.isEmpty()) {
        throw new InvalidConfigException("Parameter '" + unresolvedParameter.getKey() + "' ("
          + unresolvedParameter.getValue() + ") of workflow '" + job.getWorkflow()
          + "' is not resolved in job definition '" + job.getName() + "'.");
      }
    }

    // check for invalid store name generated by this jobs (and bucket's) parameters...
    try {
      checkValidStoreNameForWorkflow(workflow, job.getParameters(), bucketCache);
    } catch (final InvalidConfigException e) {
      Throwable cause = e;
      if (e != null && e.getCause() != null) {
        cause = e.getCause();
      }
      throw new InvalidConfigException("Parameters for job '" + job.getName() + "' and workflow '"
        + workflow.getName() + "' would result in invalid store name(s).", cause);
    }
    validateParameters(job, workflow);
  }

  /**
   * check if workflow definition is valid.
   * 
   * @param workflow
   *          workflow definition
   * @throws InvalidConfigException
   *           validation error.
   * @throws PersistenceException
   *           error accessing storage
   */
  public void validateWorkflow(final WorkflowDefinition workflow) throws InvalidConfigException,
    PersistenceException {
    // it's not allowed to update a predefined workflow
    final String workflowName = workflow.getName();
    if (_definitionAccessor.getConfiguredWorkflows().containsKey(workflowName)) {
      throw new InvalidConfigException("Workflow '" + workflowName
        + "' cannot be added, because a preconfigured workflow definition with the same name already exists.");
    }
    validateWorkflow(workflow, new HashMap<String, BucketDefinition>());
  }

  /**
   * check if workflow definition is valid.
   * 
   * @param workflow
   *          workflow definition
   * @param bucketCache
   *          the bucket cache
   * @throws InvalidConfigException
   *           validation error.
   * @throws PersistenceException
   *           error accessing storage
   */
  private void validateWorkflow(final WorkflowDefinition workflow, final Map<String, BucketDefinition> bucketCache)
    throws InvalidConfigException, PersistenceException {

    // if workflow already existed before, we have to re-validate all referencing jobs before overwriting it.
    // (if workflow doesn't exist yet, there can be not job defined referencing it)
    final String workflowName = workflow.getName();
    final Set<String> existingNames = new HashSet<String>(_definitionAccessor.getWorkflows());
    if (existingNames.contains(workflowName)) {
      final Collection<JobDefinition> jobDefs = getJobsForWorkflow(workflowName);
      for (final JobDefinition job : jobDefs) {
        try {
          validateJob(job, workflow, bucketCache);
        } catch (final InvalidConfigException ex) {
          throw new InvalidConfigException("Cannot update workflow '" + workflowName
            + "' because a job exists that would become invalid after the update: " + ex.getMessage());
        }
      }
    }

    final AnyMap workflowParameters = workflow.getParameters();

    // check for workers
    for (final String workerName : workflow.getReferencedWorkers()) {
      final WorkerDefinition workerDefinition = _definitionAccessor.getWorker(workerName);
      if (workerDefinition == null) {
        throw new InvalidConfigException("Referenced worker '" + workerName + "' of workflow '"
          + workflow.getName() + "' is not defined.");
      }
      validateWorkerParameterDefinition(workerDefinition, workflowParameters, false, workflowName, workerName);
    }
    // check if action buckets are properly defined
    checkWorkflowActionBuckets(workflow, bucketCache);

    // check for invalid Store Name generated by this workflow's (and bucket's) parameters...
    checkValidStoreNameForWorkflow(workflow, null, bucketCache);
  }

  /**
   * check if bucket definition is valid.
   * 
   * @param bucketDefinition
   *          bucket definition
   * @throws InvalidConfigException
   *           validation error.
   * @throws PersistenceException
   *           the bucket names cannot be retrieved.
   */
  public void validateBucket(final BucketDefinition bucketDefinition) throws InvalidConfigException,
    PersistenceException {
    final String bucketName = bucketDefinition.getName();
    final DataObjectTypeDefinition type =
      _definitionAccessor.getDataObjectType(bucketDefinition.getDataObjectType());
    if (type == null) {
      throw new InvalidConfigException("DataObjectType '" + bucketDefinition.getDataObjectType() + "' of bucket '"
        + bucketName + "' is not defined.");
    }
    // it's not allowed to update predefined buckets
    if (_definitionAccessor.getConfiguredBuckets().containsKey(bucketName)) {
      throw new InvalidConfigException("Bucket '" + bucketName
        + "' cannot be added, because a preconfigured bucket definition with the same name already exists.");
    }
    // check for invalid store name generated by this bucket's parameters...
    checkValidStoreNameForBucket(bucketDefinition.getName(), bucketDefinition, type, null);
  }

  /**
   * Check if the bucket can be safely added without compromising data integrity. I.e. if there is an existing workflow
   * definition that already references a bucket with the same name, it must still be valid after the update.
   * 
   * @param bucketDefinition
   *          The bucket to check
   * @throws PersistenceException
   *           error reading workflow definition(s)
   * @throws InvalidConfigException
   *           error adding definition due to dependency constraints
   */
  public void validateBucketWorkflows(final BucketDefinition bucketDefinition) throws PersistenceException,
    InvalidConfigException {
    final String bucketName = bucketDefinition.getName();
    final Map<String, BucketDefinition> bucketCache = new HashMap<String, BucketDefinition>();
    bucketCache.put(bucketName, bucketDefinition);
    final Collection<String> workflowNames = _definitionAccessor.getWorkflows();
    for (final String workflowName : workflowNames) {
      final WorkflowDefinition workflow = _definitionAccessor.getWorkflow(workflowName);
      if (workflow != null) {
        if (workflow.getReferencedBuckets().contains(bucketName)) {
          validateWorkflow(workflow, bucketCache); // this also validates all jobs referencing the workflow
        }
      }
    }
  }

  /**
   * @param jobDefinition
   *          The job definition
   * @throws InvalidConfigException
   *           an exception if validation failed
   * @throws PersistenceException
   *           if workflow not found
   */
  private void validateParameters(final JobDefinition jobDefinition, final WorkflowDefinition workflowDefinition)
    throws InvalidConfigException, PersistenceException {
    final AnyMap jobAndWorkflowParameters = DataFactory.DEFAULT.createAnyMap();
    if (jobDefinition.getParameters() != null) {
      jobAndWorkflowParameters.putAll(jobDefinition.getParameters());
    }
    if (workflowDefinition.getParameters() != null) {
      jobAndWorkflowParameters.putAll(workflowDefinition.getParameters());
    }
    final String workflowName = workflowDefinition.getName();
    validateActionParameters(workflowName, workflowDefinition.getStartAction(), jobAndWorkflowParameters);
    if (workflowDefinition.getActions() != null) {
      for (final WorkflowAction action : workflowDefinition.getActions()) {
        validateActionParameters(workflowName, action, jobAndWorkflowParameters);
      }
    }
  }

  /**
   * Validates the parameters for one action.
   * 
   * @param workflowAction
   *          The workflow action
   * @param jobAndWorkflowParameters
   *          The parameters from job and from workflow
   * @throws InvalidConfigException
   *           An exception if validation failed
   */
  private void validateActionParameters(final String workflowName, final WorkflowAction workflowAction,
    final AnyMap jobAndWorkflowParameters) throws InvalidConfigException {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    if (jobAndWorkflowParameters != null) {
      parameters.putAll(jobAndWorkflowParameters);
    }
    if (workflowAction.getParameters() != null) {
      parameters.putAll(workflowAction.getParameters());
    }
    final String workerName = workflowAction.getWorker();
    final WorkerDefinition workerDefinition = _definitionAccessor.getWorker(workerName);
    if (workerDefinition == null) {
      throw new InvalidConfigException("Unknown worker '" + workerName + "' used in workflow '" + workflowName
        + "'");
    }
    validateWorkerParameterDefinition(workerDefinition, parameters, true, workflowName, workerName);
    final TaskGenerator taskGenerator = _taskGeneratorProvider.getTaskGenerator(workerDefinition);
    final String taskGeneratorName = taskGenerator.getName();
    try {
      taskGenerator.validateParameters(ExpressionUtil.evaluateParameters(parameters));
    } catch (final TaskGeneratorException tge) {
      Throwable cause = tge;
      if (tge != null && tge.getCause() != null) {
        cause = tge.getCause();
      }
      throw new InvalidConfigException("Parameter validation failed for taskGenerator '" + taskGeneratorName
        + "' of worker '" + workerName + "' in workflow '" + workflowName + "'.", cause);
    }
  }

  /**
   * @param workerDefinition
   *          the worker definition to check
   * @param parameters
   *          the parameters to check against the worker's parameter description
   * @param allParametersResolved
   *          'true' if all parameters are resolved and also missing parameters must be reported, 'false' if parameters
   *          might still be resolved or incomplete.
   * @param workflowName
   *          the name of the workflow
   * @param workerName
   *          the name of the worker
   * @throws InvalidConfigException
   */
  private void validateWorkerParameterDefinition(final WorkerDefinition workerDefinition, final AnyMap parameters,
    final boolean allParametersResolved, final String workflowName, final String workerName)
    throws InvalidConfigException {
    final AnyMap evaluatedParameters = ExpressionUtil.evaluateParameters(parameters);
    if (workerDefinition.getParameters() != null) {
      if (evaluatedParameters != null) {
        for (final ParameterDefinition paramDef : workerDefinition.getParameters()) {
          final List<Any> matchingEntries = getMatchingParameterEntries(evaluatedParameters, paramDef.getName());
          if (matchingEntries.isEmpty()) {
            validateParameterDefinition(paramDef, null, allParametersResolved, workflowName, workerName);
          } else {
            for (final Any matchingParameter : matchingEntries) {
              validateParameterDefinition(paramDef, matchingParameter, allParametersResolved, workflowName,
                workerName);
            }
          }
        }
      } else if (allParametersResolved) {
        // all parameters should be resolved, the worker is expecting parameters but we don't have any:
        // throw an exception. This cannot be correct.
        throw new InvalidConfigException("Parameters are missing for workflow '" + workflowName + "'.");
      }
    }
  }

  /**
   * Validates a parameter against its definition.
   * 
   * @param paramDef
   *          the parameterDefinition to check with the given parameter
   * @param parameter
   *          the parameter to check against the worker's parameter description
   * @param allParametersResolved
   *          'true' if all parameters are resolved and also missing parameters must be reported, 'false' if parameters
   *          might still be resolved or incomplete.
   * @param workflowName
   *          the name of the workflow
   * @param workerName
   *          the name of the worker
   * @throws InvalidConfigException
   *           parameter invalid
   */
  private void validateParameterDefinition(final ParameterDefinition paramDef, final Any parameter,
    final boolean allParametersResolved, final String workflowName, final String workerName)
    throws InvalidConfigException {
    validateParameterUndefined(paramDef, parameter, allParametersResolved, workflowName, workerName);
    if (parameter == null) {
      return;
    }
    // not all parameters must be evaluated but this one contains a variable reference: ignore
    if (parameter.isString() && !allParametersResolved
      && new ValueExpression(parameter.asValue().asString()).referencesVariables()) {
      return;
    }
    // parameters here and not containing a variable: types must be correct.
    if (paramDef.isMulti()) {
      // SEQ
      // there may be more than just one...
      for (final Any param : parameter) {
        final AnyMap clonedParam = DataFactory.DEFAULT.cloneAnyMap(paramDef.toAny());
        clonedParam.remove(ParameterDefinition.KEY_MULTI);
        try {
          validateParameterDefinition(new ParameterDefinition(clonedParam), param, allParametersResolved,
            workflowName, workerName);
        } catch (final InvalidDefinitionException invalidDefinitionException) {
          throw new InvalidConfigException("Invalid parameter definition.", invalidDefinitionException);
        }
      }
    } else if (paramDef.getType() == ValueType.MAP) {
      // MAP
      validateParameterDefinitionForMap(paramDef, parameter, allParametersResolved, workflowName, workerName);
    } else {
      // all other types...
      validateParameterDefinitionForSimpleType(paramDef, parameter, workflowName, workerName);
    }
  }

  /**
   * Validates a MAP parameter against its definition.
   * 
   * @param paramDef
   *          the parameterDefinition to check with the given parameter
   * @param parameter
   *          the parameter to check against the worker's parameter description
   * @param allParametersResolved
   *          'true' if all parameters are resolved and also missing parameters must be reported, 'false' if parameters
   *          might still be resolved or incomplete.
   * @param workflowName
   *          the name of the workflow
   * @param workerName
   *          the name of the worker
   * @throws InvalidConfigException
   *           parameter invalid
   */
  protected void validateParameterDefinitionForMap(final ParameterDefinition paramDef, final Any parameter,
    final boolean allParametersResolved, final String workflowName, final String workerName)
    throws InvalidConfigException {
    if (parameter.isMap()) {
      // check every entry of the map
      for (final ParameterDefinition entryParamDef : paramDef.getEntries()) {
        final List<Any> matchingParameters =
          getMatchingParameterEntries(parameter.asMap(), entryParamDef.getName());
        if (matchingParameters != null && !matchingParameters.isEmpty()) {
          for (final Any matchingParameter : matchingParameters) {
            validateParameterDefinition(entryParamDef, matchingParameter, allParametersResolved, workflowName,
              workerName);
          }
        } else if (!entryParamDef.isOptional()) {
          throw new InvalidConfigException("Parameter '" + paramDef.getName() + "' of worker '" + workerName
            + "' in workflow '" + workflowName + "' is not defined.");
        }
      }
    } else {
      throw new InvalidConfigException("Parameter '" + paramDef.getName() + "' of worker '" + workerName
        + "' in workflow '" + workflowName + "' is not of type MAP.");
    }
  }

  /**
   * Validates that parameter is not undefined unless the definition allows so.
   * 
   * @param paramDef
   *          the parameterDefinition to check with the given parameter
   * @param parameter
   *          the parameter to check against the worker's parameter description
   * @param allParametersResolved
   *          'true' if all parameters are resolved and also missing parameters must be reported, 'false' if parameters
   *          might still be resolved or incomplete.
   * @param workflowName
   *          the name of the workflow
   * @param workerName
   *          the name of the worker
   * @throws InvalidConfigException
   *           parameter invalid
   */
  protected void validateParameterUndefined(final ParameterDefinition paramDef, final Any parameter,
    final boolean allParametersResolved, final String workflowName, final String workerName)
    throws InvalidConfigException {
    if (parameter == null) {
      if (paramDef.isOptional() || !allParametersResolved) {
        return;
      } else {
        // not optional but not defined: ignore when not all parameters must be present. otherwise: exception
        throw new InvalidConfigException("Parameter '" + paramDef.getName() + "' of worker '" + workerName
          + "' in workflow '" + workflowName + "' is missing.");
      }
    }
  }

  /**
   * Gets all matching parameter entries from a map of entries.
   * 
   * @param parameterMap
   *          the map of parameters.
   * @param name
   *          the name, possibly containing a variable part.
   * @return the list of matching parameters.
   */
  private List<Any> getMatchingParameterEntries(final AnyMap parameterMap, final String name) {
    final List<Any> matchingEntries = new ArrayList<Any>();
    if (parameterMap != null) {
      if (name.matches(".*" + ParameterDefinition.VARIABLE_NAME_EXPRESSION + ".*")) {
        final String regex = name.replaceAll(ParameterDefinition.VARIABLE_NAME_EXPRESSION, ".*");
        for (final Entry<String, Any> entry : parameterMap.entrySet()) {
          if (entry.getKey().matches(regex)) {
            matchingEntries.add(entry.getValue());
          }
        }
      } else {
        final Any entry = parameterMap.get(name);
        if (entry != null) {
          matchingEntries.add(entry);
        }
      }
    }
    return matchingEntries;
  }

  /**
   * @param paramDef
   *          the parameterDefinition to check with the given parameter
   * @param parameter
   *          the parameter to check against the worker's parameter description
   * @param workflowName
   *          the name of the workflow
   * @param workerName
   *          the name of the worker
   */
  protected void validateParameterDefinitionForSimpleType(final ParameterDefinition paramDef, final Any parameter,
    final String workflowName, final String workerName) throws InvalidConfigException {
    final List<Value> possibleValues = paramDef.getValues();
    Any convertedValue = parameter;
    // if it is of type any, ignore converting step
    if (paramDef.getType() != null) {
      final ValueType type = paramDef.getType();
      try {
        // try to convert it and catch conversion errors.
        switch (type) {
          case BOOLEAN:
            convertedValue = DataFactory.DEFAULT.createBooleanValue(parameter.asValue().asBoolean());
            break;
          case DATE:
            convertedValue = DataFactory.DEFAULT.createDateValue(parameter.asValue().asDate());
            break;
          case DATETIME:
            convertedValue = DataFactory.DEFAULT.createDateTimeValue(parameter.asValue().asDateTime());
            break;
          case DOUBLE:
            convertedValue = DataFactory.DEFAULT.createDoubleValue(parameter.asValue().asDouble());
            break;
          case LONG:
            convertedValue = DataFactory.DEFAULT.createLongValue(parameter.asValue().asLong());
            break;
          case STRING:
            convertedValue = DataFactory.DEFAULT.createStringValue(parameter.asValue().asString());
            break;
          default:
            convertedValue = parameter.asValue();
        }
      } catch (final InvalidValueTypeException e) {
        throw new InvalidConfigException(
          "Parameter '" + paramDef.getName() + "' of worker '" + workerName + "' in workflow '" + workflowName
            + "' is not of type '" + type + "' or could not be properly resolved.", e);
      }
    }
    // check values, we have to convert to the type of the converted value...
    // interchangeable...
    if (possibleValues != null) {
      for (final Value value : possibleValues) {
        if (convertedValue.equals(DataFactory.DEFAULT.parseFromString(value.toString(), convertedValue
          .getValueType().name()))) {
          return;
        }
      }
      throw new InvalidConfigException("Parameter '" + paramDef.getName() + "' of worker '" + workerName
        + "' in workflow '" + workflowName + "' is does not match allowed values '" + possibleValues
        + "'. Value was '" + parameter + "'.");
    }
  }

  /**
   * Checks if compiled store names for a workflow with additional given parameters would be correct.
   * 
   * @param workflow
   *          The workflow to check
   * @param parameters
   *          the additional parameters
   * @param bucketCache
   *          the bucket cache
   * @throws InvalidConfigException
   *           invalid store names would be created for this workflow.
   * @throws PersistenceException
   *           exception while trying to access the definitions
   */
  private void checkValidStoreNameForWorkflow(final WorkflowDefinition workflow, final AnyMap parameters,
    final Map<String, BucketDefinition> bucketCache) throws InvalidConfigException, PersistenceException {
    try {

      // add workflow parameters
      final AnyMap workflowParameters = DataFactory.DEFAULT.createAnyMap();
      if (parameters != null) {
        workflowParameters.putAll(parameters);
      }
      if (workflow.getParameters() != null) {
        workflowParameters.putAll(workflow.getParameters());
      }

      // check all actions
      for (final WorkflowAction action : getAllActions(workflow)) {
        // check action.
        // since action parameters MUST NOT influence buckets, they won't be checked here!
        final WorkerDefinition workerDefinition = _definitionAccessor.getWorker(action.getWorker());

        // get the data object types' parameters (resp. variables)
        checkStoreNamesForInAndOutputSlots(bucketCache, action, workflowParameters, workerDefinition);
      }

    } catch (final InvalidConfigException e) {
      throw new InvalidConfigException("Parameters for workflow '" + workflow.getName()
        + "' would result in invalid store name(s).", e);
    }
  }

  /**
   * Checks if all slots for a given action and worker with given parameters would lead to a correct store name
   * (ignoring the unresolved parameters).
   * 
   * @param bucketCache
   *          the bucket cache
   * @param action
   *          the action to check
   * @param workflowParameters
   *          the workflow's parameters
   * @param workerDefinition
   *          the definition of the worker
   * @throws PersistenceException
   *           an error occurred while accessing the definitions
   * @throws InvalidConfigException
   *           the check for correct store names failed.
   */
  private void checkStoreNamesForInAndOutputSlots(final Map<String, BucketDefinition> bucketCache,
    final WorkflowAction action, final AnyMap workflowParameters, final WorkerDefinition workerDefinition)
    throws PersistenceException, InvalidConfigException {
    if (action.getInput() != null && workerDefinition.getInput() != null) {
      for (final Input inputSlot : workerDefinition.getInput()) {
        final String bucketName = action.getInput().get(inputSlot.getName());
        if (bucketName != null) {
          checkStoreNameForActionSlot(bucketName, bucketCache, inputSlot, workflowParameters);
        }
      }
    }
    if (action.getOutput() != null && workerDefinition != null && workerDefinition.getOutput() != null) {
      for (final Output outputSlot : workerDefinition.getOutput()) {
        final String bucketName = action.getOutput().get(outputSlot.getName());
        if (bucketName != null) {
          checkStoreNameForActionSlot(bucketName, bucketCache, outputSlot, workflowParameters);
        }
      }
    }
  }

  /**
   * Returns all actions (startAction and normal actions) of a workflow as a collection.
   * 
   * @param workflow
   *          the workflow definition.
   * @return all actions (startAction and normal actions) of a workflow as a collection.
   */
  private Collection<WorkflowAction> getAllActions(final WorkflowDefinition workflow) {
    final Collection<WorkflowAction> actions = new ArrayList<WorkflowAction>();
    if (workflow.getStartAction() != null) {
      actions.add(workflow.getStartAction());
    }
    if (workflow.getActions() != null) {
      actions.addAll(workflow.getActions());
    }
    return actions;
  }

  /**
   * Checks if a slot with given parameters would lead to a correct store name (ignoring the unresolved parameters).
   * 
   * @param bucketName
   *          the name of the bucket
   * @param bucketCache
   *          the bucket cache
   * @param slot
   *          the slot to check
   * @param workflowParameters
   *          the workflow's parameters
   * @throws PersistenceException
   *           an error occurred while accessing the definitions
   * @throws InvalidConfigException
   *           the check for correct store names failed.
   */
  private void checkStoreNameForActionSlot(final String bucketName,
    final Map<String, BucketDefinition> bucketCache, final InputOutput<?> slot, final AnyMap workflowParameters)
    throws PersistenceException, InvalidConfigException {
    // try to get the BucketDefinition:
    final BucketDefinition bucketDef;
    if (bucketCache.containsKey(bucketName)) {
      bucketDef = bucketCache.get(bucketName);
    } else {
      bucketDef = _definitionAccessor.getBucket(bucketName);
      bucketCache.put(bucketName, bucketDef);
    }
    final DataObjectTypeDefinition dot = _definitionAccessor.getDataObjectType(slot.getType());
    checkValidStoreNameForBucket(bucketName, bucketDef, dot, workflowParameters);
  }

  /**
   * Returns all parameters of the workflow and its buckets and workers that are not resolved within the workflow
   * definition. The map key contains the parameter, the value a description about the origin of the parameter.
   * 
   * @param workflow
   *          The workflow definition.
   * @param bucketCache
   *          the bucket cache
   * @return Map for all unresolved parameters of the workflow. (key: parameter, value: description). Empty if no such
   *         parameters exist.
   * @throws PersistenceException
   *           error reading definition(s)
   */
  private Map<String, String> getUnresolvedParameters(final WorkflowDefinition workflow,
    final Map<String, BucketDefinition> bucketCache) throws PersistenceException {
    Map<String, String> unresolvedParams = new LinkedHashMap<String, String>();
    if (workflow != null) {

      // caching buckets.
      final Collection<WorkflowAction> actions = getAllActions(workflow);
      // check all actions
      for (final WorkflowAction action : actions) {
        // what is left is unresolved for this action
        unresolvedParams.putAll(getUnresolvedActionParameters(action, bucketCache));
      }

      // get workflow defined unresolved params
      // note, we need two iterations here for the case
      // { name : variable, value: value },
      // { name: variable2, value: ${variable} }
      // if we wouldn't use two iterations, the second part would add 'variable' to the list of unresolved variables,
      // although 'variable' is declared in first part.
      if (workflow.getParameters() != null) {
        // add only parameters with values referencing variables
        fillUnresolvedVariables(workflow.getParameters(), null, "workflow", unresolvedParams);

        unresolvedParams = removeDefinedVariables(workflow.getParameters(), unresolvedParams);
      }
    }
    removeSystemParameters(unresolvedParams);
    return unresolvedParams;
  }

  /**
   * @param topLevelParameters
   *          the parameters map
   * @param unresolvedParams
   *          the map of unresolved parameters
   */
  protected Map<String, String> removeDefinedVariables(final AnyMap topLevelParameters,
    final Map<String, String> unresolvedParams) {
    final Map<String, String> returnMap = new HashMap<String, String>(unresolvedParams);
    // removal of defined parameters in a second iteration. Only top-level values can be used to resolve
    // variables...
    for (final Entry<String, Any> entry : topLevelParameters.entrySet()) {
      if (entry.getValue() != null && entry.getValue().isValue()) {
        returnMap.remove(entry.getKey());
      }
    }
    return returnMap;
  }

  /**
   * @param parameter
   *          the parameter (SEQ, MAP or value) that potentially references a variable
   * @param variableReferencingParameters
   *          the parameters which are referencing variables
   */
  protected void fillUnresolvedVariables(final Any parameter, final String key, final String container,
    final Map<String, String> variableReferencingParameters) {
    if (parameter.isSeq()) {
      for (final Any singleParam : parameter.asSeq()) {
        fillUnresolvedVariables(singleParam, key, container, variableReferencingParameters);
      }
    } else if (parameter.isMap()) {
      for (final Entry<String, Any> entry : parameter.asMap().entrySet()) {
        final String newKey;
        if (key != null) {
          newKey = key + "/" + entry.getKey();
        } else {
          newKey = entry.getKey();
        }
        fillUnresolvedVariables(entry.getValue(), newKey, container, variableReferencingParameters);
      }
    } else if (parameter.isString()) {
      final ValueExpression ve = new ValueExpression(parameter.asValue().asString());
      for (final String variable : ve.getVariables()) {
        variableReferencingParameters.put(variable, "variable in '" + container + "' of parameter '" + key + "'");
      }
    }
  }

  /**
   * Removes the system parameters from a parameter collection.
   * 
   * @param params
   *          the parameter collection.
   */
  private void removeSystemParameters(final Map<String, String> params) {
    // remove system variables
    final Iterator<String> iter = params.keySet().iterator();
    while (iter.hasNext()) {
      final String varName = iter.next();
      if (varName.startsWith(VARIABLE_SYSTEM_PREFIX)) {
        iter.remove();
      }
    }
  }

  /**
   * Returns all parameters of the workflow action and its buckets and workers that are not resolved within the workflow
   * action itself. The map key contains the parameter, the value a description about the origin of the parameter.
   * 
   * @param action
   *          the workflow action to check
   * @param bucketCache
   *          the bucket cache
   * @return All unresolved parameters of the action. Map (key: parameter, value: description)
   * @throws PersistenceException
   *           error reading definition(s)
   */
  private Map<String, String> getUnresolvedActionParameters(final WorkflowAction action,
    final Map<String, BucketDefinition> bucketCache) throws PersistenceException {
    final Map<String, String> unresolvedWorkerParams = new HashMap<String, String>();
    final WorkerDefinition workerDefinition = _definitionAccessor.getWorker(action.getWorker());

    // get worker's parameters
    if (workerDefinition != null && workerDefinition.getMandatoryParameterNames() != null) {
      for (final String parameter : workerDefinition.getMandatoryParameterNames()) {
        unresolvedWorkerParams.put(parameter, "parameter of worker '" + workerDefinition.getName() + "'");
      }
    }

    // the actions defined parameters will be removed but newly referenced parameters inserted,
    // e.g. { "index":"${index-name}": index will be removed, but index-name will be added.
    // but the buckets parameters must not be evaluated with the local action parameters, so ignore them here.
    if (action.getParameters() != null) {
      for (final Entry<String, Any> entry : action.getParameters().entrySet()) {
        unresolvedWorkerParams.remove(entry.getKey());
        fillUnresolvedVariables(entry.getValue(), entry.getKey(), "worker action '" + action.getWorker() + "'",
          unresolvedWorkerParams);
      }
    }

    // get the data object types' parameters (resp. variables)
    unresolvedWorkerParams.putAll(getUnresolvedParametersForActionSlots(action, bucketCache, workerDefinition));
    return unresolvedWorkerParams;
  }

  /**
   * @param action
   *          the workflow action to check
   * @param bucketCache
   *          the bucket cache
   * @param workerDefinition
   *          the worker definition of the action's worker
   * @return All unresolved parameters of the action slot. Map (key: parameter, value: description)
   * @throws PersistenceException
   */
  private Map<String, String> getUnresolvedParametersForActionSlots(final WorkflowAction action,
    final Map<String, BucketDefinition> bucketCache, final WorkerDefinition workerDefinition)
    throws PersistenceException {
    final Map<String, String> unresolvedActionSlotParams = new HashMap<String, String>();
    if (workerDefinition != null && workerDefinition.getInput() != null) {
      for (final Input inputSlot : workerDefinition.getInput()) {
        final String bucketName = action.getInput().get(inputSlot.getName());
        unresolvedActionSlotParams.putAll(getUnresolvedParametersForBucket(bucketName, bucketCache, inputSlot));
      }
    }
    if (workerDefinition != null && workerDefinition.getOutput() != null && action.getOutput() != null) {
      for (final Output outputSlot : workerDefinition.getOutput()) {
        final String bucketName = action.getOutput().get(outputSlot.getName());
        unresolvedActionSlotParams.putAll(getUnresolvedParametersForBucket(bucketName, bucketCache, outputSlot));
      }
    }
    return unresolvedActionSlotParams;
  }

  /**
   * Returns variables used for the buckets in the given input or output slot of an action. The map key contains the
   * parameter, the value a description about the origin of the parameter.
   * 
   * @param bucketName
   *          the bucket to check
   * @param bucketCache
   *          the bucket cache
   * @param slot
   *          the slot to check
   * @return variables used for the buckets in the given input or output slot of an action. Map (key: parameter, value:
   *         description)
   * @throws PersistenceException
   *           error reading definitions.
   */
  private Map<String, String> getUnresolvedParametersForBucket(final String bucketName,
    final Map<String, BucketDefinition> bucketCache, final InputOutput<?> slot) throws PersistenceException {
    Map<String, String> unresolvedBucketParams = new HashMap<String, String>();
    if (bucketName != null) {
      // try to get the BucketDefinition:
      final BucketDefinition bucketDef;
      if (bucketCache.containsKey(bucketName)) {
        bucketDef = bucketCache.get(bucketName);
      } else {
        bucketDef = _definitionAccessor.getBucket(bucketName);
        bucketCache.put(bucketName, bucketDef);
      }
      final boolean isPersistent = bucketDef != null;

      // note, we need two iterations here for the case that we have bucket parameters like that:
      // { name : variable, value: value },
      // { name: variable2, value: ${variable} }
      // if we wouldn't use two iterations, the second part would add 'variable' to the list of unresolved variables,
      // although 'variable' is declared in first part.

      // 1. add data object type variables and bucket variables
      final Collection<String> dataObjectTypeVariables = getDataObjectTypeVariables(slot.getType(), isPersistent);
      for (final String dataObjectTypeVar : dataObjectTypeVariables) {
        unresolvedBucketParams.put(dataObjectTypeVar, "variable for dataObjectType '" + slot.getType()
          + "' of bucket '" + bucketName + "' used in slot '" + slot.getName() + "'");
      }
      if (bucketDef != null && bucketDef.getParameters() != null) {
        fillUnresolvedVariables(bucketDef.getParameters(), null, "bucket '" + bucketName + "' used in slot '"
          + slot.getName() + "'", unresolvedBucketParams);
        // 2. removal of defined variables
        unresolvedBucketParams = removeDefinedVariables(bucketDef.getParameters(), unresolvedBucketParams);
      }
    }
    return unresolvedBucketParams;
  }

  /**
   * Returns a collection of variable names used in the data object type definition.
   * 
   * @param typeName
   *          The name of the Data Object Type.
   * @param isPersistent
   *          'true' if the PERSISTENT mode has to be used, 'false' if the TRANSIENT mode has to be used..
   * @return A collection of variable names used in the data object type definition.
   */
  private Collection<String> getDataObjectTypeVariables(final String typeName, final boolean isPersistent) {
    final Collection<String> variables = new HashSet<String>();
    final DataObjectTypeDefinition dto = _definitionAccessor.getDataObjectType(typeName);
    if (dto != null) {
      if (isPersistent && dto.hasPersistentMode()) {
        variables.addAll(dto.getObject(Mode.PERSISTENT).getVariables());
        variables.addAll(dto.getStore(Mode.PERSISTENT).getVariables());
      }
      if (!isPersistent && dto.hasTransientMode()) {
        variables.addAll(dto.getObject(Mode.TRANSIENT).getVariables());
        variables.addAll(dto.getStore(Mode.TRANSIENT).getVariables());
      }
    }
    return variables;
  }

  /**
   * @return job definitions referencing given workflow.
   * @throws PersistenceException
   *           error reading from storage
   */
  private Collection<JobDefinition> getJobsForWorkflow(final String workflowName) throws PersistenceException {
    final List<JobDefinition> jobs = new ArrayList<JobDefinition>();
    final Collection<String> storageJobs = _definitionAccessor.getJobs();
    for (final String jobName : storageJobs) {
      final JobDefinition jobDef = _definitionAccessor.getJob(jobName);
      if (jobDef.getWorkflow().equals(workflowName)) {
        jobs.add(jobDef);
      }
    }
    return jobs;
  }

  /**
   * Checks if the action's buckets are correctly configured. If not all slots are configured or the bucket types do not
   * match, an exception is thrown. Also checks if the workflow defines only the same groups of output slots for one
   * worker, i.e. all output slots share the same group (or are not part of any group) for one worker.
   * 
   * @param workflow
   *          The workflow to check.
   * @param bucketCache
   *          the bucket cache
   * @throws InvalidConfigException
   *           This exception is thrown if the action buckets are incorrectly configured.
   * @throws PersistenceException
   *           error accessing bucket
   */
  private void checkWorkflowActionBuckets(final WorkflowDefinition workflow,
    final Map<String, BucketDefinition> bucketCache) throws InvalidConfigException, PersistenceException {
    final Collection<WorkflowAction> actions = new HashSet<WorkflowAction>();
    if (workflow.getActions() != null) {
      actions.addAll(workflow.getActions());
    }

    // check start action
    if (workflow.getStartAction() != null) {
      final WorkflowAction startAction = workflow.getStartAction();
      actions.add(startAction);
      checkWorkflowStartAction(workflow.getName(), startAction, bucketCache);
    }

    // check all actions
    for (final WorkflowAction workflowAction : actions) {
      final WorkerDefinition workerDefinition = _definitionAccessor.getWorker(workflowAction.getWorker());
      if (workerDefinition == null) {
        throw new InvalidConfigException("An action of workflow '" + workflow.getName() + "' references a worker '"
          + workflowAction.getWorker() + "' that is not defined.");
      }

      // check if all input slots of the worker definition are set in the action
      final Map<String, String> inputFromAction = workflowAction.getInput();
      checkWorkflowActionInputs(workflow.getName(), inputFromAction, workerDefinition, bucketCache);

      // check if all output slots of the worker definition are set in the action
      // and share the same group (or none)
      if (workerDefinition.getOutput() != null) {
        String actionSlotGroup = null;
        final Map<String, String> outputFromAction = workflowAction.getOutput();
        if (outputFromAction != null) {
          for (final Output outputFromWorkerDef : workerDefinition.getOutput()) {
            if (outputFromAction.keySet().contains(outputFromWorkerDef.getName())) {
              actionSlotGroup = checkSlotGroup(workflow, workflowAction, actionSlotGroup, outputFromWorkerDef);
            }
          }
        }
        if (actionSlotGroup == null && workerDefinition.hasOnlyMandatoryGroups()) {
          throw new InvalidConfigException("Each output group of worker '" + workerDefinition.getName()
            + "' contains a mandatory slot, so one group must be used in workflow '" + workflow.getName() + "'");
        }
        checkWorkflowActionOutputs(workflow.getName(), outputFromAction, workerDefinition, actionSlotGroup,
          bucketCache);
      }
    }
  }

  /**
   * Checks if all slots share the same group (or are not part of any group).
   * 
   * @param workflow
   *          the workflow name
   * @param workflowAction
   *          the workflowAction
   * @param actionSlotGroup
   *          the current actionSlotGroup, if already determined, null if not.
   * @param slotFromWorkerDef
   *          the slot to check
   * @throws InvalidConfigException
   *           if the slot's group does not match to the already defined actionSlotGroup.
   */
  private String checkSlotGroup(final WorkflowDefinition workflow, final WorkflowAction workflowAction,
    String actionSlotGroup, final Output slotFromWorkerDef) throws InvalidConfigException {
    final String slotGroup = slotFromWorkerDef.getGroup();
    if (actionSlotGroup == null) {
      actionSlotGroup = slotGroup;
    } else if (slotGroup != null && !actionSlotGroup.equals(slotGroup)) {
      throw new InvalidConfigException("An action of workflow '" + workflow.getName() + "' mixes two slot groups '"
        + actionSlotGroup + "' and '" + slotGroup + "' for worker '" + workflowAction.getWorker() + "'.");
    }
    return actionSlotGroup;
  }

  /**
   * @param workflow
   *          the workflow whose start action is checked
   * @param startAction
   *          the start action to check
   * @param bucketCache
   *          the bucket cache
   * @throws InvalidConfigException
   *           start action has invalid input bucket
   * @throws PersistenceException
   *           error accessing bucket
   */
  private void checkWorkflowStartAction(final String workflow, final WorkflowAction startAction,
    final Map<String, BucketDefinition> bucketCache) throws InvalidConfigException, PersistenceException {
    // if start action has an input bucket, it must be persistent
    final Map<String, String> startActionInputMap = startAction.getInput();
    if (startActionInputMap != null) {
      for (final String inputBucket : startActionInputMap.values()) {
        BucketDefinition bucketDefinition = bucketCache.get(inputBucket);
        if (bucketDefinition == null) {
          bucketDefinition = _definitionAccessor.getBucket(inputBucket);
          bucketCache.put(inputBucket, bucketDefinition);
        }
        if (bucketDefinition == null) {
          _log.info("Start action worker of workflow '" + workflow + "' references an input bucket '" + inputBucket
            + "' that is not persistent. The workflow can only be started in runOnce jobs!)");
        }
      }
    }
  }

  /**
   * Checks if the input buckets are correctly set and the types match.
   */
  private void checkWorkflowActionInputs(final String workflow, final Map<String, String> inputFromAction,
    final WorkerDefinition workerDefinition, final Map<String, BucketDefinition> bucketCache)
    throws InvalidConfigException {

    if (workerDefinition.getInput() != null) {
      for (final Input inputFromWorkerDef : workerDefinition.getInput()) {
        // check if input of worker definition is set in action
        if (inputFromAction == null || !inputFromAction.keySet().contains(inputFromWorkerDef.getName())) {
          throw new InvalidConfigException("Input slot '" + inputFromWorkerDef.getName()
            + "' of referenced worker '" + workerDefinition.getName() + "' of workflow '" + workflow
            + "' is not defined in workflow action.");
        } else {
          // ...and if the types do match
          if (!checkBucketType(inputFromAction.get(inputFromWorkerDef.getName()), inputFromWorkerDef.getType(),
            bucketCache)) {
            throw new InvalidConfigException("Input bucket '" + inputFromAction.get(inputFromWorkerDef.getName())
              + "' for referenced worker '" + workerDefinition.getName() + "' of workflow '" + workflow
              + "' has the wrong type.");
          }
          // ...and if the bucket mode is correct
          if (!checkBucketMode(inputFromAction.get(inputFromWorkerDef.getName()), inputFromWorkerDef.getType(),
            bucketCache)) {
            throw new InvalidConfigException("Input bucket '" + inputFromAction.get(inputFromWorkerDef.getName())
              + "' for referenced worker '" + workerDefinition.getName() + "' of workflow '" + workflow
              + "' is not defined, but has no transient mode.");
          }
        }
      }
    }
  }

  /**
   * Checks if the output buckets are correctly set and the types match.
   */
  private void checkWorkflowActionOutputs(final String workflow, final Map<String, String> outputFromAction,
    final WorkerDefinition workerDefinition, final String activeSlotGroup,
    final Map<String, BucketDefinition> bucketCache) throws InvalidConfigException {

    for (final Output outputFromWorkerDef : workerDefinition.getOutput()) {
      // check if output exists or if it is defined as optional
      if (outputFromAction == null || !outputFromAction.keySet().contains(outputFromWorkerDef.getName())) {
        // optional buckets do not have to be defined
        if (!outputFromWorkerDef.getModes().contains(OutputMode.OPTIONAL)) {
          if (outputFromWorkerDef.getGroup() == null || outputFromWorkerDef.getGroup().equals(activeSlotGroup)) {
            throw new InvalidConfigException("Output slot '" + outputFromWorkerDef.getName()
              + "' of referenced worker '" + workerDefinition.getName() + "' of workflow '" + workflow
              + "' is not defined.");
          } else {
            if (_log.isDebugEnabled()) {
              _log.debug("Mandatory output bucket '" + outputFromWorkerDef.getName() + "' of worker '"
                + workerDefinition.getName() + "' is not defined in workflow '" + workflow
                + "' but a different slot group is active.");
            }
          }
        } else {
          if (_log.isDebugEnabled()) {
            _log.debug("Optional output bucket '" + outputFromWorkerDef.getName() + "' of worker '"
              + workerDefinition.getName() + "' is not defined in workflow '" + workflow + "'.");
          }
        }
      } else {
        // and if the types do match
        if (!checkBucketType(outputFromAction.get(outputFromWorkerDef.getName()), outputFromWorkerDef.getType(),
          bucketCache)) {
          throw new InvalidConfigException("Output bucket '" + outputFromAction.get(outputFromWorkerDef.getName())
            + "' for referenced worker '" + workerDefinition.getName() + "' of workflow '" + workflow
            + "' has the wrong type.");
        }
        // and if the bucket mode is correct
        if (!checkBucketMode(outputFromAction.get(outputFromWorkerDef.getName()), outputFromWorkerDef.getType(),
          bucketCache)) {
          throw new InvalidConfigException("Output bucket '" + outputFromAction.get(outputFromWorkerDef.getName())
            + "' for referenced worker '" + workerDefinition.getName() + "' of workflow '" + workflow
            + "' is persistent, so it has to be defined.");
        }
      }
    }
  }

  /**
   * Checks bucket type.
   * 
   * @param bucketName
   *          The name of the bucket.
   * @param type
   *          The expected type
   * @param bucketCache
   *          the bucket cache
   * @return true if bucket type is correct.
   */
  private boolean checkBucketType(final String bucketName, final String type,
    final Map<String, BucketDefinition> bucketCache) {
    try {
      BucketDefinition bucketDefinition = bucketCache.get(bucketName);
      if (bucketDefinition == null) {
        bucketDefinition = _definitionAccessor.getBucket(bucketName);
        bucketCache.put(bucketName, bucketDefinition);
      }
      if (bucketDefinition == null) {
        if (_log.isTraceEnabled()) {
          _log.trace("Bucket '" + bucketName + "' does not exist, so it is assumed this bucket is transient.");
        }
        return true;
      } else {
        // persistent bucket - check type
        return bucketDefinition.getDataObjectType().equals(type);
      }
    } catch (final Exception e) {
      if (_log.isWarnEnabled()) {
        _log
          .warn("An error occurred while trying to check bucket '" + bucketName + "' for type '" + type + "'.", e);
      }
      return false;
    }
  }

  /**
   * Checks bucket mode.
   * 
   * @param bucketName
   *          The name of the bucket.
   * @param type
   *          The type to check against
   * @param bucketCache
   *          the bucket cache
   * @return true if bucket is defined/persistent, or bucket's data type provides a transient mode.
   */
  private boolean checkBucketMode(final String bucketName, final String type,
    final Map<String, BucketDefinition> bucketCache) {
    try {
      BucketDefinition bucketDefinition = bucketCache.get(bucketName);
      if (bucketDefinition == null) {
        bucketDefinition = _definitionAccessor.getBucket(bucketName);
        bucketCache.put(bucketName, bucketDefinition);
      }
      if (bucketDefinition == null) {
        // not found -> maybe transient bucket - check data type provides transient mode
        final DataObjectTypeDefinition dot = _definitionAccessor.getDataObjectType(type);
        return dot.hasTransientMode();
      }
      return true;
    } catch (final Exception e) {
      if (_log.isWarnEnabled()) {
        _log
          .warn("An error occurred while trying to check bucket '" + bucketName + "' for type '" + type + "'.", e);
      }
      return false;
    }
  }

  /**
   * Checks if the storeName (while ignoring any unresolved Parameter) would lead to a correct store name.
   * 
   * @param bucketName
   *          the name of the bucket
   * @param bucketDefinition
   *          the bucketDefinition to check. Can be null if the bucket does not exist.
   * @param type
   *          the DataObjectType to check
   * @param externalParameters
   *          external parameters to be included in evaluating the store name
   * @return 'true' if the store name is a correct store name, 'false' if not (while ignoring all unresolved variables).
   * @throws InvalidConfigException
   *           error while accessing parameters
   */
  private boolean checkValidStoreNameForBucket(final String bucketName, final BucketDefinition bucketDefinition,
    final DataObjectTypeDefinition type, final AnyMap externalParameters) throws InvalidConfigException {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    if (externalParameters != null) {
      parameters.putAll(externalParameters);
    }
    if (bucketDefinition != null && bucketDefinition.getParameters() != null) {
      parameters.putAll(bucketDefinition.getParameters());
    }
    if (parameters.size() > 0) {
      BucketDefinition bDef;
      if (bucketDefinition == null) {
        // create temporary bucketDefinition
        final AnyMap bucketAny = AccessAny.FACTORY.createAnyMap();
        try {
          bucketAny.put("name", bucketName);
          bucketAny.put("type", type.getName());
        } catch (final Exception e) {
          _log.error("Error while trying to create temporary bucket definition.", e);
          return false;
        }
        try {
          bDef = new BucketDefinition(bucketAny);
        } catch (final InvalidDefinitionException cause) {
          throw new InvalidConfigException("Invalid bucket definition", cause);
        }
      } else {
        bDef = bucketDefinition;
      }
      final Bucket bucket = new Bucket(bDef, type, bucketDefinition != null, parameters);
      final String resolvedStoreNameWithoutUnresolvedVariables =
        bucket.getStoreName().replaceAll(ValueExpression.VARIABLE_REGEX, "");
      if (resolvedStoreNameWithoutUnresolvedVariables.length() > 0) {
        if (!_objectStoreService.isValidStoreName(resolvedStoreNameWithoutUnresolvedVariables)) {
          throw new InvalidConfigException("Current parameters would result in invalid store name '"
            + resolvedStoreNameWithoutUnresolvedVariables + "' for DataObjectType '" + type.getName()
            + "' of bucket '" + bDef.getName() + "'.");
        }
      }
    }
    return true;
  }

  /**
   * Check if the workflow can be safely removed without compromising data integrity. I.e. if there is an existing job
   * definition that still references this workflow, it must not be removed.
   * 
   * @param name
   *          The name of the workflow to check
   * @throws PersistenceException
   *           error reading job definition(s)
   * @throws InvalidConfigException
   *           error deleting definition due to dependency constraints
   */
  public void checkRemoveWorkflow(final String name) throws InvalidConfigException, PersistenceException {
    if (_definitionAccessor.getConfiguredWorkflows().containsKey(name)) {
      throw new IllegalArgumentException("Workflow '" + name
        + "' cannot be removed, because it is predefined by configuration");
    }
    final Collection<String> jobNames = _definitionAccessor.getJobs();
    for (final String jobName : jobNames) {
      final JobDefinition job = _definitionAccessor.getJob(jobName);
      if (job != null) {
        // check if this job references this workflow...
        if (name.equals(job.getWorkflow())) {
          throw new InvalidConfigException("Workflow '" + name
            + "' cannot be deleted. It is still referenced by job '" + job.getName() + "'.");
        }
      }
    }
  }

  /**
   * Check if the bucket can be safely removed without compromising data integrity. I.e. if there is an existing
   * workflow definition that still references this bucket, it must not be removed.
   * 
   * @param name
   *          The name of the bucket to check
   * @throws PersistenceException
   *           error reading workflow definition(s)
   * @throws InvalidConfigException
   *           error deleting definition due to dependency constraints
   */
  public void checkRemoveBucket(final String name) throws InvalidConfigException, PersistenceException {
    if (_definitionAccessor.getConfiguredBuckets().containsKey(name)) {
      throw new IllegalArgumentException("Bucket '" + name
        + "' cannot be removed, because it is predefined by configuration");
    }
    final Collection<String> workflowNames = _definitionAccessor.getWorkflows();
    for (final String workflowName : workflowNames) {
      final WorkflowDefinition workflow = _definitionAccessor.getWorkflow(workflowName);
      if (workflow != null) {
        // check if this bucket is referenced by this workflow...
        if (workflow.getReferencedBuckets().contains(name)) {
          throw new InvalidConfigException("Bucket '" + name
            + "' cannot be deleted. It is still referenced by workflow '" + workflow.getName() + "'.");
        }
      }
    }
  }

  /**
   * Checks if the job can be removed.
   * 
   * @param jobName
   *          The name of the job to be removed.
   * @throws JobManagerException
   *           Some error occurred while checking if job is currently running.
   */
  public void checkJobRemove(final String jobName) throws JobManagerException {
    if (_definitionAccessor.getConfiguredJobs().containsKey(jobName)) {
      throw new IllegalArgumentException("Job '" + jobName
        + "' cannot be removed, because it is predefined by configuration");
    }
  }
}
