/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.JobRun;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.JobTaskProcessor;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Mode;
import org.eclipse.smila.jobmanager.definitions.WorkflowAction;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorageException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerationUtil;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerator;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorBase;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorProvider;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.taskworker.output.OutputMode;
import org.eclipse.smila.utils.collections.MultiValueMap;
import org.osgi.service.component.ComponentContext;

/**
 * Implements {@link JobTaskProcessor}.
 */
public class JobTaskProcessorImpl implements JobTaskProcessor {

  /** if this time is exceeded while generating tasks, a log message is written. */
  private static final long MIN_TASK_GENERATION_TIME_TO_LOG = 1000; // ms

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** service reference to definition persistence. */
  private DefinitionPersistence _defPersistence;

  /** service reference to definition storage. */
  private RunStorage _runStorage;

  /** service reference to taskmanager. */
  private TaskManager _taskManager;

  /** service reference to cluster config service to read job property. */
  private ClusterConfigService _clusterConfigService;

  /** service reference to object store. */
  private ObjectStoreService _objectStore;

  /** service reference to task generator provider used to select a task generator. */
  private TaskGeneratorProvider _taskGeneratorProvider;

  /** service reference to job run engine. */
  private JobRunEngine _runEngine;

  /**
   * OSGi Declarative Services service activation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void activate(final ComponentContext context) {
    final Collection<String> workers = _defPersistence.getWorkers();
    for (final String worker : workers) {
      try {
        _taskManager.addTaskQueue(worker);
      } catch (final TaskmanagerException ex) {
        _log.warn("Error creating task queue for worker '" + worker + "'", ex);
      }
    }
  }

  @Override
  public List<Task> finishTask(Task currentTask) throws JobManagerException {
    final ResultDescription resultDescription = currentTask.getResultDescription();
    final String workflowRunId = currentTask.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);
    final String jobRunId = currentTask.getProperties().get(Task.PROPERTY_JOB_RUN_ID);
    final String jobName = currentTask.getProperties().get(Task.PROPERTY_JOB_NAME);
    final String workerName = getOriginalWorkerName(currentTask);

    try {
      if (!_runStorage.hasTask(jobName, workflowRunId, currentTask.getTaskId())) {
        throw new IllegalJobStateException("Task '" + currentTask.getTaskId() + "' for job '" + jobName
          + "' and run '" + jobRunId + "' is unknown, maybe already finished or workflow run was canceled.");
      }

      try {
        final List<Task> followUpTasks = new ArrayList<Task>();
        switch (resultDescription.getStatus()) {
          case SUCCESSFUL:
            if (_log.isTraceEnabled()) {
              _log.trace("Successfully handles task '" + currentTask.getTaskId() + "' of worker '" + workerName
                + "' in job run '" + jobRunId + "' of job '" + jobName + "'");
            }
            followUpTasks.addAll(handleSuccessfulTask(jobName, jobRunId, workflowRunId, workerName, currentTask,
              resultDescription.getCounters()));
            break;
          case OBSOLETE:
            if (_log.isTraceEnabled()) {
              _log.trace("Obsolete task '" + currentTask.getTaskId() + "' of worker '" + workerName
                + "' in job run '" + jobRunId + "' of job '" + jobName + "'");
            }
            handleObsoleteTask(jobName, jobRunId, workflowRunId, workerName, currentTask);
            break;
          case RECOVERABLE_ERROR:
            // recoverable error - try to repeat the task processing
            _log.warn("A recoverable error '" + resultDescription.getErrorCode() + "'('"
              + resultDescription.getErrorMessage() + "') occurred in processing of task '"
              + currentTask.getTaskId() + "' for worker '" + workerName + "'");
            followUpTasks.addAll(handleRecoverableTaskError(jobName, jobRunId, workflowRunId, workerName,
              currentTask, resultDescription));
            break;
          default:
          case FATAL_ERROR:
            // fatal error - don't try to repeat the task processing
            _log.error("A fatal error '" + resultDescription.getErrorCode() + "'('"
              + resultDescription.getErrorMessage() + "') occurred in processing of task "
              + currentTask.getTaskId() + " of worker " + workerName + ". Workflow run '" + workflowRunId
              + "' will be marked as failed, its tasks will be canceled.");
            handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
        }
        return followUpTasks;
      } catch (final JobManagerException jme) {
        if (resultDescription.getStatus() != TaskCompletionStatus.FATAL_ERROR && !jme.isRecoverable()) {
          try {
            handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
          } catch (final JobManagerException e) {
            _log.error("Exception while handling fatal error during nonrecoverable finishing exception.", e);
          }
        }
        throw jme;
      }

    } finally {
      try {
        if (checkAndHandleWorkflowRunCompleted(jobName, jobRunId, workflowRunId)) {
          _runEngine.checkAndHandleJobRunCompleted(jobName, jobRunId);
        }
      } catch (final JobManagerException e) {
        final String message = "Exception while checking workflow/job run completion.";
        _log.error(message, e);
        throw new JobManagerException(message, e, true); // recoverable to repeat the operation
      }
    }
  }

  @Override
  public Task getInitialTask(String workerName, String jobName) throws JobManagerException {
    String jobRunId = null;
    String workflowRunId = null;
    Task task = null;
    jobRunId = _runStorage.getJobRunId(jobName);
    if (jobRunId == null || !checkJobStateForTaskCreation(jobName, jobRunId, true)) {
      throw new IllegalJobStateException("Job with name '" + jobName + "' is not running or is already finishing.");
    }
    final JobRun jobRun = _runEngine.ensureJobRun(jobName, jobRunId);
    try {
      workflowRunId = _runStorage.startWorkflowRun(jobName, jobRunId);
      task = getInitialTask(jobRun, workerName, workflowRunId);
      // initial tasks are not recoverable, because no other worker would ever retry them
      task.getProperties().put(Task.PROPERTY_RECOVERABLE, Boolean.toString(Boolean.FALSE));
      _runStorage.startTask(jobName, jobRunId, workflowRunId, workerName, task.getTaskId());
      _taskManager.addInProgressTask(task);
      return task;
    } catch (final Exception e) {
      cleanupFailedWorkflowRun(jobName, jobRunId, workflowRunId, workerName, task);
      throw new JobManagerException("Getting initial task failed for worker '" + workerName + "' for job '"
        + jobName + "' due to error.", e);
    }
  }

  /** @return value of task property {@link Task#PROPERTY_ORIGINAL_WORKER}. */
  private String getOriginalWorkerName(final Task currentTask) {
    return currentTask.getProperties().get(Task.PROPERTY_ORIGINAL_WORKER);
  }

  /**
   * @return the follow up tasks of the task that is successfully finished.
   */
  protected List<Task> handleSuccessfulTask(final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName, final Task currentTask,
    final Map<String, Number> workerCounter) throws JobManagerException {

    final List<Task> followUpTasks = new ArrayList<Task>();
    try {
      final JobRun jobRun = _runEngine.ensureJobRun(jobName, jobRunId);
      final WorkerDefinition worker = _defPersistence.getWorker(workerName);

      if (!checkJobStateForTaskCreation(jobName, jobRunId, false)) {
        throw new IllegalJobStateException("Could not finish Task. Job with name'" + jobName
          + "' is not running or is already completed.");
      }
      adaptBulksForMultipleOutputSlots(currentTask, workerName);
      // create follow up tasks
      followUpTasks.addAll(createFollowUpTasks(currentTask, jobRun, workflowRunId));
      for (final Task newTask : followUpTasks) {
        final String newTaskJobName = newTask.getProperties().get(Task.PROPERTY_JOB_NAME);
        final String newTaskJobRunId = newTask.getProperties().get(Task.PROPERTY_JOB_RUN_ID);
        final String newTaskWfRunId = newTask.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

        // TODO find a clean solution for handling the following statements in a more "atomic" way.
        // If RunStorage task entry creation is done first (as is), failures can happen if process/node is crashing
        // right between the two statements: Task entry in RunStorage will exist there until job run is canceled.
        // If instead Taskmanager task creation would be done first, this may cause bug #3566.
        _runStorage.startTask(newTaskJobName, newTaskJobRunId, newTaskWfRunId, newTask.getWorkerName(),
          newTask.getTaskId());
        _taskManager.addTask(newTask);
      }
      notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.SUCCESSFUL);
      // delete AFTER the new tasks have been added.
      // Otherwise some other concurrent invocation might think there are no tasks left and finishes the workflow run!
      _runStorage.finishTask(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId(), workerCounter,
        currentTask.getProperties());
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling successful task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
    return followUpTasks;
  }

  /**
   * finish obsolete task - don't create follow up tasks.
   */
  protected void handleObsoleteTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String workerName, final Task currentTask) throws JobManagerException {
    try {
      final WorkerDefinition worker = _defPersistence.getWorker(workerName);

      if (!checkJobStateForTaskCreation(jobName, jobRunId, false)) {
        throw new IllegalJobStateException("Could not finish obsolete task '" + currentTask.getTaskId()
          + "'. Job with name'" + jobName + "' is not running or is already completed.");
      }

      notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.OBSOLETE);
      _runStorage.obsoleteTask(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId(),
        currentTask.getProperties());

    } catch (final Exception e) {
      throw new JobManagerException("Error while handling obsolete task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
  }

  /**
   * @return follow up tasks. This could be the current task in case of a retry. If the number of retries exceeded no
   *         follow up tasks are generated. In case of mode AUTOCOMMIT task is handled as succeeded.
   */
  protected List<Task> handleRecoverableTaskError(final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName, final Task currentTask,
    final ResultDescription resultDescription) throws JobManagerException {

    final List<Task> followUpTasks = new ArrayList<Task>();
    try {
      final WorkerDefinition worker = _defPersistence.getWorker(workerName);
      if (worker == null) {
        throw new JobManagerException("Worker '" + workerName + "' not found while handling recoverable task '"
          + currentTask.getTaskId() + "' in job run '" + jobRunId + "' of job '" + jobName + "'.");
      }
      if (worker.getModes().contains(Mode.AUTOCOMMIT)) {
        // autoCommit=true -> commit task's output and handle task as "successful" processed
        handleRecoverableTaskWithAutocommit(jobName, jobRunId, workflowRunId, workerName, currentTask,
          resultDescription, followUpTasks);
      } else {
        // autoCommit=false -> repeat current task if it's recoverable
        if (currentTask.getProperties().containsKey(Task.PROPERTY_RECOVERABLE)
          && !Boolean.valueOf(currentTask.getProperties().get(Task.PROPERTY_RECOVERABLE))) {
          _log.warn("Could not retry failed task '" + currentTask.getTaskId() + "' cause it's not recoverable.");
          handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
        } else {
          final int numberOfRetries =
            _runStorage.getTaskRetries(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId()) + 1;
          // if we can retry it again, do so. If not, task is failed.
          if (numberOfRetries <= _clusterConfigService.getMaxRetries()) {
            _runStorage.retriedTask(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId(),
              !TaskManager.TASKERROR_TIME_TO_LIVE.equals(resultDescription.getErrorCode()),
              currentTask.getProperties());
            final Task retryTask = currentTask.createRetryTask(TaskGeneratorBase.createTaskId());
            _taskManager.addTask(retryTask);
            _runStorage.startTask(jobName, jobRunId, workflowRunId, retryTask.getWorkerName(),
              retryTask.getTaskId(), numberOfRetries);
            followUpTasks.add(retryTask);
          } else {
            _log.error("Could not retry failed recoverable task '" + currentTask.getTaskId()
              + "' , because maximum number of retries was reached.");
            handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, true);
          }
        }
      }
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling recoverable task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
    return followUpTasks;
  }

  /**
   * recoverable error on an AUTOCOMMIT worker's task -> task is handled as succeeded.
   */
  protected void handleRecoverableTaskWithAutocommit(final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName, final Task currentTask,
    final ResultDescription resultDescription, final List<Task> followUpTasks) throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("Task '" + currentTask.getTaskId() + "' of autocommit worker '" + workerName + "' in job '"
        + jobName + "' failed with an recoverable error: finishing it as successful.");
    }
    for (final Entry<String, List<BulkInfo>> entry : currentTask.getOutputBulks().entrySet()) {
      for (final BulkInfo bulkInfo : entry.getValue()) {
        final String storeName = bulkInfo.getStoreName();
        final String objectId = bulkInfo.getObjectName();
        // finish already created objects so the failing worker can no longer append data to it, if it's still alive.
        // and the data is closed cleanly.
        try {
          if (ObjectStoreRetryUtil.retryExistsObject(_objectStore, storeName, objectId)) {
            _objectStore.finishObject(storeName, objectId);
          }
        } catch (final ServiceUnavailableException ex) {
          _log.warn("Error finishing object '" + objectId + "' in store '" + storeName + "', retrying.", ex);
          throw new JobManagerException("Could not finish object '" + objectId + "' in store '" + storeName + "'.",
            ex);
        } catch (final Exception ex) {
          _log.warn("Error finishing object '" + objectId + "' in store '" + storeName
            + "', ignoring and continuing.", ex);
        }
      }
    }
    followUpTasks.addAll(handleSuccessfulTask(jobName, jobRunId, workflowRunId, workerName, currentTask,
      resultDescription.getCounters()));
  }

  /** 
   */
  protected void handleFatalError(final String jobName, final String jobRunId, final String workflowRunId,
    final String workerName, final Task currentTask, final boolean failedAfterRetry) throws JobManagerException {
    try {
      try {
        final WorkerDefinition worker = _defPersistence.getWorker(workerName);
        if (worker == null) {
          throw new JobManagerException("Worker '" + workerName + "' not found while handling fatal error task '"
            + currentTask.getTaskId() + "' in job run '" + jobRunId + "' of job '" + jobName + "'.");
        }
        notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.FATAL_ERROR);
      } finally {
        // delete current task first for statistical reasons
        _runStorage.failedTask(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId(),
          failedAfterRetry, currentTask.getProperties());

        // delete whole workflow run
        if (_runStorage.failedWorkflowRun(jobName, jobRunId, workflowRunId)) {
          _runEngine.deleteTransientBulks(jobName, jobRunId, workflowRunId);
          _runStorage.deleteWorkflowRun(jobName, jobRunId, workflowRunId);
          _runEngine.checkAndHandleJobRunCompleted(jobName, jobRunId);
        }
      }
      // TODO we have to do some smarter handling here:
      // - what to do with keep-alives from tasks of given workflow run?
      // - what to do with successful finished tasks of given workflow run?
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling failed task in workflow run '" + workflowRunId
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
  }

  /**
   * Checks if a workflow run is finished and handles finishing the workflow run. I.e. deleting transient buckets,
   * merging statistics and finishing job if all runs are finished...
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @param workflowRunId
   *          The id of the workflow run.
   * @return true if workflow run completed.
   * @throws JobManagerException
   *           error
   */
  private boolean checkAndHandleWorkflowRunCompleted(final String jobName, final String jobRunId,
    final String workflowRunId) throws JobManagerException {
    final boolean completed = isWorkflowRunCompleted(jobName, jobRunId, workflowRunId);
    if (completed) {
      _runEngine.deleteTransientBulks(jobName, jobRunId, workflowRunId);
      if (_runStorage.hasWorkflowRun(jobName, jobRunId, workflowRunId)) {
        // RunStorage aggregates workflow run statistics to job run
        if (_runStorage.finishWorkflowRun(jobName, jobRunId, workflowRunId)) {
          // delete workflow run from RunStorage
          _runStorage.deleteWorkflowRun(jobName, jobRunId, workflowRunId);
        }
      }
    }
    return completed;
  }

  /**
   * 
   * Checks if a task may be generated for the job run with the given jobRunId.
   * 
   * @param jobName
   *          The name of the job to check.
   * @param jobRunId
   *          The job id to check.
   * @param isInitialTask
   *          Is this task an initial task (true) or follow-up task(false).
   * @return true if tasks may be created, false if not.
   * @throws RunStorageException
   *           exception while accessing JobRun data.
   */
  private boolean checkJobStateForTaskCreation(final String jobName, final String jobRunId,
    final boolean isInitialTask) throws RunStorageException {
    final JobState jobState = _runStorage.getJobState(jobName, jobRunId);
    if (jobState == JobState.RUNNING) {
      return true;
    }
    if (!isInitialTask && jobState == JobState.FINISHING) {
      return true;
    }
    return false;
  }

  /**
   * adapt current task's output bulks for multiple output slots for which the worker has created (more than one) output
   * buckets itself.
   */
  private void adaptBulksForMultipleOutputSlots(final Task currentTask, final String workerName)
    throws JobManagerException {
    final WorkerDefinition workerDef = _defPersistence.getWorker(workerName);
    for (final String slotName : currentTask.getOutputBulks().keySet()) {
      if (workerDef != null && workerDef.getOutputModes() != null
        && workerDef.getOutputModes().get(slotName) != null
        && workerDef.getOutputModes().get(slotName).contains(OutputMode.MULTIPLE)) {
        final List<BulkInfo> taskBulks = currentTask.getOutputBulks().get(slotName);
        final BulkInfo template = taskBulks.get(0);
        final String objectNamePrefix = template.getObjectName();
        try {
          final Collection<StoreObject> objects =
            ObjectStoreRetryUtil.retryGetStoreObjectInfos(_objectStore, template.getStoreName(), objectNamePrefix);
          taskBulks.clear();
          for (final StoreObject o : objects) {
            final BulkInfo info = new BulkInfo(template.getBucketName(), template.getStoreName(), o.getId());
            taskBulks.add(info);
          }
        } catch (final ObjectStoreException e) {
          throw new JobManagerException("Error while adapting output bulks for multiple output slots", e);
        }
      }
    }
  }

  /**
   * create follow up tasks.
   * 
   * @param currentTask
   *          the currently processed task
   * @param jobRun
   *          job run
   * @param workflowRunId
   *          The workflow run id
   * @return a list of follow up tasks
   * @throws JobManagerException
   *           error
   */
  private List<Task> createFollowUpTasks(final Task currentTask, final JobRun jobRun, final String workflowRunId)
    throws JobManagerException {

    // check if all non-optional output buckets have been created, and get all output bulks
    final MultiValueMap<String, BulkInfo> bucketNameToCreatedBulksMap =
      checkOutputBulksForCreation(jobRun, currentTask);

    // check for new tasks for another job
    final Collection<JobRun> triggeredJobs = new ArrayList<JobRun>();
    for (final Entry<String, List<BulkInfo>> bucketToBulksEntry : bucketNameToCreatedBulksMap.entrySet()) {
      final String bucketName = bucketToBulksEntry.getKey();
      if (jobRun.getBucket(bucketName).isPersistent()) {
        triggeredJobs.addAll(getJobsTriggeredByBucket(jobRun, bucketName));
      } else {
        storeTransientBulks(jobRun, workflowRunId, bucketToBulksEntry.getValue());
      }
    }

    final AnyMap taskParamsToCopy = getTaskParametersToCopy(currentTask);
    final List<Task> followUpTasks =
      getFollowupTasks(jobRun, workflowRunId, bucketNameToCreatedBulksMap, taskParamsToCopy);
    for (final JobRun triggeredJob : triggeredJobs) {
      try {
        final Collection<Task> triggeredTasks = getTriggeredInitialTasks(triggeredJob, bucketNameToCreatedBulksMap);
        if (!triggeredTasks.isEmpty()) {
          final String triggeredJobWorkflowRunId =
            _runStorage.startWorkflowRun(triggeredJob.getJobName(), triggeredJob.getJobRunId());
          for (final Task task : triggeredTasks) {
            task.getProperties().put(Task.PROPERTY_WORKFLOW_RUN_ID, triggeredJobWorkflowRunId);
          }
          followUpTasks.addAll(triggeredTasks);
        }
      } catch (final Exception e) {
        _log.error(
          "Tried to create tasks for potentially triggered jobs, but got an error. Job '"
            + triggeredJob.getJobName() + "' will not be triggered.", e);
      }
    }
    return followUpTasks;
  }

  /** invoke finishTask method of task generator that belongs to the given worker. */
  private void notifyTaskGenerator(final WorkerDefinition worker, final Task task, final TaskCompletionStatus status)
    throws TaskGeneratorException {
    final TaskGenerator taskGenerator = _taskGeneratorProvider.getTaskGenerator(worker);
    taskGenerator.finishTask(task, status);
  }

  /**
   * Checks if the workflow run is finished, i.e. no tasks for this workflow run are open.
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @param workflowRunId
   *          The id of the workflow run.
   * @return 'true' if workflow run is finished
   * @throws RunStorageException
   *           exception while accessing RunStorage.
   */
  private boolean isWorkflowRunCompleted(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    return !_runStorage.hasTasks(jobName, jobRunId, workflowRunId);
  }

  /**
   * Checks if non-'maybeEmpty' output bulks have been created. If a non-'maybeEmpty' output bulk has not been created
   * an exception is thrown. If a 'maybeEmpty' output bulk has not been created, it will be removed from the map.
   * 
   * A bulk has not been created, if it is empty or does not even exist. An empty optional bulk will be deleted. TODO Stimmt das?
   * 
   * @param jobRun
   *          The JobRun instance for the current job run.
   * @param task
   *          The Task that should be checked.
   * @return output bulks without the not created optional buckets. (key: bucket name, value: BulkInfo)
   * @throws JobManagerException
   *           An error occurred while checking the objects or a non-optional bulk has not been created.
   */
  private MultiValueMap<String, BulkInfo> checkOutputBulksForCreation(final JobRun jobRun, final Task task)
    throws JobManagerException {
    final WorkerDefinition workerDef = _defPersistence.getWorker(getOriginalWorkerName(task));
    final MultiValueMap<String, BulkInfo> outputBulkMap = new MultiValueMap<String, BulkInfo>();
    final Iterator<Entry<String, List<BulkInfo>>> entryIter = task.getOutputBulks().entrySet().iterator();
    while (entryIter.hasNext()) {
      final Entry<String, List<BulkInfo>> currentEntry = entryIter.next();
      final String currentSlotName = currentEntry.getKey();
      final boolean maybeEmpty = workerDef.getOutput(currentSlotName).getModes().contains(OutputMode.MAYBEEMPTY);
      for (final BulkInfo bulkInfo : currentEntry.getValue()) {
        final String storeName = bulkInfo.getStoreName();
        final String objectName = bulkInfo.getObjectName();
        try {
          if (!ObjectStoreRetryUtil.retryExistsObject(_objectStore, storeName, objectName)) {
            if (!maybeEmpty) {
              throw new JobManagerException("Output bulk '" + objectName + "' in bucket '"
                + bulkInfo.getBucketName() + "' of worker '" + workerDef.getName() + "' (job '"
                + jobRun.getJobName() + "') has not been created but must not be empty.");
            }
          } else {
            // all's ok, add it to the output map.
            outputBulkMap.add(bulkInfo.getBucketName(), bulkInfo);
          }
        } catch (final Exception e) {
          throw new JobManagerException(e.getClass().getName() + " occurred while checking output buckets.", e);
        }
      }
    }
    return outputBulkMap;
  }

  /**
   * Returns a list of jobs (other than triggeringJob) where a startAction exists that is triggered by the persistent
   * bucket referenced in bucketStorageInfo.
   * 
   * @param triggeringJobRun
   *          the triggering job run
   * @param bucketName
   *          The BucketStorageInfo.
   * @return A list of job runs (other than triggeringJob) where a startAction exists that is triggered by the
   *         persistent bucket referenced in bucketStorageInfo.
   * @throws JobManagerException
   *           Error occurred while accessing JobDefinitions instances.
   */
  private Collection<JobRun> getJobsTriggeredByBucket(final JobRun triggeringJobRun, final String bucketName)
    throws JobManagerException {
    final Bucket triggeringBucket = triggeringJobRun.getBucket(bucketName);
    final Collection<String> jobNames = _runStorage.getTriggeredJobs(triggeringBucket.getBucketId());
    // ignore triggering job
    jobNames.remove(triggeringJobRun.getJobName());
    final Collection<JobRun> triggeredJobs = new ArrayList<JobRun>();
    for (final String jobName : jobNames) {
      try {
        final JobRunInfo runInfo = _runStorage.getJobRunInfo(jobName);
        if (runInfo != null && runInfo.getState() == JobState.RUNNING) {
          final String jobRunId = runInfo.getId();
          final JobRun jobRun = _runEngine.ensureJobRun(jobName, jobRunId);
          if (jobRun.isTriggeredBy(triggeringBucket)) {
            triggeredJobs.add(jobRun);
          }
        }
      } catch (final Exception e) {
        _log.warn("Tried to check potentially triggered jobs, but got an error. Job '" + jobName
          + "' will not be triggered.", e);
      }
    }
    return triggeredJobs;
  }

  /**
   * Stores transient bulks in RunStorage.
   * 
   * @param jobRun
   *          job run
   * @param workflowRunId
   *          the id of the workflow run.
   * @param transientBulks
   *          the bulks to store in the runStorage
   * @throws RunStorageException
   *           exception while storing the information in RunStorage
   */
  private void storeTransientBulks(final JobRun jobRun, final String workflowRunId,
    final List<BulkInfo> transientBulks) throws RunStorageException {
    final String jobName = jobRun.getJobName();
    final String jobRunId = jobRun.getJobRunId();
    for (final BulkInfo bulkInfo : transientBulks) {
      _runStorage.addTransientBulk(jobName, jobRunId, workflowRunId,
        bulkInfo.getStoreName() + "/" + bulkInfo.getObjectName());
    }
  }

  /** TODO remove this hack to copy task parameters in follow up tasks. */
  private AnyMap getTaskParametersToCopy(final Task currentTask) {
    final AnyMap taskParams = currentTask.getParameters();
    final AnyMap paramsToCopy = DataFactory.DEFAULT.createAnyMap();
    for (final Entry<String, Any> param : taskParams.entrySet()) {
      if (param.getKey().startsWith(JobManagerConstants.SYSTEM_PARAMETER_PREFIX)
        && !param.getKey().startsWith(JobManagerConstants.TEMP_PREFIX)) {
        paramsToCopy.put(param.getKey(), param.getValue());
      }
    }
    return paramsToCopy;
  }

  /**
   * cleanup possibly created workflow run data after failure during start. Exceptions during clean up are logged.
   * 
   * @param jobName
   *          job name
   * @param jobRunId
   *          job run id
   * @param workflowRunId
   *          workflow run id. may be null, then nothing has to be done.
   * @param task
   *          inital task (maybe "null")
   */
  private void cleanupFailedWorkflowRun(final String jobName, final String jobRunId, final String workerName,
    final String workflowRunId, final Task task) {
    if (jobRunId != null && workflowRunId != null) {
      try {
        handleFatalError(jobName, jobRunId, workflowRunId, workerName, task, false);
      } catch (final Exception ex) {
        _log.error("Error during cleanup of failed new workflow run '" + jobRunId + "' for job '" + jobName + "'.",
          ex);
      }
    }
  }

  /**
   * Creates the initial task for this job run's new workflow run.
   * 
   * @param workerName
   *          The name of the worker requesting the initial task.
   * @param workflowRunId
   *          The id for the new workflow run.
   * @return An initial task for the new job run.
   * @throws JobManagerException
   *           An error occurred while checking if an initial task could be created or while creating the initial task.
   */
  private Task getInitialTask(final JobRun jobRun, final String workerName, final String workflowRunId)
    throws JobManagerException {
    final WorkflowAction startAction = jobRun.getStartAction();
    if (startAction == null || !workerName.equals(startAction.getWorker())
      || !_defPersistence.getWorker(workerName).getModes().contains(Mode.BULKSOURCE)) {
      throw new JobManagerException("Worker '" + workerName + "' is not defined as start action for job '"
        + jobRun.getJobName() + "' or has no '" + Mode.BULKSOURCE.toString() + "' mode.");
    }
    final List<Task> taskList = generateTasksForAction(jobRun, workflowRunId, null, startAction, null);
    // there must be only one task
    if (taskList.size() != 1) {
      throw new JobManagerException("Not exactly one initial task produced for worker '" + workerName
        + "' in workflow run '" + workflowRunId + "'.");
    }
    return taskList.get(0);
  }

  /**
   * Gets the follow-up tasks for workers within the actual workflow run depending on a list of bulks.
   * 
   * Note that all tasks returned by this method will have thir workflowRunId parameter set to "null" since there will
   * be no workflow run started. You will have to set this parameter for any task in the returned collection.
   * 
   * @param bucketNameToReallyCreatedBulksMap
   *          The bulks potentially triggering tasks for follow up workers.
   * @return A collection of tasks triggered by the bulks within this workflow for the given workflow run.
   */
  private List<Task> getTriggeredInitialTasks(final JobRun jobRun,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap) throws JobManagerException {
    // only for the initial action
    // note: we are setting null as workflow run, the workflow-run-id has to be set outside!
    return generateTasksForAction(jobRun, null, bucketNameToReallyCreatedBulksMap, jobRun.getStartAction(), null);
  }

  /**
   * Gets the follow-up tasks for workers within the actual workflow run depending on a list of bulks.
   * 
   * @param workflowRunId
   *          The id of the current workflow run.
   * @param bucketNameToReallyCreatedBulksMap
   *          The bulks potentially triggering tasks for follow up workers.
   * @param taskParamsToCopy
   *          the parameters from the current task to copy to the follow up tasks
   * @return A collection of tasks triggered by the bulks within this workflow for the given workflow run. error while
   *         generating new tasks
   */
  private List<Task> getFollowupTasks(final JobRun jobRun, final String workflowRunId,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap, final AnyMap taskParamsToCopy)
    throws TaskGeneratorException {
    final List<Task> followUpTasks = new ArrayList<Task>();
    // check for follow up workflow actions for given bulks
    final Collection<WorkflowAction> followUpActions =
      getFollowupWorkflowActions(jobRun, bucketNameToReallyCreatedBulksMap.keySet());

    // for all follow up actions: find appropriate TaskGenerator and generate follow up tasks
    for (final WorkflowAction action : followUpActions) {
      followUpTasks.addAll(generateTasksForAction(jobRun, workflowRunId, bucketNameToReallyCreatedBulksMap, action,
        taskParamsToCopy));
    }
    return followUpTasks;
  }

  /**
   * Returns a set of follow up actions within a workflow. Follow up actions have to be unique!
   * 
   * @param bucketNames
   *          a collection of bucket names,.
   * @return A set of unique follow-up actions. Empty (but not null) if there are none.
   */
  private Collection<WorkflowAction> getFollowupWorkflowActions(final JobRun jobRun,
    final Collection<String> bucketNames) {
    final Set<WorkflowAction> followUpActions = new HashSet<WorkflowAction>();

    for (final String bucketName : bucketNames) {
      final Collection<WorkflowAction> actions = jobRun.getTriggeredActionsForBucket(bucketName);
      if (actions != null) {
        followUpActions.addAll(actions);
      }
    }
    return followUpActions;
  }

  /**
   * Generates a task for an action.
   * 
   * @param workflowRunId
   *          the workflow run id
   * @param bucketNameToReallyCreatedBulksMap
   *          the map from bucket name as key to BulkInfo-List as value
   * @param action
   *          the action to generate the tasks for
   * @param taskParamsToCopy
   *          the parameters from the current task to copy to the follow up tasks
   * @return A list of tasks for an action
   * @throws TaskGeneratorException
   *           error while generating new tasks
   */
  private List<Task> generateTasksForAction(final JobRun jobRun, final String workflowRunId,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap, final WorkflowAction action,
    final AnyMap taskParamsToCopy) throws TaskGeneratorException {

    final WorkerDefinition workerDef = _defPersistence.getWorker(action.getWorker());
    final TaskGenerator taskGenerator = _taskGeneratorProvider.getTaskGenerator(workerDef);

    // get parameters
    final AnyMap paramExpressions = DataFactory.DEFAULT.createAnyMap();
    if (jobRun.getParameters() != null) {
      paramExpressions.putAll(jobRun.getParameters());
    }
    if (action.getParameters() != null) {
      paramExpressions.putAll(action.getParameters());
    }
    final AnyMap parameters = ExpressionUtil.evaluateParameters(paramExpressions);
    if (taskParamsToCopy != null) {
      parameters.putAll(taskParamsToCopy);
    }

    final Map<String, Bucket> inputSlotNameToBucket = jobRun.getInputBucketsForAction(action);
    final Map<String, Bucket> outputSlotNameToBucketMap = jobRun.getOutputBucketsForAction(action);

    final MultiValueMap<String, BulkInfo> changedInputSlotToBulkListMap =
      prepareInputSlots(action, bucketNameToReallyCreatedBulksMap);

    // let the task generator create the tasks
    final long start = System.currentTimeMillis();
    final List<Task> tasks =
      taskGenerator.createTasks(changedInputSlotToBulkListMap, inputSlotNameToBucket, outputSlotNameToBucketMap,
        parameters, action.getWorker());
    final long t = System.currentTimeMillis() - start;
    if (t > MIN_TASK_GENERATION_TIME_TO_LOG) {
      _log.info("TaskGenerator time for creating tasks: " + t);
    }

    TaskGenerationUtil.setAdditionalTaskProperties(tasks, jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId,
      workerDef);

    if (taskParamsToCopy != null) {
      for (final Task followUpTask : tasks) {
        for (final Entry<String, Any> param : taskParamsToCopy.entrySet()) {
          followUpTask.getParameters().put(param.getKey(), param.getValue());
        }
      }
    }
    return tasks;
  }

  /**
   * associate the new input bulks that triggered this action with the slotnames.
   * 
   * @param action
   *          the action to generate the tasks for
   * @param bucketNameToReallyCreatedBulksMap
   *          the map from bucket name as key to BulkInfo-List as value
   * @return changedInputSlotToBulkListMap
   */
  private MultiValueMap<String, BulkInfo> prepareInputSlots(final WorkflowAction action,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap) {
    // prepare changed input buckets for new task
    final MultiValueMap<String, BulkInfo> changedInputSlotToBulkListMap = new MultiValueMap<String, BulkInfo>();
    final Map<String, String> actionInput = action.getInput();
    if (actionInput != null) {
      for (final Map.Entry<String, String> actionInputEntry : actionInput.entrySet()) {
        final List<BulkInfo> bulkInfoList = bucketNameToReallyCreatedBulksMap.get(actionInputEntry.getValue());
        if (bulkInfoList != null) {
          for (final BulkInfo inputBulk : bulkInfoList) {
            if (inputBulk != null) {
              if (!changedInputSlotToBulkListMap.containsKey(actionInputEntry.getKey())) {
                changedInputSlotToBulkListMap.put(actionInputEntry.getKey(), new ArrayList<BulkInfo>());
              }
              changedInputSlotToBulkListMap.get(actionInputEntry.getKey()).add(inputBulk);
            }
          }
        }
      }
    }
    return changedInputSlotToBulkListMap;
  }

  /** set OSGI service. */
  public void setDefinitionPersistence(final DefinitionPersistence defPersistence) {
    _defPersistence = defPersistence;
  }

  /** unset OSGI service. */
  public void unsetDefinitionPersistence(final DefinitionPersistence defPersistence) {
    if (_defPersistence == defPersistence) {
      _defPersistence = null;
    }
  }

  /** set OSGI service. */
  public void setRunStorage(final RunStorage runStorage) {
    _runStorage = runStorage;
  }

  /** unset OSGI service. */
  public void unsetRunStorage(final RunStorage runStorage) {
    if (_runStorage == runStorage) {
      _runStorage = null;
    }
  }

  /** set OSGI service. */
  public void setObjectStoreService(final ObjectStoreService objectStore) {
    _objectStore = objectStore;
  }

  /** unset OSGI service. */
  public void unsetObjectStoreService(final ObjectStoreService objectStore) {
    if (_objectStore == objectStore) {
      _objectStore = null;
    }
  }

  /** set OSGI service. */
  public void setTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    _taskGeneratorProvider = taskGeneratorProvider;
  }

  /** unset OSGI service. */
  public void unsetTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    if (_taskGeneratorProvider == taskGeneratorProvider) {
      _taskGeneratorProvider = null;
    }
  }

  /** set OSGI service. */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /** unset OSGI service. */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }

  /** set OSGI service. */
  public void setClusterConfigService(final ClusterConfigService clusterConfigService) {
    _clusterConfigService = clusterConfigService;
  }

  /** unset OSGI service. */
  public void unsetClusterConfigService(final ClusterConfigService clusterConfigService) {
    if (_clusterConfigService == clusterConfigService) {
      _clusterConfigService = null;
    }
  }

  /** set OSGI service. */
  public void setJobRunEngine(final JobRunEngine runEngine) {
    _runEngine = runEngine;
  }

  /** unset OSGI service. */
  public void unsetJobRunEngine(final JobRunEngine runEngine) {
    if (_runEngine == runEngine) {
      _runEngine = null;
    }
  }

}
