/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework.httphandler;

import java.util.Map;

import org.eclipse.smila.connectivity.framework.CrawlState;
import org.eclipse.smila.connectivity.framework.CrawlerController;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.util.URLCreator;

/**
 * Implements the handling of HTTP requests to request the crawlers list. <br>
 * 
 * URL pattern: <code>smila/crawlers/</code> <br>
 * 
 * Methods allowed: <code>GET</code>
 */
public class CrawlersHandler extends BaseConnectivityHandler {

  /** key for the crawlers. */
  public static final String KEY_CRAWLERS = "crawlers";

  /** key for the name. */
  public static final String KEY_NAME = "name";

  /** key for the state. */
  public static final String KEY_STATE = "state";

  /** key for the url. */
  public static final String KEY_URL = "url";

  /** Reference to the {@link CrawlerController}. */
  private CrawlerController _crawlerController;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final AnyMap resultMap = DataFactory.DEFAULT.createAnyMap();
    final AnySeq stateSeq = DataFactory.DEFAULT.createAnySeq();
    if (_crawlerController != null) {
      final Map<String, CrawlState> crawlerTaskStates = _crawlerController.getCrawlerTasksState();
      for (final String availableCrawler : _crawlerController.getAvailableConfigurations()) {
        final AnyMap stateMap = DataFactory.DEFAULT.createAnyMap();
        final CrawlState state = crawlerTaskStates.get(availableCrawler);
        String stateString;
        if (state != null) {
          stateString = state.getState().name();
        } else {
          stateString = "Undefined";
        }
        stateMap.put(KEY_NAME, availableCrawler);
        stateMap.put(KEY_STATE, stateString);
        stateMap.put(KEY_URL, URLCreator.create(getRequestHost(), requestUri, availableCrawler).toURI()
          .toASCIIString());
        stateSeq.add(stateMap);
      }
      resultMap.put(KEY_CRAWLERS, stateSeq);
    }
    return resultMap;
  }

  /**
   * {@inheritDoc}
   * 
   * only GET is allowed.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * OSGi method to set the {@link CrawlerController}. Called by DS.
   * 
   * @param crawlerController
   */
  protected void setCrawlerController(final CrawlerController crawlerController) {
    _crawlerController = crawlerController;
  }

  /**
   * OSGi method to unset the {@link CrawlerController}. Called by DS.
   * 
   * @param crawlerController
   */
  protected void unsetCrawlerController(final CrawlerController crawlerController) {
    if (_crawlerController == crawlerController) {
      _crawlerController = null;
    }
  }

}
