/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation
 *               Andreas Weber (Attensity Europe GmbH) - data model simplification
 **********************************************************************************************************************/

package org.eclipse.smila.processing.pipelets.xmlprocessing;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.pipelets.ATransformationPipelet;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;
import org.eclipse.smila.utils.xml.XMLUtils;
import org.w3c.dom.Document;

/**
 * Abstract base class for XML processing pipelets.
 */
public abstract class AXmlTransformationPipelet extends ATransformationPipelet {

  /**
   * Creates a Document from an Attribute or Attachment.
   * 
   * @param blackboard
   *          the Blackboard
   * @param id
   *          the Id of the record
   * @return a Document
   * @throws Exception
   *           if any error occurs
   */
  protected Document createDocument(final Blackboard blackboard, final String id,
    final ParameterAccessor paramAccessor) throws Exception {
    Document inputDocument = null;
    if (isReadFromAttribute(getInputType(paramAccessor))) {
      final String value = blackboard.getMetadata(id).getStringValue(getInputName(paramAccessor));
      if (value != null) {
        inputDocument = XMLUtils.parse(value.getBytes(ENCODING_ATTACHMENT), false);
      }
    } else {
      final byte[] value = blackboard.getAttachmentAsBytes(id, getInputName(paramAccessor));
      if (value != null) {
        inputDocument = XMLUtils.parse(value, false);
      }
    }
    return inputDocument;
  }

  /**
   * Stores a Document in an Attribute or Attachment.
   * 
   * @param blackboard
   *          the Blackboard
   * @param id
   *          the Id of the record
   * @param doc
   *          the Document to save
   * @throws Exception
   *           if any error occurs
   */
  protected void storeDocument(final Blackboard blackboard, final String id, final Document doc,
    final ParameterAccessor paramAccessor) throws Exception {
    if (isStoreInAttribute(getOutputType(paramAccessor))) {
      final String domString = XMLUtils.documentToString(doc);
      blackboard.getMetadata(id).put(getOutputName(paramAccessor), domString);
    } else {
      final byte[] attachment = XMLUtils.documentToBytes(doc);
      blackboard.setAttachment(id, getOutputName(paramAccessor), attachment);
    }
  }

  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    if (recordIds != null) {
      final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _config);
      final ResultCollector resultCollector =
        new ResultCollector(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
      for (final String id : recordIds) {
        paramAccessor.setCurrentRecord(id);
        try {
          processRecord(blackboard, paramAccessor, id);
          resultCollector.addResult(id);
        } catch (Exception e) {
          resultCollector.addFailedResult(id, e);
        }
      }
      return resultCollector.getResultIds();
    }
    return null;
  }

  /**
   * Processing of a single input record.
   * 
   * @param blackboard
   *          Blackboard service managing the records.
   * @param paramAccessor
   *          Parameter accessor configured with pipelet configuration and current record. Should be used to access
   *          pipelet parameters.
   * @param recordId
   *          id of the record to process
   */
  protected abstract void processRecord(final Blackboard blackboard, final ParameterAccessor paramAccessor,
    final String recordId) throws Exception;

}
