/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;

import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.TaskManager;

/**
 * Implements the handling of HTTP requests on a single task.
 * 
 * URL pattern: taskmanager/-workername-/-taskid-
 * 
 * Currently a task can be finished by POSTing a result description at the task URI or a keepalive can be sent by
 * POSTing without data.
 */
public class TaskHandler extends JsonRequestHandler {

  /** The reference to the TaskManager service. */
  private TaskManager _taskManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if ("POST".equals(method)) {
      final String workerName = getWorkerName(requestUri);
      final String taskId = getTaskId(requestUri);

      if (inputRecord == null) {
        _taskManager.keepAlive(workerName, taskId);
      } else {
        final ResultDescription resultDescription = ResultDescription.fromAny(inputRecord.getMetadata());
        _taskManager.finishTask(workerName, taskId, resultDescription);
      }
    }
    return null;
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalArgumentException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex instanceof BadParameterTaskmanagerException
      && ((BadParameterTaskmanagerException) ex).getCauseCode().equals(
        BadParameterTaskmanagerException.Cause.taskId)) {
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof BadParameterTaskmanagerException
      && ((BadParameterTaskmanagerException) ex).getCauseCode().equals(
        BadParameterTaskmanagerException.Cause.workerName)) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String requestMethod, final String requestUri, final Record inputRecord,
    final Object resultObject) {
    if (inputRecord == null) {
      // was a keep alive
      return HttpStatus.ACCEPTED;
    }
    return super.getSuccessStatus(requestMethod, requestUri, inputRecord, resultObject);
  }

  /**
   * Gets the job name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return he job name from the requestUri.
   */
  private String getWorkerName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * Gets the job id from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return he job name from the requestUri.
   */
  private String getTaskId(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 1) {
      return uriParts.get(1);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * GET and DELETE are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return ("POST".equals(method));
  }

  /**
   * @param taskManager
   *          the reference to the JobManager service.
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * @param taskManager
   *          the reference to the current JobManager service.
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }

}
