/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.search;

import static java.lang.String.format;
import static org.apache.commons.lang.StringUtils.defaultString;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.startsWith;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrQuery.ORDER;
import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.FacetParams;
import org.apache.solr.common.params.ShardParams;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.solr.SolrConstants;
import org.eclipse.smila.solr.util.SolrQueryUtils;

/**
 * This class converts a smila's solr search record into a solr(j) query.
 * 
 * @author tmenzel
 * 
 */
public class SolrQueryConverter {
  /** The logger. */
  protected final org.apache.commons.logging.Log _log = org.apache.commons.logging.LogFactory.getLog(this
    .getClass());

  /**
   * The SolrQueryParameterAccessor.
   */
  private final SolrQueryParameterAccessor _accessor;

  /**
   * The SolrQuery.
   */
  private final SolrQuery _solrQuery = new SolrQuery();

  /**
   * Constructor.
   * 
   * @param accessor
   *          the solr query parameter accessor.
   */
  public SolrQueryConverter(SolrQueryParameterAccessor accessor) {
    _accessor = accessor;
  }

  /**
   * Convert the record to a solr query string.
   * 
   * @return the SolrQuery.
   */
  public SolrQuery toSolrQuery(final List<String> schemaAttributes) {
    final String qt = _accessor.getRequestHandler();
    _solrQuery.setQueryType(qt);

    doQuery(schemaAttributes);
    doSortOrder();
    doTermsSettings();
    doFilters();
    doFacetSettings();
    if (_solrQuery.getQuery() != null || _solrQuery.getFields() != null) {
      doQuerySettings();
      doHighlightingSettings();
      doShardsSettings();
      doSpellCheckSettings();
      doMoreLikeThis();
    }

    final AnyMap map = _accessor.getSolrQueryParams().getMap(QueryConstants.NATIVE_PARAMETERS);
    addAsSolrParameters(map, null);

    return _solrQuery;
  }

  /**
   * build the query string. Takes it either from smila's query string and which must be a native string then OR if that
   * is empty constructs one from smila's fielded search syntax.
   */
  private void doQuery(final List<String> schemaAttributes) {
    String q = _accessor.getQuery();
    if (StringUtils.isEmpty(q)) {
      // search in dedicated fields instead of simple query string
      if (schemaAttributes != null && !schemaAttributes.isEmpty()) {
        final StringBuilder fieldQuery = new StringBuilder();
        for (final String field : schemaAttributes) {
          final List<Value> fieldQueryValues = _accessor.getQueryAttributeValues(field);
          if (fieldQueryValues != null) {
            // there seems to be no SolrQuery API for querying fields so we have to build our query manually
            SolrQueryUtils.appendFieldQueryPart(fieldQuery, field, fieldQueryValues);
          }
        }
        q = fieldQuery.toString().trim();
      }
      // Filters are added in doFilterSettings()
    }
    if (!StringUtils.isEmpty(q)) {
      _solrQuery.setQuery(q);
    }
  }

  /**
   * Do query settings such as offset, maxCount, ...
   */
  private void doQuerySettings() {
    // Paging
    final int start = _accessor.getOffset();
    _solrQuery.setStart(start);
    final int rows = _accessor.getMaxCount();
    _solrQuery.setRows(rows);
    // Field list
    final String[] fl = _accessor.getResultAttributes().toArray(new String[_accessor.getResultAttributes().size()]);
    _solrQuery.setFields(fl);
    // TODO: setIncludeScore is evidently not working, add score field manually.
    // _solrQuery.setIncludeScore(true);
    // must always have these or else building the result will fail
    _solrQuery.addField(SolrConstants.CORE_FIELD_SCORE);
    _solrQuery.addField(SolrConstants.CORE_FIELD_ID);

  }

  /**
   * 
   */
  private void doSortOrder() {
    // for (Entry<String, SortOrder> item : _accessor.getSortByConfig().entrySet()) {
    // final ORDER sortOrder = item.getValue() == SortOrder.ASCENDING ? ORDER.asc : ORDER.desc;
    // _solrQuery.addSortField(item.getKey(), sortOrder);
    // }

    // faster impl. than above but more prone to migration changes
    final List<AnyMap> annotations = _accessor.getSubParameters(QueryConstants.SORTBY);
    if (annotations != null) {
      for (final AnyMap annotation : annotations) {
        final String attributeName = annotation.getStringValue(QueryConstants.ATTRIBUTE);
        final String orderModeValue = annotation.getStringValue(QueryConstants.ORDER);
        if (startsWith(orderModeValue, "asc")) {
          _solrQuery.addSortField(attributeName, ORDER.asc);
        } else {
          _solrQuery.addSortField(attributeName, ORDER.desc);
        }
      }
    }
  }

  /**
   * Do filter settings.
   * 
   * Will add native _solr.query/fq filters as well as the SMILA filters.
   */
  private void doFilters() {
    addFilterSolrSyntax();

    if (_accessor.hasFilters()) {
      addFilterSmilaSyntax();
    }

  }

  /**
   * for each filter defined, adds one fq param to the solr query. All FQs are ANDed (mandatory). This method is the
   * head just iterating thru the Seq of filter definitions usually one per attribute.
   */
  private void addFilterSmilaSyntax() {
    final List<AnyMap> filters = _accessor.getSubParameters(QueryConstants.FILTER);

    // instead of creating a new buffer each time this is reused and truncated each time
    final StringBuilder fq = new StringBuilder(0xfff);
    for (int i = 0; i < filters.size(); i++) {
      final AnyMap filter = filters.get(i);

      final String attribute = filter.getStringValue(QueryConstants.ATTRIBUTE);
      if (attribute == null) {
        _log.warn(MessageFormat.format("no attribute defined for given filter @ index: {0}. It is ignored.", i));
        continue;
      }

      addFilterSmilaSyntax(fq, attribute, filter, i);
    }

  }

  /**
   * adds the filters for the given attribute.
   */
  private void addFilterSmilaSyntax(final StringBuilder fq, final String attribute, final AnyMap filter, int i) {
    for (Entry<String, Any> condition : filter.entrySet()) {
      fq.setLength(0); // reset it

      final String filterType = condition.getKey();

      if (QueryConstants.ATTRIBUTE.equals(filterType)) {
        // dont do nothing here as this element has been read before
        continue;
      } else if (QueryConstants.FILTER_ONEOF.equals(filterType)) {
        appendFilterSetExpression(fq, attribute, condition.getValue().asSeq(), "");
      } else if (QueryConstants.FILTER_ALLOF.equals(filterType)) {
        appendFilterSetExpression(fq, attribute, condition.getValue().asSeq(), "+");
      } else if (QueryConstants.FILTER_NONEOF.equals(filterType)) {
        appendFilterSetExpression(fq, attribute, condition.getValue().asSeq(), "-");
      } else if (QueryConstants.FILTER_ATLEAST.equals(filterType)) {
        final String value = condition.getValue().asValue().asString();
        addFilterRangeExpression(attribute, fq, value, "*", false);
      } else if (QueryConstants.FILTER_GREATERTHAN.equals(filterType)) {
        final String value = condition.getValue().asValue().asString();
        addFilterRangeExpression(attribute, fq, value, "*", true);
      } else if (QueryConstants.FILTER_ATMOST.equals(filterType)) {
        final String value = condition.getValue().asValue().asString();
        addFilterRangeExpression(attribute, fq, "*", value, false);
      } else if (QueryConstants.FILTER_LESSTHAN.equals(filterType)) {
        final String value = condition.getValue().asValue().asString();
        addFilterRangeExpression(attribute, fq, "*", value, true);
      } else {
        _log.warn(format("unknown (filter) element in filters encountered on attribute %s: %s[%s]. It is ignored.",
          attribute, filterType, i));
      }

      if (fq.length() == 0) {
        _log.warn(format("Filter on attribute %s: %s[%d] has no values and is ignored!", attribute, filterType, i));
      } else {
        final String fqString = fq.toString();
        if (_log.isDebugEnabled()) {
          _log.debug(format("Filter converted from Smila Syntax on attribute %s: %s[%d]: %s", attribute,
            filterType, i, fqString));
        } // if
        _solrQuery.addFilterQuery(fqString);
      }
    }
  }

  /**
   * @param attribute
   * @param fq
   */
  private void addFilterRangeExpression(final String attribute, final StringBuilder fq, String lower, String upper,
    boolean exclusive) {

    if (isBlank(lower)) {
      throw new IllegalArgumentException("lower bound must not be blank");
    }
    if (isBlank(upper)) {
      throw new IllegalArgumentException("upper bound must not be blank");
    }
    lower = appendFilterRangeExpressionBoundExclusionIfNotStar(lower, exclusive, fq, attribute);
    upper = appendFilterRangeExpressionBoundExclusionIfNotStar(upper, exclusive, fq, attribute);

    fq.append(" +(");
    fq.append(attribute);
    fq.append(":[");
    fq.append(lower);
    fq.append(" TO ");
    fq.append(upper);
    fq.append("])");
  }

  /**
   * appends a not expresion if a range bound if it is {@code != "*" } and exlusive is true.
   */
  private String appendFilterRangeExpressionBoundExclusionIfNotStar(String value, boolean exclusive,
    StringBuilder fq, String attribute) {
    if (!"*".equals(value)) {
      value = SolrQueryUtils.escapeQuery(value, SolrQueryUtils.ESCAPE_CHARS_WS);

      // in case of exclusive we must add a NOT query
      if (exclusive) {
        fq.append(" -(");
        fq.append(attribute);
        fq.append(":");
        fq.append(value);
        fq.append(")");
      }
    }
    return value;
  }

  /**
   * converts the smila filter syntax into lucene's for the filters: oneOf, noneOf, allOf.
   * 
   * @param values
   *          TODO
   */
  private void appendFilterSetExpression(final StringBuilder fq, final String attribute, AnySeq values,
    String operator) {

    final List<String> filterValues;
    try {
      filterValues = values.asStrings();
    } catch (InvalidValueTypeException e) {
      throw new InvalidValueTypeException(
        "Filter Syntax Error: Filter must be of Type Seq and contain only Val elements.", e);
    }

    for (final String filterValue : filterValues) {
      fq.append(operator);
      fq.append("(");
      fq.append(attribute);
      fq.append(":");
      fq.append(SolrQueryUtils.escapeQuery(filterValue, SolrQueryUtils.ESCAPE_CHARS_WS));
      fq.append(")");
    }

  }

  /**
   * just adds all {@link SolrConstants#QUERY_MAP}/fq Vals as filters to solr.
   */
  private void addFilterSolrSyntax() {
    final AnySeq seq = _accessor.getFilterQuery();
    if (seq != null) {
      final String[] fq = seq.asStrings().toArray(new String[seq.size()]);
      _solrQuery.setFilterQueries(fq);
    }
  }

  /**
   * Do facet settings.
   */
  private void doFacetSettings() {
    final List<AnyMap> facetByConfig = _accessor.getFacetByConfig();
    if (!facetByConfig.isEmpty()) {
      final StringBuilder fq = new StringBuilder(0xfff);
      // turn on facetting if the map is present
      _solrQuery.add(FacetParams.FACET, "true");

      int facetByIndex = 0;
      for (AnyMap facetConfig : facetByConfig) {
        facetByIndex++;
        /*
         * PERF: do this faster by just iterating thru all childeren and collecting all vars on the fly instead of
         * pulling them from the map | TM @ Jan 18, 2012
         */

        final String attribute = facetConfig.getStringValue(QueryConstants.ATTRIBUTE);
        if (attribute == null) {
          _log.warn(MessageFormat.format("no attribute defined for facet @ index: {0}. It is ignored.",
            facetByIndex));
          continue;
        }
        /* TODO | TM | facetting: apply record -> solr mapping ?| TM @ Jan 18, 2012 */
        final String fieldName = attribute;

        // add attribute/field to facet set
        final String facetType =
          defaultString(facetConfig.getStringValue(SolrConstants.FACET_TYPE), FacetParams.FACET_FIELD);
        _solrQuery.add(facetType, fieldName);

        final String maxCount = facetConfig.getStringValue(QueryConstants.MAXCOUNT);
        if (maxCount != null) {
          addFieldParameter(fieldName, FacetParams.FACET_LIMIT, maxCount);
        }

        final AnyMap sortby = facetConfig.getMap(QueryConstants.SORTBY);
        if (sortby != null) {
          final String criterion = sortby.getStringValue(QueryConstants.FACETBY_SORTCRITERION);
          addFieldParameter(fieldName, FacetParams.FACET_SORT, criterion);

          if (_log.isWarnEnabled()) {
            if (sortby.containsKey(QueryConstants.ORDER)) {
              _log.warn(format(
                "facet config for field %s contains value for unsupported sort order. It is ignored", fieldName));
            }
          } // if
        }

        // add all native params as given
        final AnyMap map = facetConfig.getMap(QueryConstants.NATIVE_PARAMETERS);
        addAsSolrParameters(map, fieldName);

        // support filter expressions local to facetby config
        final AnySeq filters = facetConfig.getSeq(QueryConstants.FILTER_ONEOF);
        if (filters != null) {
          // instead of creating a new buffer each time this is reused and truncated each time
          appendFilterSetExpression(fq, attribute, filters, "");
        }
        if (fq.length() > 0) {
          _solrQuery.addFilterQuery(fq.toString());
          fq.setLength(0);
        }

      } // facet config

    }
  }

  /**
   * Do terms settings.
   */
  private void doTermsSettings() {
    final AnyMap terms = _accessor.getTerms();
    if (terms != null) {
      _solrQuery.setParam(CommonParams.QT, "/terms");
      addAsSolrParameters(terms, null);
    }
  }

  /**
   * Do highlighting settings.
   */
  private void doHighlightingSettings() {
    final AnySeq seq = _accessor.getHighlighting();
    if (seq != null) {
      for (Any map : seq) {
        if (map.isMap()) {
          final AnyMap highlighting = map.asMap();
          final String attribute = highlighting.getStringValue(QueryConstants.ATTRIBUTE);
          if (attribute.equals(SolrConstants.GLOBAL)) {
            addAsSolrParameters(highlighting, null);
          } else {
            addAsSolrParameters(highlighting, attribute);
          }
        }
      }
    }
  }

  /**
   * Adds contained values as solr query parameters. Only Value items is supported. An item with the name "attribute" is
   * skipped.
   * 
   * @param map
   *          the map containing parameter as key value pairs. If null does nothing.
   * @param field
   *          if not blank then the config is added for the given field name (field level config, e.g.
   *          f.${field}.${key}) otherwise global
   */
  private void addAsSolrParameters(AnyMap map, String field) {
    if (map == null) {
      return;
    }
    for (Entry<String, Any> entry : map.entrySet()) {

      final String key = entry.getKey();
      if (key.equals(QueryConstants.ATTRIBUTE)) {
        continue;
      }
      final String value = entry.getValue().asValue().asString();
      if (isBlank(field)) {
        _solrQuery.add(key, value);
      } else {
        addFieldParameter(field, key, value);
      }
    }
  }

  /**
   * adds the given parameter for a field after the form f.${field}.${paramName}.
   */
  private void addFieldParameter(final String field, final String paramName, final String value) {
    final String parmName = SolrConstants.FIELD_PREFIX + field + SolrConstants.FIELD_SUFFIX + paramName;
    _solrQuery.add(parmName, value);
  }

  /**
   * Do shards settings.
   */
  private void doShardsSettings() {
    final AnySeq seq = _accessor.getShards();
    if (seq != null) {
      final String shards = StringUtils.join(seq.asStrings(), ",");
      _solrQuery.setParam(ShardParams.SHARDS, shards);
    }
  }

  /**
   * Do spellcheck settings.
   */
  private void doSpellCheckSettings() {
    final AnyMap spellcheck = _accessor.getSpellcheck();
    if (spellcheck != null) {
      addAsSolrParameters(spellcheck, null);
    }
  }

  /**
   * 
   */
  private void doMoreLikeThis() {
    final AnyMap moreLikeThis = _accessor.getMoreLikeThis();
    if (moreLikeThis != null) {
      addAsSolrParameters(moreLikeThis, null);
    }

  }

}
