/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.util;

import org.apache.commons.lang.StringUtils;
import org.apache.solr.common.params.HighlightParams;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.solr.SolrConstants;

/**
 * HighlightingQueryConfigAdapter class.
 * 
 * @author pwissel
 * 
 */
public class HighlightingQueryConfigAdapter {

  /**
   * HighlightingType enum.
   * 
   * @author pwissel
   * 
   */
  public enum HighlightingType {
    /**
     * Global.
     */
    GLOBAL,
    /**
     * Field.
     */
    FIELD;
  }

  /**
   * HighlightingFragmenter enum.
   * 
   * @author pwissel
   * 
   */
  public enum HighlightingFragmenter {
    /**
     * Gap.
     */
    GAP,
    /**
     * Regex
     */
    REGEX;
    @Override
    public String toString() {
      return this.name().toLowerCase();
    };
  }

  /**
   * The internal map.
   */
  private final AnyMap _map = DataFactory.DEFAULT.createAnyMap();

  /**
   * Default Constructor.
   */
  public HighlightingQueryConfigAdapter() {
    _map.put(QueryConstants.ATTRIBUTE, SolrConstants.GLOBAL);
    _map.put(HighlightParams.HIGHLIGHT, true);
  }

  /**
   * Constructor per field.
   * 
   * @param field
   *          the field.
   */
  public HighlightingQueryConfigAdapter(String field) {
    _map.put(QueryConstants.ATTRIBUTE, field);
  }

  /**
   * Get the underlying any map.
   * 
   * @return the map.
   */
  public AnyMap getAnyMap() {
    return _map;
  }

  /**
   * Set hl.fl parameter.
   * 
   * @param values
   *          the values.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingFields(String... values) {
    _map.put(HighlightParams.FIELDS, StringUtils.join(values, " "));
    return this;
  }

  /**
   * Set hl.snippets parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingSnippets(int value) {
    _map.put(HighlightParams.SNIPPETS, value);
    return this;
  }

  /**
   * Set hl.fragsize parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingFragsize(int value) {
    _map.put(HighlightParams.FRAGSIZE, value);
    return this;
  }

  /**
   * Set hl.mergeContiguous parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingMergeContiguous(Boolean value) {
    _map.put(HighlightParams.MERGE_CONTIGUOUS_FRAGMENTS, value);
    return this;
  }

  /**
   * Set hl.requireFieldMatch parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingRequireFieldMatch(Boolean value) {
    _map.put(HighlightParams.FIELD_MATCH, value);
    return this;
  }

  /**
   * Set hl.maxAnalyzedChars parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingMaxAnalyzedChars(int value) {
    _map.put(HighlightParams.MAX_CHARS, value);
    return this;
  }

  /**
   * Set hl.alternateField parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingAlternateField(String value) {
    _map.put(HighlightParams.ALTERNATE_FIELD, value);
    return this;
  }

  /**
   * Set hl.maxAlternateFieldLength parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingAlternateFieldLength(int value) {
    _map.put(HighlightParams.ALTERNATE_FIELD_LENGTH, value);
    return this;
  }

  /**
   * Set hl.simple.pre parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingSimplePre(String value) {
    _map.put(HighlightParams.SIMPLE_PRE, value);
    return this;
  }

  /**
   * Set hl.simple.post parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingSimplePost(String value) {
    _map.put(HighlightParams.SIMPLE_POST, value);
    return this;
  }

  /**
   * Set hl.fragmenter parameter.
   * 
   * @param fragmenter
   *          the fragmenter.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingFragmenter(HighlightingFragmenter fragmenter) {
    return setHighlightingFragmenter(fragmenter.toString());
  }

  /**
   * Set hl.fragmenter parameter.
   * 
   * @param fragmenter
   *          the fragmenter name.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingFragmenter(String fragmenter) {
    _map.put(HighlightParams.FRAGMENTER, fragmenter);
    return this;
  }

  /**
   * Set hl.usePhraseHighlighter parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingUsePhraseHighlighter(Boolean value) {
    _map.put(HighlightParams.USE_PHRASE_HIGHLIGHTER, value);
    return this;
  }

  /**
   * Set hl.highlinghtMultiTerm parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingMultiTerm(Boolean value) {
    _map.put(HighlightParams.HIGHLIGHT_MULTI_TERM, value);
    return this;
  }

  /**
   * Set hl.regex.slop parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingRegexSlop(Double value) {
    _map.put(HighlightParams.SLOP, value);
    return this;
  }

  /**
   * Set hl.regex.pattern parameter.
   * 
   * @param value
   *          the value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingRegexPattern(String value) {
    _map.put(HighlightParams.PATTERN, value);
    return this;
  }

  /**
   * Set hl.regexMaxAnalysedChars parameter.
   * 
   * @param value
   *          teh value.
   * @return HighlightingQueryConfigAdapter.
   */
  public HighlightingQueryConfigAdapter setHighlightingRegexMaxAnalysedChars(int value) {
    _map.put(HighlightParams.MAX_RE_CHARS, value);
    return this;
  }

}
