/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.taskgenerator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.osgi.service.component.ComponentContext;

/**
 * Base class for TaskGenerators that have their name configured in the OSGi component description as property
 * {@link #PROPERTY_GENERATOR_NAME}.
 */
public abstract class TaskGeneratorBase implements TaskGenerator {

  /**
   * name of property that specifies the generator name in the component description.
   */
  public static final String PROPERTY_GENERATOR_NAME = "generatorName";

  /** local logger. */
  protected final Log _log = LogFactory.getLog(getClass());

  /** service reference to objectstore. */
  protected ObjectStoreService _objectStore;

  /** task generator name. */
  private String _name;

  /**
   * Creates an unique id.
   * 
   * @return unique id
   */
  public static String createTaskId() {
    return UUID.randomUUID().toString();
  }

  /**
   * @return generator name, read from component description.
   */
  @Override
  public String getName() {
    return _name;
  }

  /** {@inheritDoc} */
  @Override
  public void finishTask(final Task task, final TaskCompletionStatus taskState) throws TaskGeneratorException {
    // do nothing here, is overwritten in specific task generators
  }

  /**
   * get value of {@link #PROPERTY_GENERATOR_NAME} in component description.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void activate(final ComponentContext context) {
    _name = context.getProperties().get(PROPERTY_GENERATOR_NAME).toString();
  }

  /**
   * @param workerName
   *          name of worker
   * @param parameters
   *          task parameters
   * @return task for worker and with given parameters.
   */
  protected Task createTask(final String workerName, final AnyMap parameters) {
    final String taskId = createTaskId();
    final Task task = new Task(taskId, workerName);
    task.getParameters().putAll(parameters);
    return task;
  }

  /**
   * create a bulk Id for each output bucket and add it to the task.
   * 
   * @param task
   *          task
   * @param inputBucket
   *          needed for special uuid handling
   * @param outputBuckets
   *          map of slot names to buckets
   * @param variableValues
   *          predefined (e.g. extracted from input bulks) values for data object type variables. May be null.
   */
  protected void addOutputBulks(final Task task, final Bucket inputBucket, final Map<String, Bucket> outputBuckets,
    final AnyMap variableValues) {
    if (isRecursion(inputBucket, outputBuckets)) {
      variableValues.remove(Bucket.UUID_PARAMETER);
    }
    for (final Map.Entry<String, Bucket> outputs : outputBuckets.entrySet()) {
      final String slot = outputs.getKey();
      final Bucket bucket = outputs.getValue();
      final List<BulkInfo> bulks = new ArrayList<BulkInfo>();
      bulks.add(bucket.createDataObject(variableValues));
      task.getOutputBulks().put(slot, bulks);
    }
  }

  /** check if one of the output buckets is the same as the input bucket. */
  private boolean isRecursion(final Bucket inputBucket, final Map<String, Bucket> outputBuckets) {
    if (inputBucket != null) {
      for (final Map.Entry<String, Bucket> outputs : outputBuckets.entrySet()) {
        if (inputBucket.getBucketId().equals(outputs.getValue().getBucketId())) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * check number of buckets passed as input and output. Use a negative number to disable a check.
   * 
   * @param inputBuckets
   *          .
   * @param expectedNumberOfInputBuckets
   *          .
   * @param outputBuckets
   *          .
   * @param expectedNumberOfOutputBuckets
   *          .
   * @throws TaskGeneratorException
   *           if size of bucket maps does not match the corresponding expectedNumber argument.
   */
  protected void checkBucketCounts(final Map<String, Bucket> inputBuckets, final int expectedNumberOfInputBuckets,
    final Map<String, Bucket> outputBuckets, final int expectedNumberOfOutputBuckets) throws TaskGeneratorException {
    if (expectedNumberOfInputBuckets >= 0 && inputBuckets.size() != expectedNumberOfInputBuckets) {
      throw new TaskGeneratorException("Expected exactly " + expectedNumberOfInputBuckets
        + " input bucket(s), but was: " + inputBuckets.size());
    }
    if (expectedNumberOfOutputBuckets >= 0 && outputBuckets.size() != expectedNumberOfOutputBuckets) {
      throw new TaskGeneratorException("Expected exactly " + expectedNumberOfOutputBuckets
        + " output bucket(s), but was: " + outputBuckets.size());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Task> createRunOnceTasks(final Map<String, Bucket> inputBuckets,
    final Map<String, Bucket> outputBuckets, final AnyMap parameters, final String workerName)
    throws TaskGeneratorException {
    throw new TaskGeneratorException("Cannot create tasks for the mode '" + JobRunMode.RUNONCE.name()
      + "' for worker '" + workerName + "' and input buckets '" + inputBuckets + "'.");
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param objectStore
   *          ObjectStoreService reference.
   */
  public void setObjectStoreService(final ObjectStoreService objectStore) {
    _objectStore = objectStore;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param objectStore
   *          ObjectStoreService reference.
   */
  public void unsetObjectStoreService(final ObjectStoreService objectStore) {
    if (_objectStore == objectStore) {
      _objectStore = null;
    }
  }

  /**
   * Delivers value from given string value. If something goes wrong, default value will be delivered.
   * 
   * @param parameterValue
   *          The given parameter value
   * @return The value
   */
  protected int getParameterValue(final String parameterValue, final int defaultValue) {
    try {
      if (parameterValue != null) {
        return new Integer(parameterValue);
      } else {
        return defaultValue;
      }
    } catch (final Exception e) {
      return defaultValue;
    }
  }

  /**
   * Delivers value from given string value. If something goes wrong, default value will be delivered.
   * 
   * @param parameterValue
   *          The given parameter value
   * @return The value
   */
  protected long getParameterValue(final String parameterValue, final long defaultValue) {
    try {
      if (parameterValue != null) {
        return new Long(parameterValue);
      } else {
        return defaultValue;
      }
    } catch (final Exception e) {
      return defaultValue;
    }
  }

  /**
   * Delivers value from given string value. If something goes wrong, default value will be delivered.
   * 
   * @param parameterValue
   *          The given parameter value
   * @return The value
   */
  protected double getParameterValue(final String parameterValue, final double defaultValue) {
    try {
      if (parameterValue != null) {
        return new Double(parameterValue);
      } else {
        return defaultValue;
      }
    } catch (final Exception e) {
      return defaultValue;
    }
  }

  /** {@inheritDoc} */
  @Override
  public void validateParameters(final AnyMap parameters) throws TaskGeneratorException {
    // do nothing here, overwrite in respective TaskGenerators
  }

  /**
   * The method extracts and overwrites the task parameters. Only the parameters that have not yet been set are
   * overwritten (and so no parameters of the workflow or the taskgenerator will be overwritten automatically). Only
   * parameters will be extracted and overwritten by the parameters derived from the object id of the incoming bucket.
   * This is done by using all parameters for the job/workflow/taskgenerator which do not start with temp prefix
   * ('_temp').
   * 
   * @param task
   *          The task with parameters
   * @param objectVariables
   *          The object variables
   */
  protected void extractAndOverwriteParameters(final Task task, final Map<String, String> objectVariables) {
    if (objectVariables != null && !objectVariables.isEmpty()) {
      final Set<Entry<String, String>> objectVariableEntries = objectVariables.entrySet();
      for (final Entry<String, String> entry : objectVariableEntries) {
        final String key = entry.getKey();
        if (!key.startsWith(JobManagerConstants.TEMP_PREFIX) && task.getParameters().get(key) == null) {
          task.getParameters().put(key, entry.getValue());
        }
      }
    }
  }

  /**
   * @return a task for the given input bulk
   */
  protected Task createSingleBulkTask(final String objectId, final String inputSlotName, final Bucket inputBucket,
    final Map<String, Bucket> outputBuckets, final AnyMap parameters, final String workerName) {
    final Map<String, String> objectVariables = inputBucket.getDataObjectVariableValues(objectId);
    final Task task = createTask(workerName, parameters);
    extractAndOverwriteParameters(task, objectVariables);
    task.getInputBulks().put(
      inputSlotName,
      Collections.singletonList(new BulkInfo(inputBucket.getBucketDefinition().getName(), inputBucket
        .getStoreName(), objectId)));
    addOutputBulks(task, inputBucket, outputBuckets, task.getParameters());
    return task;
  }
}
