/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation 
 **********************************************************************************************************************/

package org.eclipse.smila.restapi;

import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskManager;

/**
 * Implements the handling of requests to /smila:
 * 
 * <pre>
 * {
 *   "taskmanager" : {
 *     "todo" : 0,
 *    "inprogress" : 0
 *   },
 *   "api" : {
 *     "jobmanager" : "http://localhost:8080/smila/jobmanager/",
 *     "tasks" : "http://localhost:8080/smila/tasks/",
 *     "pipelines" : "http://localhost:8080/smila/pipeline/",
 *     "store" : "http://localhost:8080/smila/store/",
 *     "crawlers" : "http://localhost:8080/smila/crawlers/",
 *     "agents" : "http://localhost:8080/smila/agents/"
 *   }
 * }
 * </pre>
 */
public class StateHandler extends JsonRequestHandler {

  /** the taskmanager to use for getting task counters. */
  private TaskManager _taskManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) {
    final AnyMap result = FACTORY.createAnyMap();
    result.put("taskmanager", buildTaskManagerSection());
    result.put("api", buildApiSection(requestUri));
    return result;
  }

  /**
   * build taskmanager section from task counters.
   * 
   * @return Any, representing taskmanager section
   */
  private AnyMap buildTaskManagerSection() {
    final AnyMap taskManagerSection = FACTORY.createAnyMap();
    try {
      int todoCount = 0;
      int inprogressCount = 0;
      final Map<String, TaskCounter> counters = _taskManager.getTaskCounters();
      for (final TaskCounter counter : counters.values()) {
        todoCount += counter.getTasksTodo();
        inprogressCount += counter.getTasksInProgress();
      }
      taskManagerSection.put("todo", todoCount);
      taskManagerSection.put("inprogress", inprogressCount);
      return taskManagerSection;
    } catch (final Exception e) {
      return buildErrorSection(e);
    }
  }

  /**
   * Build "api" section.
   * 
   * @return Any, representing api section
   */
  private AnyMap buildApiSection(final String requestUri) {
    final AnyMap section = FACTORY.createAnyMap();
    try {
      section.put("jobmanager", URLCreator.create(getRequestHost(), requestUri, "jobmanager").toString());
      section.put("tasks", URLCreator.create(getRequestHost(), requestUri, "tasks").toString());
      section.put("pipelines", URLCreator.create(getRequestHost(), requestUri, "pipeline").toString());
      section.put("pipelets", URLCreator.create(getRequestHost(), requestUri, "pipelets").toString());
      section.put("store", URLCreator.create(getRequestHost(), requestUri, "store").toString());
      section.put("crawlers", URLCreator.create(getRequestHost(), requestUri, "crawlers").toString());
      section.put("agents", URLCreator.create(getRequestHost(), requestUri, "agents").toString());
      section.put("importing", URLCreator.create(getRequestHost(), requestUri, "importing").toString());
    } catch (final Exception e) {
      section.put("error", e.getClass().getCanonicalName() + ": " + e.getMessage());
    }
    return section;
  }

  /** convert exception to AnyMap. */
  private AnyMap buildErrorSection(final Exception e) {
    final AnyMap errorSection = FACTORY.createAnyMap();
    errorSection.put("error", AnyUtil.exceptionToAny(e));
    return errorSection;
  }

  /**
   * {@inheritDoc}
   * 
   * GET is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * called by DS: set taskmanager to use before activation.
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * called by DS: remove taskmanager after deactivation.
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (taskManager == _taskManager) {
      _taskManager = null;
    }
  }

}
