/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.httphandler;

import java.io.IOException;
import java.util.List;

import org.codehaus.jackson.JsonParseException;
import org.eclipse.jetty.http.HttpException;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.objectstore.BadRequestException;
import org.eclipse.smila.objectstore.NoSuchObjectException;
import org.eclipse.smila.objectstore.NoSuchStoreException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.ServiceUnavailableException;

/**
 * base class for request handlers concerning objectstore access.
 * 
 * @author jschumacher
 * 
 */
public abstract class AStoreHandler extends JsonRequestHandler {

  /**
   * ObjectStore service.
   */
  private ObjectStoreService _objectStoreService;

  /**
   * @return object store service reference
   */
  public ObjectStoreService getObjectStoreService() {
    return _objectStoreService;
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param objectStoreService
   *          ObjectStoreService reference.
   */
  public void setObjectStoreService(final ObjectStoreService objectStoreService) {
    _objectStoreService = objectStoreService;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param objectStoreService
   *          ObjectStoreService reference.
   */
  public void unsetObjectStoreService(final ObjectStoreService objectStoreService) {
    if (_objectStoreService == objectStoreService) {
      _objectStoreService = null;
    }
  }

  /**
   * Extracts the store name from the requestURI.
   * 
   * @param requestUri
   *          the request URI
   * @return the store name
   */
  protected String getStoreName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern " + getUriPattern());
  }

  /**
   * Extracts the object id from the requestURI.
   * 
   * @param requestUri
   *          the request URI
   * @return the object id
   */
  protected String getObjectId(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 1) {
      final String objectId = uriParts.get(1);
      final int lastPos = objectId.length() - 1;
      if (objectId.charAt(lastPos) == '/') {
        return objectId.substring(0, lastPos);
      }
      return objectId;
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern " + getUriPattern());
  }

  /**
   * Select appropriate HTTP status codes for {@link ObjectStoreException} and subclasses.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof NoSuchStoreException || ex instanceof NoSuchObjectException) {
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof BadRequestException || ex instanceof IllegalArgumentException
      || ex instanceof JsonParseException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex instanceof ServiceUnavailableException) {
      return HttpStatus.SERVICE_UNAVAILABLE;
    }
    if (ex instanceof IOException && !(ex instanceof HttpException)) {
      return HttpStatus.SERVICE_UNAVAILABLE;
    }
    if (ex instanceof ObjectStoreException) {
      return HttpStatus.INTERNAL_SERVER_ERROR;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }
}
