/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.state.objectstore;

import java.util.Properties;

/**
 * configuration for ObjectStoreStateService. Main purpose is creation of configurable entry key so that it's possible
 * to optimize the StateService for varying numbers of entries and to adapt it to underlying objectstore
 * implementations.
 */
public class StateStoreConfiguration {

  /** property name to configure the length of the "shard" part of the entry key. */
  public static final String KEY_SHARDLENGTH = "shard.length";

  /** property name to configure the length of the segments in the remaining part of the entry key. */
  public static final String KEY_SEGMENTLENGTH = "segment.length";

  /** property name to configure the count of the segments in the remaining part of the entry key. */
  public static final String KEY_SEGMENTCOUNT = "segment.count";

  /**
   * property name to the {@link String#format(String, Object...) pattern that combines shard part and segmented rest of
   * digest into an entry key.
   */
  public static final String KEY_KEYPATTERN = "key.pattern";

  /** default length of shard part of entry key. */
  private static final String DEFAULT_SHARDLENGTH = "2";

  /** default length of segments in rest of entry key. */
  private static final String DEFAULT_SEGMENTLENGTH = "2";

  /** default number of segments in rest of entry key. */
  private static final String DEFAULT_SEGMENTCOUNT = "1";

  /** default key pattern: concatenate shard and segmented rest, seperated by '/'. */
  private static final String DEFAULT_KEYPATTERN = "%s/%s";

  /** actual shard length. */
  private final int _shardLength;

  /** actual segment length. */
  private final int _segmentLength;

  /** actual segment number. */
  private final int _segmentCount;

  /** actual key pattern. */
  private final String _keyPattern;

  /** create configuration from properties. */
  public StateStoreConfiguration(final Properties properties) {
    _shardLength = Integer.parseInt(properties.getProperty(KEY_SHARDLENGTH, DEFAULT_SHARDLENGTH));
    _segmentLength = Integer.parseInt(properties.getProperty(KEY_SEGMENTLENGTH, DEFAULT_SEGMENTLENGTH));
    _segmentCount = Integer.parseInt(properties.getProperty(KEY_SEGMENTCOUNT, DEFAULT_SEGMENTCOUNT));
    _keyPattern = properties.getProperty(KEY_KEYPATTERN, DEFAULT_KEYPATTERN);
  }

  /** compute entry key from id digest according to configuration properties. */
  public String getEntryKey(final String idDigest) {
    final String shard = idDigest.substring(0, _shardLength);
    final String segmentedDigest = getSegmentedRestOfDigest(idDigest);
    return String.format(_keyPattern, shard, segmentedDigest);
  }

  /** add '/' characters to rest of digest after shard part according to segmentation parameters. */
  private String getSegmentedRestOfDigest(final String digest) {
    final StringBuilder segmentedDigest = new StringBuilder();
    int startPos = _shardLength;
    if (_segmentLength > 0) {
      for (int i = 0; i < _segmentCount; i++) {
        segmentedDigest.append(digest, startPos, startPos + _segmentLength);
        segmentedDigest.append('/');
        startPos += _segmentLength;
      }
    }
    segmentedDigest.append(digest, startPos, digest.length());
    return segmentedDigest.toString();
  }
}
