/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.workermanager;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.smila.clusterconfig.ClusterConfigService;

/**
 * Helper class for handling (java) worker scale up.
 */
public class ScaleUpControl {

  /** key: worker name, value: current number of processed tasks for this worker. */
  private final Map<String, Integer> _taskCounter = new HashMap<String, Integer>();

  /** key: worker name, value: max scale up limit for this worker. */
  private final Map<String, Integer> _scaleUpLimit = new HashMap<String, Integer>();

  /**
   * instance with default scale up limits.
   */
  public ScaleUpControl() {
  }

  /**
   * @param clusterService
   *          holds scale up configuration for each worker
   */
  public ScaleUpControl(final ClusterConfigService clusterService) {
    for (final String worker : clusterService.getWorkersWithScaleUp()) {
      final Long workerScaleUp = clusterService.getWorkerScaleUp(worker);
      setScaleUpLimit(worker, workerScaleUp);
    }
  }

  /**
   * @return 'true' if this worker is allowed to get another task, 'false' otherwise.
   */
  public boolean canGetTask(final String workerName, final boolean isRunAlways) {
    final int current = getTaskCounter(workerName);
    if (isRunAlways) {
      return current < 1;
    } else {
      final int max = getScaleUpLimit(workerName);
      if (max > 0) {
        return current < max;
      } else {
        return max < 0; // < 0 -> no scale up limit
      }
    }
  }

  /**
   * @param worker
   *          the worker for which to increment the task counter.
   */
  public void incTaskCounter(final String worker) {
    Integer taskCount = _taskCounter.get(worker);
    if (taskCount == null) {
      _taskCounter.put(worker, Integer.valueOf(1));
    } else {
      _taskCounter.put(worker, Integer.valueOf(++taskCount));
    }
  }

  /**
   * @param worker
   *          the worker for which to decrement the task counter.
   */
  public void decTaskCounter(final String worker) {
    Integer taskCount = _taskCounter.get(worker);
    if (taskCount != null) {
      _taskCounter.put(worker, Integer.valueOf(--taskCount));
    }
  }

  /**
   * @param worker
   *          the worker for which to get the counter.
   * @return a counter of currently processed tasks
   */
  public int getTaskCounter(final String worker) {
    Integer current = _taskCounter.get(worker);
    if (current == null) {
      current = 0;
      _taskCounter.put(worker, current);
    }
    return current;
  }

  /**
   * @param worker
   *          the worker for which to get the limit.
   * @return scaleUp limit for this worker. defaults to 1 if no value is set.
   */
  public int getScaleUpLimit(final String worker) {
    Integer limit = _scaleUpLimit.get(worker);
    if (limit == null) {
      limit = 1;
      _scaleUpLimit.put(worker, limit);
    }
    return limit;
  }

  /**
   * @param worker
   *          a worker
   * @param scaleUp
   *          max scale up limit for the worker.
   */
  public void setScaleUpLimit(final String worker, final Long scaleUp) {
    _scaleUpLimit.put(worker, scaleUp.intValue());
  }

}
