/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.workermanager.internal;

import java.util.Collection;
import java.util.Map;
import java.util.Queue;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Mode;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.taskworker.DefaultTaskLogFactory;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.TaskLog;
import org.eclipse.smila.taskworker.TaskLogFactory;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.taskworker.internal.TaskContextImpl;
import org.eclipse.smila.taskworker.util.Counters;
import org.eclipse.smila.workermanager.ScaleUpControl;
import org.eclipse.smila.workermanager.TaskKeepAliveListener;
import org.eclipse.smila.workermanager.WorkerManager;
import org.eclipse.smila.workermanager.keepalive.TaskKeepAlive;
import org.osgi.service.component.ComponentContext;

/**
 * WorkerManager implementation. Workers are registered via addWorker() method.
 */
public class WorkerManagerImpl implements WorkerManager, TaskKeepAliveListener {

  /** the initial delay in ms after which to start getting and processing tasks. */
  private static final long TASK_LOOP_INITIAL_DELAY = 50;

  /** the periodic delay in ms after which to start the next loop of getting and finishing tasks. */
  private static final long TASK_LOOP_PERIODIC_DELAY = 5;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** service reference to definiton persistence. */
  private DefinitionPersistence _defPersistence;

  /** service reference to taskmanager. */
  private TaskManager _taskManager;

  /** service reference to object store. */
  private ObjectStoreService _objectStore;

  /** holds the cluster configuration is needed to get the scale up configuration of the workers. */
  private ClusterConfigService _clusterConfigService;

  /** localhost. */
  private String _localhost;

  /** list of worker names. */
  private final Queue<String> _workerNames = new ConcurrentLinkedQueue<String>();

  /** key: worker name, value: Worker. */
  private final Map<String, Worker> _workers = new ConcurrentHashMap<String, Worker>();

  /** key: worker name, value: Worker definition. */
  private final Map<String, WorkerDefinition> _workerDefinitions =
    new ConcurrentHashMap<String, WorkerDefinition>();

  /** aggregated counters of internal workers. */
  private final Map<String, Map<String, Number>> _internalWorkerCounters =
    new ConcurrentHashMap<String, Map<String, Number>>();

  /** Helper class for handling worker scale up and holding counters. */
  private ScaleUpControl _scaleUpControl;

  /** thread pool that holds all active TaskWorkers. */
  private WorkerPool _workerPool;

  /** TaskLog provider. */
  private TaskLogFactory _taskLogFactory = new DefaultTaskLogFactory(); // may be overwritten by provided service

  /** keep alive mechanism. */
  private TaskKeepAlive _keepAliveRunner;

  /** key: taskId, value: TaskContext of currently processed task. */
  private final Map<String, TaskContext> _currentlyProcessedTasks = new ConcurrentHashMap<String, TaskContext>();

  /** Periodically repeats the keep alive for currently processed tasks. */
  private ScheduledExecutorService _keepAliveExecutor = Executors.newScheduledThreadPool(1);

  /** Periodically repeats the getting and finishing of tasks. */
  private final ScheduledExecutorService _taskLoopExecutor = Executors.newScheduledThreadPool(1);

  /** Runnable to get and to finish tasks. */
  private final Runnable _taskLoopRunner = new Runnable() {
    @Override
    public void run() {
      final boolean haveChanges = startAndFinishTasks();
      // don't flood taskmanager with requests if there is nothing to do or we don't have workers available.
      if (!haveChanges) {
        try {
          Thread.sleep(TASK_LOOP_INITIAL_DELAY);
        } catch (final Exception ex) {
          ; // ignore
        }
      }
    }
  };

  /** do one turn in the "get-task-finish-task" loop. */
  public boolean startAndFinishTasks() {
    boolean haveChanges = false;
    if (!_workerNames.isEmpty()) {
      haveChanges |= getAndStartTasks();
      // change priority of workers, otherwise workers could starve,
      // because the scaleUp limit is utilized by other workers.
      _workerNames.add(_workerNames.poll());
      haveChanges |= finishCompletedTasks();
    }
    return haveChanges;
  }

  /** get and start tasks as long as scaleup limits are not utilized. */
  public boolean getAndStartTasks() {
    boolean newTaskStarted = false;
    for (final String workerName : _workerNames) {
      try {
        newTaskStarted |= getAndStartTask(workerName);
      } catch (final Exception ex) {
        _log.warn("Unexpected exception when trying to get a task for worker " + workerName, ex);
      }
    }
    return newTaskStarted;
  }

  /** finish completed tasks. */
  public boolean finishCompletedTasks() {
    boolean finishedTask = false;
    for (final WorkerRunner taskWorker : _workerPool.getCompleted()) {
      try {
        finishTask(taskWorker);
        finishedTask = true;
      } catch (final Exception ex) {
        _log.warn("Unexpected exception when trying to finish task " + taskWorker.getTask(), ex);
      }
    }
    return finishedTask;
  }

  /**
   * get and submit task for the given worker if that doesn't exceed worker's max scale up.
   * 
   * @param worker
   *          the worker for which to get a task
   * @return true if a new task was started
   */
  public boolean getAndStartTask(final String worker) {
    final WorkerDefinition workerDef = getWorkerDefinition(worker);
    if (workerDef == null) {
      // can happen when shutting down...
      if (_workerNames.contains(worker)) {
        _log.error("Cannot find worker definition for worker '" + worker
          + "', please check jobmanager configuration files.");
        _log.error("Removing worker '" + worker + "' from operation. "
          + "Please amend the worker configuration and restart the system to make this worker operational.");
        removeWorker(_workers.get(worker));
      }
      return false;
    }
    final boolean isRunAlways = workerDef.getModes().contains(Mode.RUNALWAYS);
    if (_scaleUpControl.canGetTask(worker, isRunAlways)) {
      try {
        final String workerHost = isRunAlways ? null : _localhost;
        final Task task = _taskManager.getTask(worker, workerHost);
        if (task != null) {
          final TaskLog taskLog = _taskLogFactory.getTaskLog(task);
          final TaskContext taskContext =
            new TaskContextImpl(task, taskLog, _objectStore, workerDef.getOutputModes());
          final WorkerRunner taskWorker = new WorkerRunner(_workers.get(worker), taskContext);
          _currentlyProcessedTasks.put(task.getTaskId(), taskContext);
          _keepAliveRunner.addTask(task);
          _workerPool.submit(taskWorker);
          _scaleUpControl.incTaskCounter(worker);
          return true;
        }
      } catch (final TaskmanagerException e) {
        _log.info("Error getting task for worker '" + worker + "', will retry later.", e);
      }
    }
    return false;
  }

  /**
   * @param worker
   *          the worker to return the worker definition for
   * @return worker definition for given worker
   */
  public WorkerDefinition getWorkerDefinition(final String worker) {
    WorkerDefinition workerDef = _workerDefinitions.get(worker);
    if (workerDef == null) {
      workerDef = _defPersistence.getWorker(worker);
    }
    if (workerDef == null && isInternalWorker(worker)) {
      workerDef = createWorkerDefinitionForInternalWorker(worker);
    }
    if (workerDef != null) {
      _workerDefinitions.put(worker, workerDef);
      try {
        // this prevents some error messages about missing workers because they were not yet initialized by the
        // jobmanager.
        _taskManager.addTaskQueue(worker);
      } catch (final TaskmanagerException ex) {
        _log.info("Failed to initialize task queue for worker '" + worker
          + "'. This is probably a temporary problem only.", ex);
      }
    }
    return workerDef;
  }

  /**
   * @param taskWorker
   *          the TaskWorker containing the (completed) Task to finish.
   */
  public void finishTask(final WorkerRunner taskWorker) {
    final String workerName = taskWorker.getWorkerName();
    try {
      _keepAliveRunner.removeTask(taskWorker.getTask());
      _currentlyProcessedTasks.remove(taskWorker.getTask().getTaskId());
      if (isInternalWorker(workerName)) {
        addInternalWorkerCounters(workerName, taskWorker.getResult().getCounters());
      }
      if (!taskWorker.getTaskContext().isCanceled()) {
        _taskManager.finishTask(workerName, taskWorker.getTask().getTaskId(), taskWorker.getResult());
      }
    } catch (final TaskmanagerException e) {
      _log.error("TaskManager exception while finishing task", e);
    } finally {
      _scaleUpControl.decTaskCounter(workerName);
    }
  }

  /**
   * OSGi Declarative Services service activation method.
   */
  protected void activate(final ComponentContext context) {

    setScaleUpControl(new ScaleUpControl(_clusterConfigService));

    _workerPool = new WorkerPool();

    // startup the task keep alive mechanism.
    _keepAliveRunner = new TaskKeepAlive(_taskManager);
    _keepAliveRunner.addKeepAliveFailureListener(this);
    _keepAliveExecutor.scheduleWithFixedDelay(_keepAliveRunner, TaskKeepAlive.DEFAULT_SCHEDULE_MILLIS,
      TaskKeepAlive.DEFAULT_SCHEDULE_MILLIS, TimeUnit.MILLISECONDS);

    // create and start periodic task loop (get/finish)
    _taskLoopExecutor.scheduleWithFixedDelay(_taskLoopRunner, TASK_LOOP_INITIAL_DELAY, TASK_LOOP_PERIODIC_DELAY,
      TimeUnit.MILLISECONDS);
  }

  /**
   * OSGi Declarative Services service deactivation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void deactivate(final ComponentContext context) {
    if (_taskLoopExecutor != null) {
      _taskLoopExecutor.shutdown();
    }
    if (_keepAliveExecutor != null) {
      _keepAliveExecutor.shutdown();
    }
    if (_workerPool != null) {
      _workerPool.close();
    }
  }

  /**
   * Add the given worker.
   */
  public void addWorker(final Worker worker) {
    _workerNames.add(worker.getName());
    _workers.put(worker.getName(), worker);
    if (_log.isInfoEnabled()) {
      _log.info("Added worker " + worker.getName() + " to WorkerManager.");
    }
  }

  /**
   * Removes the given worker.
   */
  public void removeWorker(final Worker worker) {
    _workerNames.remove(worker.getName());
    _workers.remove(worker.getName());
    _workerDefinitions.remove(worker.getName());
    if (_log.isInfoEnabled()) {
      _log.info("Removed worker " + worker.getName() + " from WorkerManager.");
    }
  }

  @Override
  public boolean containsWorker(final String worker) {
    return _workerNames.contains(worker);
  }

  @Override
  public void addKeepAliveListener(final TaskKeepAliveListener listener) {
    _keepAliveRunner.addKeepAliveFailureListener(listener);
  }

  @Override
  public void removeKeepAliveListener(final TaskKeepAliveListener listener) {
    _keepAliveRunner.removeKeepAliveFailureListener(listener);
  }

  @Override
  public void addKeepAliveTask(final Task task) {
    _keepAliveRunner.addTask(task);
  }

  @Override
  public void removeKeepAliveTask(final Task task) {
    _keepAliveRunner.removeTask(task);
  }

  /** Called by keep alive mechanism when a task was cancelled. */
  @Override
  public void removedTask(final Task task) {
    final TaskContext taskContext = _currentlyProcessedTasks.get(task.getTaskId());
    if (taskContext != null) {
      if (_log.isInfoEnabled()) {
        _log.info("Marked TaskContext of task '" + task.getTaskId() + "' as canceled.");
      }
      taskContext.cancel();
    }
  }

  @Override
  public void setKeepAliveInterval(final long keepAliveCheckMillis, final long keepAliveSendSeconds) {
    final Collection<TaskKeepAliveListener> listeners = _keepAliveRunner.getKeepAliveFailureListeners();
    _keepAliveExecutor.shutdown();
    _keepAliveRunner = new TaskKeepAlive(_taskManager, keepAliveSendSeconds);
    _keepAliveRunner.addKeepAliveFailureListeners(listeners);
    _keepAliveExecutor = Executors.newScheduledThreadPool(1);
    _keepAliveExecutor.scheduleAtFixedRate(_keepAliveRunner, keepAliveCheckMillis, keepAliveCheckMillis,
      TimeUnit.MILLISECONDS);
  }

  /**
   * @param scaleUpControl
   *          defines scale up limit for workers
   */
  @Override
  public void setScaleUpControl(final ScaleUpControl scaleUpControl) {
    _scaleUpControl = scaleUpControl;
  }

  @Override
  public AnyMap getInfo() {
    final DataFactory dataFactory = DataFactory.DEFAULT;
    final AnyMap info = dataFactory.createAnyMap();
    info.put("host", _localhost);
    info.put("currentlyProcessedTasks", _currentlyProcessedTasks.size());

    final AnySeq workersSection = dataFactory.createAnySeq();
    for (final WorkerDefinition workerDef : _workerDefinitions.values()) {
      final AnyMap worker = dataFactory.createAnyMap();
      final String workerName = workerDef.getName();
      worker.put("name", workerName);
      worker.put("runAlways", workerDef.getModes().contains(Mode.RUNALWAYS));
      worker.put("scaleUpLimit", _scaleUpControl.getScaleUpLimit(workerName));
      worker.put("scaleUpCurrent", _scaleUpControl.getTaskCounter(workerName));
      final Map<String, Number> counters = _internalWorkerCounters.get(workerName);
      if (counters != null) {
        final AnyMap counterInfo = dataFactory.createAnyMap();
        for (final String key : new TreeSet<String>(counters.keySet())) {
          counterInfo.put(key, counters.get(key));
        }
        worker.put("counter", counterInfo);
      }
      workersSection.add(worker);
    }
    info.put("workers", workersSection);

    final AnyMap workerPoolSection = dataFactory.createAnyMap();
    final ExecutorService threadPool = _workerPool.getThreadPool();
    if (threadPool instanceof ThreadPoolExecutor) {
      final ThreadPoolExecutor tpe = (ThreadPoolExecutor) threadPool;
      workerPoolSection.put("activeCount", tpe.getActiveCount());
      workerPoolSection.put("corePoolSize", tpe.getCorePoolSize());
      workerPoolSection.put("poolSize", tpe.getPoolSize());
      workerPoolSection.put("largestPoolSize", tpe.getLargestPoolSize());
      workerPoolSection.put("queueSize", tpe.getQueue().size());
      workerPoolSection.put("taskCount", tpe.getTaskCount());
    }
    info.put("workerPool", workerPoolSection);

    return info;
  }

  /**
   * needed to get cluster configuration.
   */
  public void setClusterConfigService(final ClusterConfigService ccs) {
    _clusterConfigService = ccs;
    _localhost = ccs.getLocalHost();
  }

  /**
   * remove an {@link ClusterConfigService}. To be called by DS runtime after deactivation.
   */
  public void unsetClusterConfigService(final ClusterConfigService ccs) {
    if (_clusterConfigService == ccs) {
      _clusterConfigService = null;
    }
  }

  /** set OSGI service. */
  public void setDefinitionPersistence(final DefinitionPersistence defPersistence) {
    _defPersistence = defPersistence;
  }

  /** unset OSGI service. */
  public void unsetDefinitionPersistence(final DefinitionPersistence defPersistence) {
    if (_defPersistence == defPersistence) {
      _defPersistence = null;
    }
  }

  /**
   * method for DS to set a service reference.
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * method for DS to unset a service reference.
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }

  /**
   * method for DS to set a service reference.
   */
  public void setTaskLogFactory(final TaskLogFactory taskLogFactory) {
    _taskLogFactory = taskLogFactory;
  }

  /**
   * method for DS to unset a service reference.
   */
  public void unsetTaskLogFactory(final TaskLogFactory taskLogFactory) {
    if (_taskLogFactory == taskLogFactory) {
      _taskLogFactory = new DefaultTaskLogFactory();
    }
  }

  /**
   * method for DS to set a service reference.
   */
  public void setObjectStoreService(final ObjectStoreService objectStore) {
    _objectStore = objectStore;
  }

  /**
   * method for DS to unset a service reference.
   */
  public void unsetObjectStoreService(final ObjectStoreService objectStore) {
    if (_objectStore == objectStore) {
      _objectStore = null;
    }
  }

  /**
   * @param worker
   * @return 'true' if worker is an internal worker, 'false' otherwise.
   */
  private boolean isInternalWorker(final String worker) {
    return worker.startsWith(WorkerManager.PREFIX_INTERNAL);
  }

  /**
   * @param worker
   * @return on-the-fly worker definition for given (internal) worker
   */
  private WorkerDefinition createWorkerDefinitionForInternalWorker(final String worker) {
    final AnyMap workerDefAny = DataFactory.DEFAULT.createAnyMap();
    final AnySeq modesAny = DataFactory.DEFAULT.createAnySeq();
    modesAny.add(WorkerDefinition.Mode.RUNALWAYS.name());
    workerDefAny.put(WorkerDefinition.KEY_NAME, worker);
    workerDefAny.put(WorkerDefinition.KEY_MODES, modesAny);
    try {
      return new WorkerDefinition(workerDefAny);
    } catch (final InvalidDefinitionException e) {
      _log.error("Error while creating internal worker definition for worker " + worker);
      return null; // normally, this can't happen
    }
  }

  /** store counters of internal workers, as they are not retrievable via job run data. */
  private void addInternalWorkerCounters(final String workerName, final Map<String, Number> counters) {
    if (counters != null && !counters.isEmpty()) {
      final Map<String, Number> currentCounters = _internalWorkerCounters.get(workerName);
      if (currentCounters == null) {
        _internalWorkerCounters.put(workerName, counters);
      } else {
        Counters.addAll(currentCounters, counters);
      }
    }
  }

}
