/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.core.CoreContainer;
import org.eclipse.smila.solr.admin.SolrAdmin;
import org.eclipse.smila.solr.admin.SolrEmbeddedAdmin;
import org.eclipse.smila.solr.admin.SolrHttpAdmin;
import org.eclipse.smila.solr.server.SolrEmbeddedServers;
import org.eclipse.smila.solr.server.SolrHttpServers;
import org.eclipse.smila.solr.server.SolrServers;
import org.eclipse.smila.solr.util.SolrHelper;
import org.eclipse.smila.solr.util.SolrProperties;

/**
 * The SolrManager class.
 * 
 * @author pwissel
 * 
 */
public class SolrManager {

  /**
   * The SolrProperties.
   */
  private final SolrProperties _properties = new SolrProperties();

  /**
   * The log.
   */
  private final Log _log = LogFactory.getLog(SolrManager.class);

  /**
   * The core container (only for embedded use).
   */
  private CoreContainer _container;

  /**
   * creator and holder for one {@link SolrServers} per core.
   */
  private SolrServers _servers;

  /**
   * The SolrAdmin.
   */
  private SolrAdmin _admin;

  /**
   * The SolrHelper.
   */
  private SolrHelper _helper;

  /**
   * 
   */
  public SolrManager() {
    this(new SolrProperties());
  }

  /**
   * Default constructor.
   */
  public SolrManager(SolrProperties properties) {
    try {
      if (_log.isDebugEnabled()) {
        _log.debug("Start SolrManager instance...");
      }

      _helper = new SolrHelper(properties);

      if (_properties.isEmbedded()) {
        loadCoreContainer();
        _servers = new SolrEmbeddedServers(_container);
        _admin = new SolrEmbeddedAdmin(_helper, _servers, _container);
      } else {
        _servers = new SolrHttpServers(_properties);
        _admin = new SolrHttpAdmin(_helper, _servers);
      }

    } catch (Exception exception) {
      _log.error("Error while initialize SolrManager", exception);
    }
  }

  /**
   * Load CoreContainer configuration file form solr.home (solr.xml).
   */
  private void loadCoreContainer() throws Exception {
    final File solr = _helper.getSolrXML();
    _container = new CoreContainer();
    _container.load(_helper.getSolrHome().getAbsolutePath(), solr);

    if (_log.isInfoEnabled()) {
      _log.info("Solr CoreContainer for EmbeddedSolrServer loaded successfully.");
    }

    if (_log.isDebugEnabled()) {
      _log.debug("Available solr cores: " + StringUtils.join(_container.getCoreNames(), File.pathSeparator));
    }
  }

  /**
   * Get the SolrProperties.
   * 
   * @return the SolrProperties.
   */
  public SolrProperties getSolrProperties() {
    return _properties;
  }

  /**
   * Get the SolrAdmin.
   * 
   * @return the SolrAdmin.
   */
  public SolrAdmin getSolrAdmin() {
    return _admin;
  }

  /**
   * Get the SolrHelper.
   * 
   * @return the SolrHelper.
   */
  public SolrHelper getSolrHelper() {
    return _helper;
  }

  /**
   * Get CoreContainer.
   * 
   * @return the core container.
   */
  public CoreContainer getCoreContainer() {
    return _container;
  }

  /**
   * Get core names.
   * 
   * @return the core names.
   */
  public Collection<String> getCoreNames() {
    try {
      return _admin.getCoreNames();
    } catch (Exception ex) {
      _log.error("Error getting core names from Solr", ex);
      return Collections.emptyList();
    }
  }

  /**
   * Get a SolrServer by core name.
   * 
   * @param coreName
   *          the core name.
   * @return the SolrServer.
   * @throws SolrServerException
   *           SolrServerException if no core by this name is alive
   * @throws IOException
   *           IOException.
   */
  public SolrServer getSolrServer(String coreName) throws SolrServerException, IOException {
    return _servers.getSolrServer(coreName);
  }

}
