/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager;

import java.util.Collection;
import java.util.Map;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;

/**
 * The TaskManager.
 */
public interface TaskManager {

  /** prefix for internal (worker) queues. */
  String PREFIX_INTERNAL = "_";

  /** The configuration bundle. */
  String CONFIGURATION_BUNDLE = "org.eclipse.smila.taskmanager";

  /** error code used in result descriptions to finish tasks. */
  String TASKERROR_TIME_TO_LIVE = "TimeToLive";

  /** queue/worker containing tasks to finish. */
  String FINISHING_TASKS_WORKER = "_finishingTasks";

  /**
   * @param task
   *          the task to add to the todo queue of its worker (contained in task)
   */
  void addTask(final Task task) throws TaskmanagerException;

  /**
   * @param task
   *          the task to add to the in progress queue of its worker (contained in task)
   * @throws TaskmanagerException
   */
  void addInProgressTask(final Task task) throws TaskmanagerException;

  /**
   * add tasks to the system (e.g. for manually triggered tasks).
   * 
   * @param taskList
   *          the list of tasks to add.
   */
  void addTasks(final Collection<Task> taskList) throws TaskmanagerException;

  /**
   * Get next task for worker with name workerName.
   * 
   * @param workerName
   *          the worker name
   * @param host
   *          the host name where the worker is running that requested the task
   * @return next task for worker with name workerName
   * @throws TaskmanagerException
   *           an error while looking up or returning the task for the given worker.
   */
  Task getTask(String workerName, String host) throws TaskmanagerException;

  /**
   * Get next task for worker with name workerName that match the given qualifiers.
   * 
   * @param workerName
   *          the worker name
   * @param host
   *          the host name where the worker is running that requested the task
   * @param qualifiers
   *          qualifiers to describe the task.
   * @return Next task for worker with name workerName that match the given qualifiers.
   * @throws TaskmanagerException
   *           an error while looking up or returning the task for the given worker.
   */
  Task getTask(String workerName, String host, Collection<String> qualifiers) throws TaskmanagerException;

  /**
   * Remove task from in-progress and put it into finishing queue.
   * 
   * @param workerName
   *          the worker name
   * @param taskId
   *          the id of the task to be finished
   * @param resultDescription
   *          the result description for the task to be finished.
   * @throws TaskmanagerException
   *           an error occurred while trying to finish the task.
   */
  void finishTask(String workerName, String taskId, ResultDescription resultDescription)
    throws TaskmanagerException;

  /**
   * Finish all qualified tasks matching the qualifier condition. Lock qualifiers 
   * 
   * @param workerName
   *          the worker name.
   * @param qualifiers
   *          qualifiers describing the condition. Must not be null. If it is empty, nothing is done.
   * @param resultDescription
   *          the result description for the tasks to be finished.
   * @throws TaskmanagerException
   *           an exception occurred while trying to delete the tasks.
   */
  void finishTasks(String workerName, Collection<String> qualifiers, ResultDescription resultDescription)
    throws TaskmanagerException;

  /**
   * Prevent rollback of task due to exceeded time-to-live.
   * 
   * @param workerName
   *          the worker name.
   * @param taskId
   *          the ID of the task.
   * @throws TaskmanagerException
   *           an error occurred while trying to send keep alive for this task.
   */
  void keepAlive(String workerName, String taskId) throws TaskmanagerException;

  /**
   * Get number of tasks in all current queues.
   * 
   * @return task counters
   * @throws TaskmanagerException
   *           counting fails
   */
  Map<String, TaskCounter> getTaskCounters() throws TaskmanagerException;

  /**
   * Get task list for current task pipe and sub-type.
   * 
   * @param workerName
   *          name/id as string
   * @param section
   *          task pipe sub-type: "inprogress" or "todo".
   * @param maxCount
   *          max. number of tasks in return list
   * @return task list
   * @throws TaskmanagerException
   *           if accessing task list fails
   */
  TaskList getTaskList(final String workerName, final String section, final int maxCount)
    throws TaskmanagerException;

  /**
   * Prepares information on a task stored in the task storage.
   * 
   * @param workerName
   *          pipe in which the task is stored
   * @param section
   *          queue section ("todo" or "inprogress")
   * @param taskName
   *          task identifier
   * @return task information
   * 
   * @throws TaskmanagerException
   *           on error
   */
  Any getTaskInfo(final String workerName, final String section, final String taskName) throws TaskmanagerException;

  /**
   * @return number of nodes that may fail before tasks get lost.
   */
  long getFailSafetyLevel();

  /**
   * @return number of tasks in progress on different hosts.
   * @throws TaskmanagerException
   *           on error.
   */
  Map<String, Integer> getScaleUpCounters() throws TaskmanagerException;

  /**
   * @return maximum number of tasks that should be delivered to a host. (scale up control)
   */
  long getMaxScaleUp();

  /**
   * Remove canceled tasks identified by the filter map.
   * 
   * @param filterMap
   *          map to identify tasks to be removed
   * @throws TaskmanagerException
   *           an exception if something went wrong
   */
  void removeTasks(final AnyMap filterMap) throws TaskmanagerException;

  /**
   * @param workerName
   *          worker for which to add a task queue with given name.
   * @throws TaskmanagerException
   */
  void addTaskQueue(final String workerName) throws TaskmanagerException;

}
