/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.httphandler;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException;
import org.eclipse.smila.taskmanager.TaskList;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;

/**
 * Implements the handling of HTTP requests for the task state API.
 * 
 * URL pattern: /smila/tasks/--workername--/inprogress/ or /smila/tasks/--workername--/todo/
 */
public class TaskTypeStateHandler extends JsonRequestHandler {

  /** The reference to the TaskManager service. */
  private TaskManager _taskManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if ("GET".equals(method)) {
      final String workerName = getWorkerName(requestUri);
      final String sectionName = getSection(requestUri);
      return buildTaskList(requestUri, workerName, sectionName);
    }
    return null;
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalArgumentException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex.getCause() != null && ex.getCause() instanceof BadParameterTaskmanagerException) {
      return HttpStatus.NOT_FOUND;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * build task list for given task pipe and sub-type.
   * 
   * @param requestUri
   *          request url, e.g. "http://localhost:8050/smila/tasks/TMEAnalyseDoc/todo/"
   * @param taskPipe
   *          name/id as string
   * @param taskQueueSubType
   *          task pipe sub-type: "inprogress" or "todo".
   * @throws TaskmanagerException
   *           for task manager errors including parameter errors.
   * @throws MalformedURLException
   *           if url created for task is invalid (internal error).
   * @return Any, representing task list
   */
  private Map<String, Object> buildTaskList(final String requestUri, final String taskPipe,
    final String taskQueueSubType) throws TaskmanagerException, MalformedURLException {

    final Map<String, Object> result = new LinkedHashMap<String, Object>();
    // final String baseUri = getStaticUriParts(requestUri);

    final TaskList taskList = _taskManager.getTaskList(taskPipe, taskQueueSubType, 100);
    result.put("count", Integer.valueOf(taskList.getSize()));
    final List<String> taskIDs = new ArrayList<String>();
    for (final String taskName : taskList.getTaskNames()) {
      // we'll get something like /smila/task/...., but must remove the root path to match it to our root context path.
      final String taskPath = taskName.replaceFirst("/[^/]+/", "");
      final URL taskURL = URLCreator.create(getRequestHost(), getRootContextPath(), taskPath);
      taskIDs.add(taskURL.toString());
    }
    result.put("tasks", taskIDs);
    return result;
  }

  /**
   * Gets the worker name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return the worker name from the requestUri.
   */
  private String getWorkerName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * Gets the task section name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return the section name from the requestUri.
   */
  private String getSection(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(1);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * GET and DELETE are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return ("GET".equals(method));
  }

  /**
   * @param taskManager
   *          the reference to the JobManager service.
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * @param taskManager
   *          the reference to the current JobManager service.
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }
}
