/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file;

import java.io.File;
import java.io.IOException;
import java.util.Date;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.importing.ImportingConstants;

/**
 * Utility class to create records from File objects.
 * 
 * @author stuc07
 * 
 */
public final class FileToRecordConverter {

  public static final String PROPERTY_FILE_NAME = "file.name";

  public static final String PROPERTY_FILE_PATH = "file.path";

  public static final String PROPERTY_FILE_FOLDER = "file.folder";

  public static final String PROPERTY_FILE_SIZE = "file.size";

  public static final String PROPERTY_FILE_LAST_MODIFIED = "file.lastModified";

  public static final String PROPERTY_FILE_EXTENSION = "file.extension";

  public static final String ATTACHMENT_FILE_CONTENT = "file.content";

  private final DataFactory _factory;

  public FileToRecordConverter(final DataFactory factory) {
    _factory = factory;
  }

  /**
   * Create a Record for the given file and dataSource and fill it with all file properties. Optionally also fill with
   * content attachment.
   * 
   * @param file
   * @param dataSource
   * @param withContent
   *          if true fill also content attachment, otherwise not
   * @return a Record
   * @throws IOException
   */
  public Record fileToRecord(final File file, final String dataSource, final boolean withContent)
    throws IOException {
    if (file.isDirectory()) {
      throw new IllegalArgumentException("parameter file is a directory");
    }

    final Record record = createRecord(file, dataSource);
    addFileProperties(file, record);
    record.getMetadata().put(ImportingConstants.ATTRIBUTE_DELTA_HASH, Long.toString(file.lastModified()));
    if (withContent) {
      addAttachment(record);
    }
    return record;
  }

  /**
   * Create a Record for the given directory and dataSource and fill it with ImportingConstants.ROOT_FOLDER_ATTRIBUTE.
   * 
   * @param directory
   * @param dataSource
   * @return a record
   */
  public Record dirToRecord(final File directory, final String dataSource) {
    if (directory.isFile()) {
      throw new IllegalArgumentException("parameter directory is not a directory");
    }
    final Record record = createRecord(directory, dataSource);
    record.getMetadata().put(PROPERTY_FILE_FOLDER, directory.getAbsolutePath());
    return record;
  }

  /**
   * Add the content attachment to the given record. The file content is expected to be referenced by PROPERTY_FILE_PATH
   * in the record.
   * 
   * @param record
   * @throws IOException
   */
  public void addAttachment(final Record record) throws IOException {
    final String path = record.getMetadata().getStringValue(PROPERTY_FILE_PATH);
    if (path == null) {
      throw new IllegalArgumentException("Record '" + record.getId() + "' does not contain attribute '"
        + PROPERTY_FILE_PATH + "'");
    }
    record.setAttachment(ATTACHMENT_FILE_CONTENT, FileUtils.readFileToByteArray(new File(path)));
  }

  private Record createRecord(final File file, final String dataSource) {
    return _factory.createRecord(file.getAbsolutePath(), dataSource);
  }

  private void addFileProperties(final File file, final Record record) {
    record.getMetadata().put(PROPERTY_FILE_NAME, file.getName());
    record.getMetadata().put(PROPERTY_FILE_PATH, file.getAbsolutePath());
    record.getMetadata().put(PROPERTY_FILE_FOLDER, file.getParent());
    record.getMetadata().put(PROPERTY_FILE_SIZE, file.length());
    final Value lastModifiedDate = _factory.createDateTimeValue(new Date(file.lastModified()));
    record.getMetadata().put(PROPERTY_FILE_LAST_MODIFIED, lastModifiedDate);
    record.getMetadata().put(PROPERTY_FILE_EXTENSION, FilenameUtils.getExtension(file.getName()));
  }
}
