/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.ipc;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.StoredAttachment;
import org.eclipse.smila.ipc.IpcFactory;
import org.eclipse.smila.ipc.IpcStreamReader;
import org.eclipse.smila.ipc.IpcToken;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.ipc.json.JsonFactory;

/**
 * Utility class for reading Record objects from binary (BON) and JSON objects.
 * 
 * @author aweber
 */
public class IpcRecordReader {

  /** attachment names for record (de)serialization. */
  private static final String ATTACHMENT_NAMES = "_attachmentNames";

  /** Helper to read Record metadata. */
  private final IpcAnyReader _ipcAnyReader = new IpcAnyReader();

  /** BON IPC factory. */
  private final IpcFactory _binaryFactory = new BinaryFactory();

  /** JSON IPC factory. */
  private final IpcFactory _jsonFactory = new JsonFactory();

  /**
   * @param binary
   * @return Record converted from BON object.
   * @throws IOException
   */
  public Record readBinaryObject(final byte[] binary) throws IOException {
    ByteArrayInputStream bais = null;
    try {
      bais = new ByteArrayInputStream(binary);
      return readBinaryStream(bais);
    } finally {
      IOUtils.closeQuietly(bais);
    }
  }

  /**
   * @param stream
   * @return Record converted from input stream
   * @throws IOException
   */
  public Record readBinaryStream(final InputStream stream) throws IOException {
    final IpcStreamReader reader = _binaryFactory.newStreamReader(stream);
    return readStream(reader);
  }

  /**
   * @param json
   * @return Record converted from input JSON string.
   * @throws IOException
   */
  public Record readJsonObject(final String json) throws IOException {
    ByteArrayInputStream bais = null;
    try {
      bais = new ByteArrayInputStream(json.getBytes(IpcAnyReader.ENCODING));
      final Record record = readJsonStream(bais);
      return record;
    } finally {
      IOUtils.closeQuietly(bais);
    }
  }

  /**
   * @param stream
   * @return Record converted from input JSON stream.
   * @throws IOException
   */
  public Record readJsonStream(final InputStream stream) throws IOException {
    final IpcStreamReader reader = _jsonFactory.newStreamReader(stream);
    try {
      return readStream(reader);
    } finally {
      reader.closeWithoutStream(); // do not close underlying stream
    }
  }

  /**
   * @param reader
   * @return Record converted from input reader's underlying stream.
   * @throws IOException
   */
  public Record readStream(final IpcStreamReader reader) throws IOException {
    IpcToken token = reader.nextToken();
    if (token == null) {
      return null;
    }
    if (token != IpcToken.OBJECT_START) {
      throw new IllegalStateException("Expected OBJECT_START Token; Token = " + token);
    }
    token = reader.nextToken();
    if (token != IpcToken.MAPPING_START) {
      throw new IllegalStateException("Expected MAPPING_START Token; Token = " + token);
    }
    final AnyMap metadata = _ipcAnyReader.readMap(reader);
    final Record record = DataFactory.DEFAULT.createRecord();
    record.getMetadata().putAll(metadata);

    token = reader.nextToken();
    if (token == IpcToken.ATTACHMENTS_START) {
      // attachments may be set (with content) in an attachments section (BON)
      readAttachments(reader, record);
      if (metadata.containsKey(ATTACHMENT_NAMES)) {
        metadata.remove(ATTACHMENT_NAMES); // if attachment binaries are set, it's not allowed to set this attribute
      }
      token = reader.nextToken();
    } else if (metadata.containsKey(ATTACHMENT_NAMES)) {
      // ... or we may have the attachment names in a system attribute (JSON)
      for (final Any attName : metadata.getSeq(ATTACHMENT_NAMES)) {
        record.setAttachment(new StoredAttachment(attName.asValue().asString()));
      }
      record.getMetadata().remove(ATTACHMENT_NAMES);
    }

    if (token != IpcToken.OBJECT_END) {
      throw new IllegalStateException("Expected OBJECT_END Token; Token = " + token);
    }
    return record;
  }

  /**
   * helper method to read attachments.
   */
  private void readAttachments(final IpcStreamReader reader, final Record record) throws IOException {
    IpcToken token;
    while ((token = reader.nextToken()) != IpcToken.ATTACHMENTS_END) {
      if (token != IpcToken.SCALAR_STRING) { // expect attachment name
        throw new IllegalStateException("Expected SCALAR_STRING Token for attachment name; Token = " + token);
      }
      final String attachmentName = reader.currentStringValue();
      token = reader.nextToken();
      if (token != IpcToken.BINARY) { // expect attachment content
        throw new IllegalStateException("Expected BINARY Token; Token = " + token);
      }
      record.setAttachment(attachmentName, reader.currentBinaryValue());
    }
  }
}
