/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/
package org.eclipse.smila.importing;

import java.io.IOException;

import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.validation.InvalidRecordException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.taskworker.input.Inputs;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.Outputs;
import org.eclipse.smila.taskworker.output.RecordOutput;

/**
 * Worker that pushes input records to the bulkbuilder and marks the records as updated in delta indexing.
 */
public class UpdatePusherWorker implements Worker {

  /** worker's name. */
  private static final String WORKER_NAME = "updatePusher";

  /** input slot name. */
  private static final String INPUT_SLOT_NAME = "recordsToPush";

  /** (optional) output slot name. */
  private static final String OUTPUT_SLOT_NAME = "pushedRecords";

  /** the indexing job where the records should be pushed to. */
  private static final String JOB_TO_PUSH_TO_PARAM = "jobToPushTo";

  /** Reference to the BulkBuilder. */
  private BulkbuilderService _bulkbuilder;

  /** Reference to Deltaindexing. */
  private DeltaService _deltaIndexing;

  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    final String jobName = taskContext.getTaskParameters().getStringValue(JOB_TO_PUSH_TO_PARAM);
    if (jobName == null) {
      throw new IllegalArgumentException("Missing parameter '" + JOB_TO_PUSH_TO_PARAM + "' in task: "
        + taskContext.getTask());
    }
    final String jobRunId = taskContext.getTask().getProperties().get(Task.PROPERTY_JOB_RUN_ID);
    if (jobRunId == null) {
      throw new IllegalArgumentException("Missing parameter '" + Task.PROPERTY_JOB_RUN_ID + "' in task: "
        + taskContext.getTask());
    }
    final Inputs inputs = taskContext.getInputs();
    final RecordInput recordInput = inputs.getAsRecordInput(INPUT_SLOT_NAME);
    final Outputs outputs = taskContext.getOutputs();
    final RecordOutput recordOutput = outputs.getAsRecordOutput(OUTPUT_SLOT_NAME);
    Record record = recordInput.getRecord();
    while (record != null) {
      // this hash value reflects the changes of the record. It is set if a DeltaIndexing-Worker is called before.
      final String deltaHash = record.getMetadata().getStringValue(ImportingConstants.ATTRIBUTE_DELTA_HASH);
      if (deltaHash != null) {
        // check is done to avoid duplicates, because in case of an error the task may be retried
        if (!(_deltaIndexing.checkState(record.getSource(), record.getId(), jobRunId, deltaHash) == State.UPTODATE)) {
          pushToBulkbuilder(taskContext, jobName, record, recordOutput);
          _deltaIndexing.markAsUpdated(record.getSource(), record.getId(), jobRunId, deltaHash);
        }
      } else {
        pushToBulkbuilder(taskContext, jobName, record, recordOutput);
      }
      record = recordInput.getRecord();
    }
  }

  /**
   * send to record to bulkbuilder and write it to output bulk, if available, catch {@link InvalidRecordException} and
   * log it.
   */
  private void pushToBulkbuilder(final TaskContext taskContext, final String jobName, final Record record,
    final RecordOutput recordOutput) throws BulkbuilderException, ObjectStoreException, IOException {
    try {
      _bulkbuilder.addRecord(jobName, record);
      if (recordOutput != null) {
        recordOutput.writeRecord(record);
      }
    } catch (final InvalidRecordException ex) {
      taskContext.getLog().warn("Could not submit record '" + record.getId() + "' to bulkbuilder.", ex);
    }
  }

  @Override
  public String getName() {
    return WORKER_NAME;
  }

  /**
   * @param service
   *          the new deltaindexing
   */
  public void setDeltaIndexingService(final DeltaService service) {
    _deltaIndexing = service;
  }

  /**
   * @param service
   *          the deltaindexing to unset
   */
  public void unsetDeltaIndexingService(final DeltaService service) {
    if (_deltaIndexing == service) {
      _deltaIndexing = null;
    }
  }

  /**
   * @param service
   *          the new bulkbuilder
   */
  public void setBulkbuilderService(final BulkbuilderService service) {
    _bulkbuilder = service;
  }

  /**
   * @param service
   *          the bulkbuilder to unset
   */
  public void unsetBulkbuilderService(final BulkbuilderService service) {
    if (_bulkbuilder == service) {
      _bulkbuilder = null;
    }
  }

}
