/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/

package org.eclipse.smila.jobmanager;

import java.util.Map;

import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;

/**
 * Job management component handling/processing the lifecycle of a job run.
 */
public interface JobRunEngine {

  /**
   * start job with default job run mode (as defined by workflow or job, or {@link JobRunMode#STANDARD}.
   * 
   * @param jobName
   *          name of the job to start
   * @return id of the new job run
   * @throws JobManagerException
   *           error starting job
   */
  String startJob(String jobName) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job to start
   * @param jobMode
   *          the mode for the job to start. May be null, then method behaves just like {@link #startJob(String)}.
   * @return id of the new job run
   * @throws JobManagerException
   *           error starting job
   */
  String startJob(String jobName, JobRunMode jobMode) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job for which to finish the run.
   * @param jobRunId
   *          id of the job run to finish
   * @throws JobManagerException
   *           error finishing job
   */
  void finishJob(String jobName, String jobRunId) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job for which to cancel the run.
   * @param jobRunId
   *          id of the job run to cancel
   * @throws JobManagerException
   *           error canceling job
   */
  void cancelJob(String jobName, String jobRunId) throws JobManagerException;

  /**
   * Check if the job run data are still valid creates a new job run if not.
   * 
   * @param jobName
   *          The job name
   * @param jobRunId
   *          The job run id
   * @return The job run
   * @throws JobManagerException
   *           The job manager exception
   */
  JobRun ensureJobRun(final String jobName, final String jobRunId) throws JobManagerException;

  /**
   * Deletes transient instances of buckets from the collection of bucket instance ids. If the bucket instance is not
   * found in this instance of jobmanager it is lazily loaded before deleted.
   * 
   * @param jobName
   *          The job name
   * @param jobRunId
   *          The job run id
   * @param workflowRunId
   *          The workflow run id
   */
  void deleteTransientBulks(final String jobName, final String jobRunId, final String workflowRunId);

  /**
   * Checks if a job run is completed.
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @throws JobManagerException
   *           error
   */
  void checkAndHandleJobRunCompleted(final String jobName, final String jobRunId) throws JobManagerException;

  /**
   * delete a previously completed job run.
   * 
   * @param jobName
   *          name of the job for which to monitor the run.
   * @param jobRunId
   *          id of the complete job run to delete the monitoring data.
   * @throws JobManagerException
   *           run id is not a completed run or error deleting it.
   */
  void deleteJobRunData(String jobName, String jobRunId) throws JobManagerException;

  /**
   * Creates a map with active job runs using the given store.
   * 
   * @param storeName
   *          the store name
   * @return a map with all job runs using the store with given name. The job name is used as key, the job run id as
   *         value.
   */
  Map<String, String> getJobRunsUsingStore(String storeName);

}
