/*******************************************************************************
 * Copyright (c) 2009 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (empolis GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.blackboard.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.binarystorage.BinaryStorageException;
import org.eclipse.smila.binarystorage.BinaryStorageService;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Attachment;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InMemoryAttachment;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.StoredAttachment;
import org.eclipse.smila.datamodel.filter.RecordFilterHelper;
import org.eclipse.smila.datamodel.filter.RecordFilterNotFoundException;
import org.eclipse.smila.recordstorage.RecordStorage;
import org.eclipse.smila.recordstorage.RecordStorageException;
import org.eclipse.smila.utils.digest.DigestHelper;

/**
 * A non-persisting Blackboard implementation.
 */
public class BlackboardImpl implements Blackboard {

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** RecordStorage used by blackboard, may be null. */
  private RecordStorage _recordStorage;

  /** BinaryStorage used by blackboard, may be null. */
  private BinaryStorageService _binaryStorage;

  /** records currently loaded on blackboard. */
  private final Map<String, Record> _recordMap = new HashMap<String, Record>();

  /** IDs of records to remove from storages on commit. */
  private final Collection<String> _recordsToDelete = new HashSet<String>();

  /**
   * Global notes map: { note name -> note value }.
   */
  private final Map<String, Serializable> _globalNotes = new HashMap<String, Serializable>();

  /**
   * Record notes map: record Id -> { note name -> note value }.
   */
  private final Map<String, Map<String, Serializable>> _recordNotesMap =
    new HashMap<String, Map<String, Serializable>>();

  /**
   * The record filter helper.
   */
  private final RecordFilterHelper _filterHelper;

  /**
   * the data factory used in this blackboard for creating new records.
   */
  private final DataFactory _dataFactory = DataFactory.DEFAULT;

  /**
   * create instance.
   * 
   * @param filterHelper
   *          record filter manager.
   */
  public BlackboardImpl(final RecordFilterHelper filterHelper) {
    super();
    _filterHelper = filterHelper;
  }

  @Override
  public DataFactory getDataFactory() {
    return _dataFactory;
  }

  @Override
  public Record getRecord(final String id, final Get mode) throws BlackboardAccessException {
    assertNotNull(id);
    Record record = null;
    if (mode == Get.NEW) {
      synchronized (_recordMap) {
        evictRecord(id);
        record = getDataFactory().createRecord(id);
        putRecord(id, record);
      }
    } else {
      synchronized (_recordMap) {
        record = _recordMap.get(id);
        if (record == null) {
          if (!_recordsToDelete.contains(id)) {
            record = loadRecord(id);
          }
          if (record == null && mode == Get.AUTO_CREATE) {
            record = _dataFactory.createRecord(id);
          }
          putRecord(id, record);
        }
      }
    }
    return record;
  }

  @Override
  public Record getRecord(final String id) throws BlackboardAccessException {
    return getRecord(id, Get.EXISTING);
  }

  @Override
  public AnyMap getMetadata(final String id) throws BlackboardAccessException {
    final Record record = getRecord(id, Get.EXISTING);
    if (record == null) {
      return null;
    }
    return record.getMetadata();
  }

  @Override
  public Record getRecord(final String id, final String filterName) throws RecordFilterNotFoundException,
    BlackboardAccessException {
    final Record record = getRecord(id, Get.EXISTING);
    return filterRecord(record, filterName);
  }

  @Override
  public Record filterRecord(final Record record, final String filterName) throws RecordFilterNotFoundException {
    if (record == null) {
      return null;
    }
    final Record filteredRecord = _filterHelper.filter(record, filterName);
    if (_log.isDebugEnabled()) {
      _log.debug("record to filter: " + record);
      _log.debug("filtered record: " + filteredRecord);
    }
    return filteredRecord;
  }

  @Override
  public Record getRecordCopy(final String id, final boolean withAttachments) throws BlackboardAccessException {
    final Record record = getRecord(id, Get.EXISTING);
    final Record copy = record.getFactory().createRecord(record, withAttachments);
    if (withAttachments && _binaryStorage != null) {
      for (final Iterator<String> attachmentNames = copy.getAttachmentNames(); attachmentNames.hasNext();) {
        final String attachmentName = attachmentNames.next();
        final Attachment attachment = getAttachment(id, attachmentName);
        copy.setAttachment(attachment);
      }
    }
    return copy;
  }

  @Override
  public void setRecord(final Record record) throws BlackboardAccessException {
    final String id = assertIdNotNull(record);
    synchronized (_recordMap) {
      putRecord(id, record);
    }
    if (_binaryStorage != null && record.hasAttachments()) {
      for (final Iterator<String> attachmentNames = record.getAttachmentNames(); attachmentNames.hasNext();) {
        final String attachmentName = attachmentNames.next();
        final Attachment attachment = record.getAttachment(attachmentName);
        setAttachment(id, attachment);
      }
    }
  }

  @Override
  public void synchronizeRecord(final Record record) throws BlackboardAccessException {
    final String id = assertIdNotNull(record);
    final Record oldRecord = getRecord(id);
    if (oldRecord == null) {
      // no old version exists or can be loaded -> use new record as is
      setRecord(record);
    } else {
      synchronized (oldRecord) {
        final AnyMap metadata = record.getMetadata();
        copyAttributes(metadata, oldRecord.getMetadata());
        for (final Iterator<String> attachmentNames = record.getAttachmentNames(); attachmentNames.hasNext();) {
          final String attachmentName = attachmentNames.next();
          final Attachment attachment = record.getAttachment(attachmentName);
          setAttachment(id, attachment);
        }
      }
    }
  }

  @Override
  public void removeRecord(final String id) {
    assertNotNull(id);
    _recordsToDelete.add(id);
  }

  @Override
  public void removeAll() {
    _recordsToDelete.addAll(getRecordIds());
  }

  // - record attachments
  @Override
  public boolean hasAttachment(final String id, final String name) throws BlackboardAccessException {
    final Record record = getRecord(id, Get.EXISTING);
    if (record == null) {
      return false;
    }
    return record.hasAttachment(name);
  }

  @Override
  public Attachment getAttachment(final String id, final String name) throws BlackboardAccessException {
    final Record record = getRecord(id, Get.EXISTING);
    if (record != null && record.hasAttachment(name)) {
      if (_binaryStorage != null) {
        try {
          return new InMemoryAttachment(name, _binaryStorage.fetchAsByte(getAttachmentId(id, name)));
        } catch (final BinaryStorageException ex) {
          throw new BlackboardAccessException("Could not get the attachment from binary storage for record " + id,
            ex);
        }
      } else {
        return record.getAttachment(name);
      }
    }
    return null;
  }

  @Override
  public byte[] getAttachmentAsBytes(final String id, final String name) throws BlackboardAccessException {
    final Attachment attachment = getAttachment(id, name);
    if (attachment == null) {
      return null;
    }
    return attachment.getAsBytes();
  }

  @Override
  public InputStream getAttachmentAsStream(final String id, final String name) throws BlackboardAccessException {
    final Record record = getRecord(id, Get.EXISTING);
    if (record != null && record.hasAttachment(name)) {
      if (_binaryStorage == null) {
        return record.getAttachment(name).getAsStream();
      } else {
        try {
          return _binaryStorage.fetchAsStream(getAttachmentId(id, name));
        } catch (final BinaryStorageException bsex) {
          throw new BlackboardAccessException(
            "Could not get the attachment stream from binary storage for record having id :" + id, bsex);
        }
      }
    }
    return null;
  }

  @Override
  public void setAttachment(final String id, final Attachment attachment) throws BlackboardAccessException {
    final Record record = assertRecordExists(id);
    if (_binaryStorage == null || attachment instanceof StoredAttachment) {
      record.setAttachment(attachment);
    } else {
      final String name = attachment.getName();
      storeAttachment(id, name, attachment.getAsStream());
      record.setAttachment(new StoredAttachment(name, attachment.size()));
    }
  }

  @Override
  public void setAttachment(final String id, final String name, final byte[] attachment)
    throws BlackboardAccessException {
    setAttachment(id, new InMemoryAttachment(name, attachment));
  }

  @Override
  public void setAttachmentFromStream(final String id, final String name, final InputStream attachmentStream)
    throws BlackboardAccessException {
    final Record record = assertRecordExists(id);
    if (_binaryStorage == null) {
      try {
        record.setAttachment(new InMemoryAttachment(name, IOUtils.toByteArray(attachmentStream)));
      } catch (final IOException ex) {
        throw new BlackboardAccessException("Error loading attachment stream in memory.", ex);
      }
    } else {
      final long size = storeAttachment(id, name, attachmentStream);
      record.setAttachment(new StoredAttachment(name, size));
    }
  }

  @Override
  public void setAttachmentFromFile(final String id, final String name, final File attachmentFile)
    throws BlackboardAccessException {
    InputStream fileStream = null;
    try {
      fileStream = new FileInputStream(attachmentFile);
      setAttachmentFromStream(id, name, fileStream);
    } catch (final IOException ex) {
      throw new BlackboardAccessException(ex);
    } finally {
      IOUtils.closeQuietly(fileStream);
    }
  }

  @Override
  public void removeAttachment(final String id, final String name) throws BlackboardAccessException {
    final Record record = assertRecordExists(id);
    record.removeAttachment(name);
    if (_binaryStorage != null) {
      try {
        final String attachmentId = getAttachmentId(id, name);
        _binaryStorage.remove(attachmentId);
      } catch (final BinaryStorageException ex) {
        throw new BlackboardAccessException("Failed to remove attachment " + name
          + " from binary storage for record " + id, ex);
      }
    }
  }

  // - notes methods
  @Override
  public boolean hasGlobalNote(final String name) {
    synchronized (_globalNotes) {
      return _globalNotes.containsKey(name);
    }
  }

  @Override
  public Serializable getGlobalNote(final String name) {
    synchronized (_globalNotes) {
      return _globalNotes.get(name);
    }
  }

  @Override
  public void setGlobalNote(final String name, final Serializable object) {
    synchronized (_globalNotes) {
      _globalNotes.put(name, object);
    }
  }

  @Override
  public boolean hasRecordNote(final String id, final String name) {
    assertNotNull(id);
    synchronized (_recordNotesMap) {
      final Map<String, Serializable> recordNotes = _recordNotesMap.get(id);
      if (recordNotes == null) {
        return false;
      } else {
        return recordNotes.containsKey(name);
      }
    }
  }

  @Override
  public Serializable getRecordNote(final String id, final String name) {
    assertNotNull(id);
    synchronized (_recordNotesMap) {
      final Map<String, Serializable> recordNotes = _recordNotesMap.get(id);
      if (recordNotes == null) {
        return null;
      }
      return recordNotes.get(name);
    }
  }

  @Override
  public void setRecordNote(final String id, final String name, final Serializable object) {
    assertNotNull(id);
    synchronized (_recordNotesMap) {
      Map<String, Serializable> recordNotes = _recordNotesMap.get(id);
      if (recordNotes == null) {
        recordNotes = new HashMap<String, Serializable>();
        recordNotes.put(name, object);
        _recordNotesMap.put(id, recordNotes);
      } else {
        recordNotes.put(name, object);
        _recordNotesMap.put(id, recordNotes);
      }
    }
  }

  @Override
  public void commitRecord(final String id) throws BlackboardAccessException {
    assertNotNull(id);
    boolean isRemoved = false;
    synchronized (_recordsToDelete) {
      isRemoved = _recordsToDelete.contains(id);
    }
    if (isRemoved) {
      deleteRecord(id);
      evictRecord(id);
    } else {
      storeRecord(id);
    }
  }

  @Override
  public void unloadRecord(final String id) {
    final Record record = evictRecord(id);
    try {
      if (_recordStorage != null && !_recordStorage.existsRecord(id)) {
        removeAttachmentsFromBinaryStorage(record);
      }
    } catch (final RecordStorageException ex) {
      _log.warn("Error checking if record " + id + " exists in record storage", ex);
    }
  }

  @Override
  public void commit() throws BlackboardAccessException {
    int commitFailures = 0;
    int numberOfRecords = 0;
    synchronized (_recordMap) {
      numberOfRecords = _recordMap.size();
      for (final String id : getRecordIds()) {
        try {
          commitRecord(id);
        } catch (final Exception ex) {
          commitFailures++;
          _log.error("failed to commit: " + id, ex);
        }
      }
    }
    if (commitFailures > 0) {
      throw new BlackboardAccessException("Failed to commit " + commitFailures + " of " + numberOfRecords
        + " records on blackboard, see log for IDs.");
    }
  }

  @Override
  public void unload() {
    synchronized (_recordMap) {
      for (final String id : getRecordIds()) {
        try {
          unloadRecord(id);
        } catch (final Exception ex) {
          _log.warn("failed to invalidate: " + id, ex);
        }
      }
      _recordMap.clear();
    }
    synchronized (_globalNotes) {
      _globalNotes.clear();
    }
    synchronized (_recordNotesMap) {
      _recordNotesMap.clear();
    }
    synchronized (_recordsToDelete) {
      _recordsToDelete.clear();
    }
  }

  /**
   * Set the record service for blackboard. To be used by Declarative Services as the bind method.
   * 
   * @param recordStorage
   *          RecordStorage - the record storage service interface
   */
  protected void setRecordStorage(final RecordStorage recordStorage) {
    _recordStorage = recordStorage;
  }

  /**
   * Set the binary service for blackboard. To be used by Declarative Services as the bind method.
   * 
   * @param binaryStorage
   *          BinaryStorageService - the binary storage service interface
   */
  protected void setBinaryStorage(final BinaryStorageService binaryStorage) {
    _binaryStorage = binaryStorage;
  }

  /**
   * @throw {@link IllegalArgumentException} if record is null or record's ID is not set.
   * @return record's ID
   */
  private String assertIdNotNull(final Record record) {
    if (record == null) {
      throw new IllegalArgumentException("Record cannot be null");
    }
    final String id = record.getId();
    assertNotNull(id);
    return id;
  }

  /**
   * @throw {@link IllegalArgumentException} if ID null
   */
  private void assertNotNull(final String id) {
    if (id == null) {
      throw new IllegalArgumentException("Record Id cannot be null");
    }
  }

  /**
   * create a collection of all IDs of records on the blackboard.
   * 
   * @return collection containing IDs of all currently loaded records.
   */
  private Collection<String> getRecordIds() {
    return new ArrayList<String>(_recordMap.keySet());
  }

  /**
   * @return record if it exists
   * @throw BlackboardAccessException else.
   */
  private Record assertRecordExists(final String id) throws BlackboardAccessException {
    final Record record = getRecord(id, Get.EXISTING);
    if (record == null) {
      throw new BlackboardAccessException("Record " + id + " does not exist.");
    }
    return record;
  }

  // Utility methods
  /** load record from record storage, if available. */
  private Record loadRecord(final String id) throws BlackboardAccessException {
    if (id == null) {
      throw new IllegalArgumentException("Record ID cannot be null");
    }
    Record record = null;
    if (_recordStorage != null) {
      try {
        if (_log.isDebugEnabled()) {
          _log.debug("Loading record id: " + id + " from record storage.");
        }
        record = _recordStorage.loadRecord(id);
      } catch (final RecordStorageException ex) {
        throw new BlackboardAccessException("Error loading record with id = " + id, ex);
      }
    }
    return record;
  }

  /** write record to record storage, if attached. */
  private void storeRecord(final String id) throws BlackboardAccessException {
    try {
      Record record = null;
      synchronized (_recordMap) {
        record = _recordMap.get(id);
        if (record == null) {
          throw new BlackboardAccessException("Record " + id + " not loaded on blackboard, cannot commit it.");
        }
      }
      if (_recordStorage != null) {
        _recordStorage.storeRecord(record);
      }
    } catch (final RecordStorageException ex) {
      throw new BlackboardAccessException("Error committing record " + id + " to record storage", ex);
    }
  }

  /** delete record to record storage and attachments from binary storage, if attached. */
  private void deleteRecord(final String id) {
    try {
      final Record record = getRecord(id, Get.EXISTING);
      if (record != null) {
        removeAttachmentsFromBinaryStorage(record);
      }
    } catch (final BlackboardAccessException ex) {
      _log.warn("Failed to read record " + id + " for removal of attachments.", ex);
    }
    if (_recordStorage != null) {
      try {
        _recordStorage.removeRecord(id);
      } catch (final RecordStorageException ex) {
        _log.warn("Failed to record " + id + " from record storage.", ex);
      }

    }
  }

  /** remove record related data from memory. */
  private Record evictRecord(final String id) {
    assertNotNull(id);
    Record record = null;
    synchronized (_recordMap) {
      record = _recordMap.remove(id);
    }
    synchronized (_recordNotesMap) {
      _recordNotesMap.remove(id);
    }
    return record;
  }

  /** store record on blackboard, remove from delete list. */
  private void putRecord(final String id, final Record record) {
    _recordMap.put(id, record);
    _recordsToDelete.remove(id);
  }

  /**
   * Copy attributes.
   * 
   * @param source
   *          the source
   * @param destination
   *          the destination
   */
  private void copyAttributes(final AnyMap source, final AnyMap destination) {
    for (final String name : source.keySet()) {
      if (!Record.RECORD_ID.equals(name)) {
        final Any sourceAttribute = source.get(name);
        final Any attribute = _dataFactory.cloneAny(sourceAttribute);
        destination.put(name, attribute);
      }
    }
  }

  /**
   * Calculates the attachment id that will be used as a key in binary storage.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * 
   * @return the attachment id
   */
  private String getAttachmentId(final String id, final String name) {
    return DigestHelper.calculateDigest(id + "_ATTACHMENT_" + name);
  }

  /**
   * Saves attachment to binary storage from given InputStream and replaces actual attachment with null into
   * corresponding record.
   * 
   * @param id
   *          Record Id
   * @param name
   *          Attachment name
   * @param attachment
   *          Attachment object
   * @return size of attachment.
   * 
   * @throws BlackboardAccessException
   *           BlackboardAccessException
   */
  private long storeAttachment(final String id, final String name, final InputStream attachment)
    throws BlackboardAccessException {
    final String attachmentId = getAttachmentId(id, name);
    if (_log.isDebugEnabled()) {
      _log.debug("Saving attachment " + attachmentId + " to binary storage");
    }

    try {
      _binaryStorage.store(attachmentId, attachment);
      return _binaryStorage.fetchSize(attachmentId);
    } catch (final BinaryStorageException bsex) {
      throw new BlackboardAccessException(
        "Failed to save attachment in binary storage for record having id :" + id, bsex);
    }
  }

  /**
   * Removes attachments of the record from binary storage.
   */
  private void removeAttachmentsFromBinaryStorage(final Record record) {
    if (_binaryStorage != null && record != null && record.hasAttachments()) {
      for (final Iterator<String> attachmentNames = record.getAttachmentNames(); attachmentNames.hasNext();) {
        final String attachmentName = attachmentNames.next();
        try {
          _binaryStorage.remove(getAttachmentId(record.getId(), attachmentName));
        } catch (final BinaryStorageException storageException) {
          _log.error("Could not delete the attachment-file from binary storage for record having id :"
            + record.getId() + " - " + storageException.getMessage());
        }
      }
    }
  }

}
