/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.io;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * base class for {@link org.eclipse.smila.taskworker.input.Inputs} and
 * {@link org.eclipse.smila.taskworker.output.Outputs}. Provides simplified access to the {@link BulkInfo} objects in
 * the task and manages the input/output data object wrappers created by the subclasses. Finally, calculates some basic
 * IO durations for the task result counters.
 */
public abstract class IODataObjects {
  /** data objects associated to input slot names. */
  protected final Map<String, List<BulkInfo>> _dataObjects;

  /** input wrappers associated to input slot names, created on demand. */
  protected final Map<String, List<IODataObject>> _ioData = new HashMap<String, List<IODataObject>>();

  /** reference to objectstore service. */
  private final ObjectStoreService _objectStore;

  /**
   * create instance. Each subclass must implement the same constructor so that it can be instantiated by the
   * TaskContext.
   */
  public IODataObjects(final Map<String, List<BulkInfo>> dataObjects, final ObjectStoreService objectStore) {
    _dataObjects = dataObjects;
    _objectStore = objectStore;
  }

  /** @return number of data objects in the given slot. 0 if slot is not set. */
  public int getDataObjectCount(final String slotName) {
    if (_dataObjects.containsKey(slotName)) {
      return _dataObjects.get(slotName).size();
    }
    return 0;
  }

  /** @return first data object in given slot. Null, if slot is not set. */
  public BulkInfo getDataObject(final String slotName) {
    return getDataObject(slotName, 0);
  }

  /** @return n-th data object in given slot. Null, if slot is not set or has less objects. */
  public BulkInfo getDataObject(final String slotName, final int index) {
    if (index < getDataObjectCount(slotName)) {
      return _dataObjects.get(slotName).get(index);
    }
    return null;
  }

  /** @return reference to object store. */
  public ObjectStoreService getObjectStore() {
    return _objectStore;
  }

  /**
   * check if index is valid for slotname and no input object has yet been created for this object.
   */
  protected boolean canCreate(final String slotName, final int index) {
    final int dataObjectCount = getDataObjectCount(slotName);
    if (dataObjectCount > index) {
      List<IODataObject> slot = _ioData.get(slotName);
      if (slot == null) {
        slot = new ArrayList<IODataObject>();
        for (int i = 0; i <= dataObjectCount; i++) {
          slot.add(null); // list has to be initialized to access list element by index
        }
        _ioData.put(slotName, slot);
      }
      return slot.get(index) == null;
    }
    return false;
  }

  /** store IOData wrapper. */
  protected void putIOData(final String slotName, final int index, final IODataObject ioData) {
    final List<IODataObject> slotInputs = _ioData.get(slotName);
    slotInputs.add(index, ioData);
  }

  /** get stored IODataObjects. */
  protected Map<String, List<IODataObject>> getIOData() {
    return _ioData;
  }

  /** aggregate counters of input slots. */
  public void addCounters(final Map<String, Number> counters, final String name) {
    final String durationPerformName = Counters.DURATION_PERFORM + "." + name;
    for (final Map.Entry<String, List<IODataObject>> slots : _ioData.entrySet()) {
      final String slotName = slots.getKey();
      final String prefix = name + "." + slotName;
      for (final IODataObject io : slots.getValue()) {
        if (io != null) {
          final long durationOpen = io.getDurationOpen();
          if (durationOpen > 0) {
            Counters.addDuration(counters, Counters.DURATION_IODATA_OPEN, io.getDurationOpen());
          }
          final long durationPerform = io.getDurationPerform();
          if (durationPerform > 0) {
            Counters.addDuration(counters, durationPerformName, durationPerform);
            Counters.addDuration(counters, durationPerformName + "." + slotName, durationPerform);
          }
          Counters.addAll(counters, io.getCounter(), prefix);
        }
      }
    }
  }
}
