/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Dmitry Hazin (brox IT Solutions GmbH) - initial creator Sebastian Voigt (Brox IT Solutions GmbH) -
 * initial creator Alexander Eliseyev (brox IT Solutions GmbH)
 **********************************************************************************************************************/
package org.eclipse.smila.blackboard.test;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;

/**
 * The Class BlackboardServiceTest.
 */
public class TestTransientBlackboard extends DeclarativeServiceTestCase {

  /** This bundle id. */
  private static final String BUNDLE_ID = "org.eclipse.smila.blackboard.test";

  /** The _blackboard. */
  private Blackboard _blackboard;

  /** factory for data objects. */
  private DataFactory _factory;

  /**
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
    final BlackboardFactory factory = getService(org.eclipse.smila.blackboard.BlackboardFactory.class);
    assertNotNull(factory);
    _blackboard = factory.createTransientBlackboard();
    assertNotNull(_blackboard);
    _factory = _blackboard.getDataFactory();
    assertNotNull(_factory);
  }

  /**
   * test create method - just ensure that it doesn't throw an exception - there is no possibility to really test
   * anything.
   * 
   * @throws Exception
   *           test fails
   */
  public void testCreate() throws Exception {
    final String id = "testCreate";
    _blackboard.getRecord(id, Get.NEW);
    assertNotNull(_blackboard.getRecord(id));
  }

  /**
   * 
   * @throws Exception
   *           test fails
   */
  public void testFilter() throws Exception {
    final String id = "testFilter";
    final Record record = _blackboard.getRecord(id, Get.AUTO_CREATE);
    final AnyMap metadata = record.getMetadata();
    metadata.put("attribute", _factory.createStringValue("value"));
    metadata.put("hidden-attribute", _factory.createStringValue("value"));
    final Record filtered = _blackboard.getRecord(id, "filter");
    assertNotNull(filtered);
    final AnyMap filteredMetadata = filtered.getMetadata();
    assertNotNull(filteredMetadata);
    assertTrue(filteredMetadata.containsKey("attribute"));
    assertFalse(filteredMetadata.containsKey("hidden-attribute"));

    final Record full = _blackboard.getRecord(id);
    assertNotNull(full);
    final AnyMap fullMetadata = full.getMetadata();
    assertNotNull(fullMetadata);
    assertTrue(fullMetadata.containsKey("attribute"));
    assertTrue(fullMetadata.containsKey("hidden-attribute"));

    final Record filteredFull = _blackboard.filterRecord(full, "filter");
    assertNotNull(filteredFull);
    final AnyMap filteredFullMetadata = filteredFull.getMetadata();
    assertNotNull(filteredFullMetadata);
    assertTrue(filteredFullMetadata.containsKey("attribute"));
    assertFalse(filteredFullMetadata.containsKey("hidden-attribute"));
  }

  /**
   * 
   * @throws Exception
   *           test fails
   */
  public void testHasAttachment() throws Exception {
    final String id = "testHasAttachment";
    _blackboard.getRecord(id, Get.AUTO_CREATE);
    assertFalse(_blackboard.hasAttachment(id, "attachment"));
    _blackboard.setAttachment(id, "attachment", new byte[0]);
    assertTrue(_blackboard.hasAttachment(id, "attachment"));
  }

  /**
   * Test get record note.
   * 
   * @throws Exception
   *           the exception
   */
  public void testGetRecordNote() throws Exception {
    final String id = setTestRecord();
    assertEquals(_blackboard.hasRecordNote(id, "name1"), false);
    assertNull(_blackboard.getRecordNote(id, "name1"));
    _blackboard.setRecordNote(id, "name1", "note1");
    _blackboard.setRecordNote(id, "name2", "note2");
    assertEquals("note1", _blackboard.getRecordNote(id, "name1"));
    assertEquals("note2", _blackboard.getRecordNote(id, "name2"));
  }

  /**
   * Test has record note.
   * 
   * @throws Exception
   *           the exception
   */
  public void testHasRecordNote() throws Exception {
    final String id = setTestRecord();
    _blackboard.setRecordNote(id, "name1", "note1");
    assertEquals(_blackboard.hasRecordNote(id, "name1"), true);
    assertEquals(_blackboard.hasRecordNote(id, "name5"), false);
  }

  /**
   * Test set and get attachment.
   * 
   * @throws Exception
   *           the exception
   */
  public void testSetAndGetAttachment() throws Exception {
    final String attachment1 = "testattachment1";
    final String attachment2 = "testattachment2";
    final String id = setTestRecord();
    _blackboard.setAttachment(id, "test1", attachment1.getBytes());
    _blackboard.setAttachment(id, "test2", attachment2.getBytes());
    final byte[] storageAttachment1 = _blackboard.getAttachmentAsBytes(id, "test1");
    assertEquals(attachment1, new String(storageAttachment1));
    final byte[] storageAttachment2 = _blackboard.getAttachmentAsBytes(id, "test2");
    assertEquals(attachment2, new String(storageAttachment2));
  }

  /**
   * Test invalidate.
   * 
   * @throws Exception
   *           the exception
   */
  public void testUnload() throws Exception {
    final String id = setTestRecord();

    // Attach an attachment to check removing cached attachments
    final String attachment = "testattachment";
    _blackboard.setAttachment(id, "test", attachment.getBytes());
    _blackboard.commitRecord(id);
    assertNotNull(_blackboard.getRecord(id));
    _blackboard.unloadRecord(id);
    assertNull(_blackboard.getRecord(id));
  }

  /**
   * Test set and get attachment from stream.
   * 
   * @throws Exception
   *           the exception
   */
  public void testSetAndGetAttachmentFromStream() throws Exception {
    final String attachment = "testattachment1";
    final InputStream attachmentStream = new ByteArrayInputStream(attachment.getBytes());
    final String id = setTestRecord();
    _blackboard.setAttachmentFromStream(id, "test1", attachmentStream);

    final InputStream storageAttachmentStream = _blackboard.getAttachmentAsStream(id, "test1");
    assertEquals(attachment, new String(IOUtils.toByteArray(storageAttachmentStream)));

    assertNull(_blackboard.getAttachmentAsStream(id, "test2"));
    final byte[] storageAttachment = _blackboard.getAttachmentAsBytes(id, "test1");
    assertEquals(attachment, new String(storageAttachment));
  }

  /**
   * Test set, get and remove attachment.
   * 
   * @throws Exception
   *           the exception
   */
  public void testSetGetAndRemoveAttachment() throws Exception {
    // set attachments
    final String attachment1 = "testattachment1";
    final String attachment2 = "testattachment2";
    final String id = setTestRecord();
    _blackboard.setAttachment(id, "test1", attachment1.getBytes());
    _blackboard.setAttachment(id, "test2", attachment2.getBytes());

    // get the attachments
    final byte[] storageAttachment1 = _blackboard.getAttachmentAsBytes(id, "test1");
    assertEquals(attachment1, new String(storageAttachment1));
    final byte[] storageAttachment2 = _blackboard.getAttachmentAsBytes(id, "test2");
    assertEquals(attachment2, new String(storageAttachment2));

    // remove the attachments
    _blackboard.removeAttachment(id, "test1");
    _blackboard.removeAttachment(id, "test2");

    // check that the attachments do no longer exist
    try {
      _blackboard.getAttachmentAsBytes(id, "test1");
    } catch (final BlackboardAccessException e) {
      // final String expectedMsg = "Record with id = " + id + " doesn't have the attachment [test1]";
      final String message = e.getMessage();
      assertTrue(message.contains(id));
      assertTrue(message.contains("test1"));
      assertTrue(message.toLowerCase().contains("attachment"));
    }
    try {
      _blackboard.getAttachmentAsBytes(id, "test2");
    } catch (final BlackboardAccessException e) {
      // final String expectedMsg = "Record with id = " + id + " doesn't have the attachment [test2]";
      final String message = e.getMessage();
      assertTrue(message.contains(id));
      assertTrue(message.contains("test2"));
      assertTrue(message.toLowerCase().contains("attachment"));
    }

  }

  /**
   * Test set record.
   * 
   * @throws Exception
   *           the exception
   */
  public void testSetRecord() throws Exception {
    final String id = setTestRecord();
    final Record record = _blackboard.getRecord(id);
    final String attachment = "testSetRecord1";
    record.setAttachment(attachment, attachment.getBytes());
    // replace old record entry with new one
    _blackboard.setRecord(record);
    final Record getRecord = _blackboard.getRecord(id);
    assertEquals(record.getMetadata(), getRecord.getMetadata());
    assertTrue(Arrays.equals(attachment.getBytes(), record.getAttachmentAsBytes(attachment)));
    assertEquals(attachment, new String(_blackboard.getAttachmentAsBytes(id, attachment)));
    final AnyMap metadata = _blackboard.getMetadata(id);
    assertEquals(record.getMetadata(), metadata);
  }

  /**
   * Test setting attachment from file.
   * 
   * @throws Exception
   *           the exception
   */
  public void testSetAttachmentFromFile() throws Exception {
    final String attachment1 = "testSetAttachmentFromFile1";
    final File tempDir = WorkspaceHelper.createWorkingDir(BUNDLE_ID);
    final File attachmentFile = new File(tempDir, attachment1);
    final OutputStream output = FileUtils.openOutputStream(attachmentFile);
    IOUtils.write(attachment1.getBytes(), output);
    IOUtils.closeQuietly(output);

    final String id = setTestRecord();
    _blackboard.setAttachmentFromFile(id, attachment1, attachmentFile);

    final byte[] storageAttachment1 = _blackboard.getAttachmentAsBytes(id, attachment1);
    assertEquals(storageAttachment1.length, attachment1.getBytes().length);

    try {
      _blackboard.setAttachmentFromFile(id, attachment1, new File("unexistingFile"));
      fail("Must throw BlackboardAccessException");
    } catch (final BlackboardAccessException e) {
      ; // ok
    }
  }

  /** check the removeAll implementation. */
  public void testRemoveAll() throws Exception {
    final String id = setTestRecord();
    final Record record = _blackboard.getRecord(id);
    final Record clone1 = record.getFactory().cloneRecord(record, id + "clone1");
    _blackboard.setRecord(clone1);
    final Record clone2 = record.getFactory().cloneRecord(record, id + "clone2");
    _blackboard.setRecord(clone2);
    assertEquals(record.getMetadata(), _blackboard.getMetadata(record.getId()));
    assertEquals(clone1.getMetadata(), _blackboard.getMetadata(clone1.getId()));
    assertEquals(clone2.getMetadata(), _blackboard.getMetadata(clone2.getId()));
    _blackboard.removeAll();
    _blackboard.commit();
    assertNull(_blackboard.getMetadata(record.getId()));
    assertNull(_blackboard.getMetadata(clone1.getId()));
    assertNull(_blackboard.getMetadata(clone2.getId()));
  }

  /** check the getRecordCopy(..., true) method. */
  public void testgetRecordCopyWithAttachments() throws Exception {
    final String id = setTestRecord();
    final byte[] att1 = "attachment1".getBytes();
    final byte[] att2 = "attachment2".getBytes();
    _blackboard.setAttachment(id, "attachment1", att1);
    _blackboard.setAttachment(id, "attachment2", att2);
    final Record record = _blackboard.getRecord(id);
    final Record copy = _blackboard.getRecordCopy(id, true);
    assertNotSame(record, copy);
    assertNotSame(record.getMetadata(), copy.getMetadata());
    assertEquals(record.getMetadata(), copy.getMetadata());
    assertSame(att1, copy.getAttachmentAsBytes("attachment1"));
    assertSame(att2, copy.getAttachmentAsBytes("attachment2"));
  }

  /** check the getRecordCopy(..., false) method. */
  public void testgetRecordCopyWithoutAttachments() throws Exception {
    final String id = setTestRecord();
    final byte[] att1 = "attachment1".getBytes();
    final byte[] att2 = "attachment2".getBytes();
    _blackboard.setAttachment(id, "attachment1", att1);
    _blackboard.setAttachment(id, "attachment2", att2);
    final Record record = _blackboard.getRecord(id);
    final Record copy = _blackboard.getRecordCopy(id, false);
    assertNotSame(record, copy);
    assertNotSame(record.getMetadata(), copy.getMetadata());
    assertEquals(record.getMetadata(), copy.getMetadata());
    assertFalse(copy.hasAttachments());
  }

  /**
   * Sets the test record.
   * 
   * @return the id
   * 
   * @throws Exception
   *           the exception
   */
  private String setTestRecord() throws Exception {
    // CHECKSTYLE:OFF
    /*
     * Test record structure: <Record> <A n="author"> <L>l1</L> <L>l2</L> <O> <A n="firstName"> <L> <V>Igor</V> </L>
     * </A> <A n="lastName"> <L> <V>Novakovic</V> </L> </A> </O> <O st="appl:Author"> <A n="firstName1"> <L>
     * <V>Georg</V> </L> </A> <A n="lastName1"> <L> <V>Schmidt</V> </L> </A> </O> </A> <Record>
     */
    // CHECKSTYLE:ON
    // Create root attribute
    final AnySeq attribute = _factory.createAnySeq();

    // Create root attribute literals
    final Value l1 = _factory.createStringValue("l1");
    final Value l2 = _factory.createStringValue("l2");
    attribute.add(l1);
    attribute.add(l2);

    // Create first metadata object
    final AnyMap metadata1 = _factory.createAnyMap();
    final Value attribute11 = _factory.createStringValue("Igor");
    final Value attribute12 = _factory.createStringValue("Novakovic");
    // Set attributes to metadata1
    metadata1.put("firstName", attribute11);
    metadata1.put("lastName", attribute12);

    // Create second metadata object
    final AnyMap metadata2 = _factory.createAnyMap();
    // Create attribute21
    final Value attribute21 = _factory.createStringValue("Georg");
    final Value attribute22 = _factory.createStringValue("Schmidt");
    // Set attributes to metadata22
    metadata2.put("firstName1", attribute21);
    metadata2.put("lastName1", attribute22);

    // Add metadata1 and metadata2 to root attribute
    attribute.add(metadata1);
    attribute.add(metadata2);

    // Create record
    final Record record = _factory.createRecord();
    record.setId("source+key");

    // Create root metadata object
    final AnyMap metadata = record.getMetadata();
    metadata.put("author", attribute);

    // Set record to the blackboard
    _blackboard.setRecord(record);

    return record.getId();
  }
}
