/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.test;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.DeleteMethod;
import org.apache.commons.httpclient.methods.InputStreamRequestEntity;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.RequestEntity;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.eclipse.smila.bulkbuilder.BulkbuilderConstants;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.datamodel.validation.InvalidRecordException;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * base class for BulkBuilder tests.
 * 
 */
public abstract class BulkbuilderTestBase extends DeclarativeServiceTestCase {

  /**
   * Dummy worker.
   */
  protected static class DummyRecordAddWorker implements Runnable {

    /** the no of records. */
    public static final int NO_OF_RECORDS = 50;

    /** 'true': the worker also deletes, 'false': the worker only adds bulks. */
    private final boolean _deletes;

    /** the name of the job. */
    private final String _jobName;

    /** the bulk builder service reference. */
    private final BulkbuilderService _service;

    /** the base for the record ids. */
    private final int _recordNoBase;

    /** latch to coordinate start of run. */
    private final CountDownLatch _startLatch;

    /**
     * Constructs a new Worker.
     * 
     * @param deletes
     *          'true': the worker also deletes, 'false': the worker only adds bulks.
     * @param jobName
     *          the name of the job
     * @param service
     *          the bulbuilder service
     * @param recordNoBase
     *          the base record id for this worker.
     */
    public DummyRecordAddWorker(final boolean deletes, final String jobName, final BulkbuilderService service,
      final int recordNoBase, final CountDownLatch startLatch) {
      _deletes = deletes;
      _service = service;
      _jobName = jobName;
      _recordNoBase = recordNoBase;
      _startLatch = startLatch;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      if (_startLatch != null) {
        try {
          _startLatch.await();
        } catch (final InterruptedException e) {
          e.printStackTrace();
          Thread.currentThread().interrupt();
        }
      }
      for (int i = 0; i < NO_OF_RECORDS; i++) {
        try {
          _service.addRecord(_jobName, createRecord(i + _recordNoBase));
          if (_deletes) {
            _service.deleteRecord(_jobName, createRecord(i + _recordNoBase));
          }
        } catch (final BulkbuilderException e) {
          e.printStackTrace();
        } catch (final InvalidRecordException e) {
          e.printStackTrace();
        }
        try {
          Thread.sleep(100);
        } catch (final InterruptedException e) {
          e.printStackTrace();
          Thread.currentThread().interrupt();
        }
      }
    }
  }

  /** Exception for HTTP handling. */
  protected class HttpClientException extends Exception {
    /** serial version id. */
    private static final long serialVersionUID = 1L;

    /** HTTP status code of the response. */
    private final int _responseCode;

    /** constructs a new HttpClientException. */
    public HttpClientException(final int code) {
      super();
      _responseCode = code;
    }

    /** constructs a new HttpClientException. */
    public HttpClientException(final Throwable cause, final int code) {
      super(cause);
      _responseCode = code;
    }

    /** constructs a new HttpClientException. */
    public HttpClientException(final String message, final int code) {
      super(message);
      _responseCode = code;
    }

    /** constructs a new HttpClientException. */
    public HttpClientException(final String message, final Throwable cause, final int code) {
      super(message, cause);
      _responseCode = code;
    }

    /** @return the response code of the exception. */
    public int getResponseCode() {
      return _responseCode;
    }
  }

  /** 1 second in milliseconds. */
  protected static final int ONE_SECOND = 1000;

  /** 1200 characters for the content attribute. */
  protected static final String CONTENT =
    "Lorem ipsum dolor sit amet, consectetuer adipiscing elit, sed diam nonummy nibh euismod tincidunt ut "
      + "laoreet dolore magna aliquam erat volutpat. Ut wisi enim ad minim veniam, quis nostrud exerci tation "
      + "ullamcorper suscipit lobortis nisl ut aliquip ex ea commodo consequat. Duis autem vel eum iriure dolor "
      + "in hendrerit in vulputate velit esse molestie consequat, vel illum dolore eu feugiat nulla facilisis at "
      + "vero et accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue duis dolore te "
      + "feugait nulla facilisi. Lorem ipsum dolor sit amet, consectetuer adipiscing elit, sed diam nonummy nibh "
      + "euismod tincidunt ut laoreet dolore magna aliquam erat volutpat. Ut wisi enim ad minim veniam, quis nostrud "
      + "exerci tation ullamcorper suscipit lobortis nisl ut aliquip ex ea commodo consequat. Duis autem vel eum "
      + "iriure dolor in hendrerit in vulputate velit esse molestie consequat, vel illum dolore eu feugiat nulla "
      + "facilisis at vero et accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue "
      + "duis dolore te feugait nulla facilisi. Nam liber tempor cum soluta nobis eleifend option congue nihil "
      + "imperdiet doming id quod mazim placerat facer possim assum.";

  protected static final String UMLAUTS = "\u00e4\u00f6\u00fc\u00c4\u00d6\u00dc\u00df"; // "äöüÄÖÜß"

  /** name of test worker that consumes the bulks of added records. */
  protected static final String WORKER_BULKCONSUMER = "testBulkConsumer";

  /** name of test worker that consumes the bulks of deleted records. */
  protected static final String WORKER_DELETESCONSUMER = "testDeletesConsumer";

  /** local SMILA REST API host name and port. */
  protected static final String BASE_URI = "http://localhost:8080/smila/";

  /** waiting time for finishing tasks or running worker. */
  private static final long WAITING_TIME = 600000L;

  /** service under test. */
  protected BulkbuilderService _builder;

  /** job run data provider used for tests. */
  protected JobRunDataProvider _jobRunDataProvider;

  /** job run engine used for tests. */
  protected JobRunEngine _jobRunEngine;

  /** taskmanger used for tests. */
  protected TaskManager _taskManager;

  /** objectstore service for tests. */
  protected ObjectStoreService _objectStore;

  /** definition persistence reference. */
  protected DefinitionPersistence _defPersistence;

  /** http client. */
  protected HttpClient _httpClient = new HttpClient();

  /** converter reading Any from BON/JSON. */
  protected IpcAnyReader _ipcAnyReader = new IpcAnyReader();

  /** record writer, writes unformatted JSON. */
  protected IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  /*
   * �
   * 
   * /** prepare BulkBuilder for Bulk Mode.
   * 
   * {@inheritDoc}
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _builder = getService(BulkbuilderService.class);
    assertNotNull(_builder);
    _jobRunDataProvider = getService(JobRunDataProvider.class);
    assertNotNull(_jobRunDataProvider);
    _jobRunEngine = getService(JobRunEngine.class);
    assertNotNull(_jobRunEngine);
    _defPersistence = getService(DefinitionPersistence.class);
    assertNotNull(_defPersistence);
    _taskManager = getService(TaskManager.class);
    assertNotNull(_taskManager);
    _objectStore = getService(ObjectStoreService.class);
    assertNotNull(_objectStore);
    for (final String storeName : _objectStore.getStoreNames()) {
      try {
        _objectStore.clearStore(storeName);
      } catch (final Exception e) {
        // maybe windows is a bit too slow. retry it.
        Thread.sleep(500);
        _objectStore.clearStore(storeName);
      }
    }
    cleanupTasks(WORKER_DELETESCONSUMER);
    cleanupTasks(WORKER_BULKCONSUMER);
  }

  /**
   * create and start a bulkBuilderTest job.
   * 
   * @param jobName
   *          name of job
   * @param store
   *          tempStore to use for bulks.
   * @return job run id
   * @throws Exception
   *           job creation or start failed.
   */
  public String startJob(final String jobName, final String store) throws Exception {
    return startJob(jobName, store, (AnyMap) null);
  }

  /**
   * create and start a bulkBuilderTest job.
   * 
   * @param jobName
   *          name of job
   * @param store
   *          tempStore to use for bulks.
   * @return job run id
   * @throws Exception
   *           job creation or start failed.
   */
  public String startJob(final String jobName, final String store, final AnyMap additionalParameters)
    throws Exception {
    return startJob(jobName, "bulkBuilderTest", store, additionalParameters);
  }

  /**
   * create and start a job.
   * 
   * @param jobName
   *          name of job
   * @param workflowName
   *          name of job workflow
   * @param store
   *          tempStore to use for bulks.
   * @return job run id
   * @throws Exception
   *           job creation or start failed.
   */
  public String startJob(final String jobName, final String workflowName, final String store) throws Exception {
    return startJob(jobName, workflowName, store, null);
  }

  /**
   * create and start a job.
   * 
   * @param jobName
   *          name of job
   * @param workflowName
   *          name of job workflow
   * @param store
   *          tempStore to use for bulks.
   * @return job run id
   * @throws Exception
   *           job creation or start failed.
   */
  public String startJob(final String jobName, final String workflowName, final String store,
    final AnyMap additionalParameters) throws Exception {
    createJob(jobName, workflowName, store, additionalParameters);
    return _jobRunEngine.startJob(jobName);
  }

  /**
   * Cancel a job.
   * 
   * @param jobName
   *          The job name
   * @param jobId
   *          The job id
   * @throws Exception
   *           An exception if something went wrong
   */
  public void cancelJob(final String jobName, final String jobId) throws Exception {
    _jobRunEngine.cancelJob(jobName, jobId);
  }

  /**
   * create job.
   * 
   * @param jobName
   *          name of job
   * @param workflowName
   *          name of job workflow
   * @param store
   *          tempStore to use for bulks.
   * @throws Exception
   *           job creation failed.
   */
  protected void createJob(final String jobName, final String workflowName, final String store) throws Exception {
    createJob(jobName, workflowName, store, null);
  }

  /**
   * create job.
   * 
   * @param jobName
   *          name of job
   * @param workflowName
   *          name of job workflow
   * @param store
   *          tempStore to use for bulks.
   * @throws Exception
   *           job creation failed.
   */
  protected void createJob(final String jobName, final String workflowName, final String store,
    final AnyMap additionalParameters) throws Exception {
    final AnyMap jobDef = DataFactory.DEFAULT.createAnyMap();
    jobDef.put("name", DataFactory.DEFAULT.createStringValue(jobName));
    jobDef.put("workflow", DataFactory.DEFAULT.createStringValue(workflowName));
    final AnyMap parametersAny = DataFactory.DEFAULT.createAnyMap();
    parametersAny.put("tempStore", DataFactory.DEFAULT.createStringValue(store));
    if (additionalParameters != null) {
      parametersAny.putAll(additionalParameters);
    }
    jobDef.put("parameters", parametersAny);
    _defPersistence.removeJob(jobName);
    _defPersistence.addJob(new JobDefinition(jobDef));
  }

  /**
   * @param id
   *          record id
   * @return a record for testing.
   */
  protected static Record createRecord(final int id) {
    final Record record = DataFactory.DEFAULT.createRecord(Integer.toString(id));
    final Any content = DataFactory.DEFAULT.createStringValue(CONTENT);
    record.getMetadata().put("content", content);
    return record;
  }

  /**
   * @return a record without id for testing.
   */
  protected static Record createRecordWithoutId() {
    final Record record = DataFactory.DEFAULT.createRecord();
    final Any content = DataFactory.DEFAULT.createStringValue(CONTENT);
    record.getMetadata().put("content", content);
    return record;
  }

  /**
   * assert the attribute "_recordid" has the expected value.
   * 
   * @param record
   *          a record
   * @param id
   *          expected id.
   */
  protected void assertRecordId(final Record record, final int id) {
    final String idAttribute = record.getMetadata().getStringValue("_recordid");
    assertNotNull(idAttribute);
    assertEquals(Integer.toString(id), idAttribute);
  }

  /**
   * assert that "_recordId" is an Integer and returns that value.
   * 
   * @param record
   *          a record
   * @return record id
   */
  protected Integer getRecordIdAsInteger(final Record record) {
    final String idAttribute = record.getMetadata().getStringValue("_recordid");
    assertNotNull(idAttribute);
    return Integer.valueOf(idAttribute);
  }

  /**
   * @param task
   *          a task
   * @param slotName
   *          of input slot to read.
   * @return BulkStreamReader for this bulk to parse records.
   * @throws Exception
   *           task does not contain 1 bulk in slot or reading fails.
   */
  protected RecordInput readBulk(final Task task, final String slotName) throws Exception {
    assertNotNull(task.getInputBulks());
    assertNotNull("Slot '" + slotName + "' not found in input bulks for task " + task.toString(), task
      .getInputBulks().get(slotName));
    assertEquals(1, task.getInputBulks().get(slotName).size());
    final BulkInfo bulk = task.getInputBulks().get(slotName).get(0);
    assertTrue(_objectStore.existsObject(bulk.getStoreName(), bulk.getObjectName()));
    return new RecordInput(bulk, _objectStore);
  }

  /**
   * assert that a task exists for the given worker.
   * 
   * @param workerName
   *          worker name.
   * @return task, never null
   * @throws Exception
   *           error getting task or task was null.
   */
  protected Task getTask(final String workerName) throws Exception {
    final Task task = _taskManager.getTask(workerName, null);
    assertNotNull("expected a task for worker " + workerName, task);
    return task;
  }

  /**
   * check that there is not task for the given worker.
   * 
   * @param workerName
   *          worker name
   * @throws Exception
   *           error getting task or reveiced one.
   */
  protected void assertNoTask(final String workerName) throws Exception {
    final Task task = _taskManager.getTask(workerName, null);
    if (task != null) {
      finishTaskWithSuccess(task);
      fail("no task expected for " + workerName);
    }
  }

  /**
   * wait a given time until a new task for the worker is produced.
   * 
   * @param workerName
   *          name of worker.
   * @param seconds
   *          max wait time
   * @return task
   * @throws Exception
   *           got not task
   */
  protected Task waitForTaskWithoutAssertion(final String workerName, final int seconds) throws Exception {
    for (int i = 0; i < seconds; i++) {
      Thread.sleep(ONE_SECOND);
      final Task task = _taskManager.getTask(workerName, null);
      if (task != null) {
        return task;
      }
    }
    return null; // never reached.
  }

  /**
   * wait a given time until a new task for the worker is produced.
   * 
   * @param workerName
   *          name of worker.
   * @param seconds
   *          max wait time
   * @return task
   * @throws Exception
   *           got not task
   */
  protected Task waitForTask(final String workerName, final int seconds) throws Exception {
    final Task task = waitForTaskWithoutAssertion(workerName, seconds);
    assertNotNull("got no task for worker " + workerName + " after " + seconds + " seconds", task);
    return task;
  }

  /**
   * finish task with SUCCESSFUL result.
   * 
   * @param task
   *          task
   * @throws TaskmanagerException
   *           error
   */
  protected void finishTaskWithSuccess(final Task task) throws TaskmanagerException {
    _taskManager.finishTask(task.getWorkerName(), task.getTaskId(), successResult());
  }

  /**
   * @return a ResultDescription for a task with completion status SUCCESS.
   */
  protected ResultDescription successResult() {
    return new ResultDescription(TaskCompletionStatus.SUCCESSFUL, "", "", new HashMap<String, Number>());
  }

  /**
   * @return a ResultDescription for a task with completion status FATAL_ERROR.
   */
  protected ResultDescription fatalError() {
    return new ResultDescription(TaskCompletionStatus.FATAL_ERROR, "", "", new HashMap<String, Number>());
  }

  /**
   * finish all tasks for the worker.
   */
  protected void cleanupTasks(final String workerName) throws TaskmanagerException {
    Task task = null;
    do {
      task = _taskManager.getTask(workerName, null);
      if (task != null) {
        finishTaskWithSuccess(task);
      }
    } while (task != null);
  }

  /**
   * check bulk builder counters.
   */
  protected void checkCounters(final String jobName, final String jobId, final int expectedAddBulks,
    final int expectedAddRecords, final int expectedDelBulks, final int expectedDelRecords) throws Exception {
    final AnyMap jobRun = _jobRunDataProvider.getJobRunData(jobName, jobId);
    assertTrue(jobRun.containsKey("worker"));
    final AnyMap workerData = jobRun.getMap("worker");
    assertTrue(workerData.containsKey(BulkbuilderConstants.BULK_BUILDER_WORKER_NAME));
    final AnyMap builderCounter = workerData.getMap(BulkbuilderConstants.BULK_BUILDER_WORKER_NAME);
    checkSlotCounters(BulkbuilderConstants.BULK_BUILDER_INSERTED_RECORDS_SLOT, expectedAddBulks,
      expectedAddRecords, builderCounter);
    checkSlotCounters(BulkbuilderConstants.BULK_BUILDER_DELETED_RECORDS_SLOT, expectedDelBulks, expectedDelRecords,
      builderCounter);
  }

  /**
   * check bulk builder counters.
   */
  protected AnyMap checkKvoCounters(final String jobName, final String jobId, final int expectedBulks,
    final int expectedRecords) throws Exception {
    final AnyMap jobRun = _jobRunDataProvider.getJobRunData(jobName, jobId);
    assertTrue(jobRun.containsKey("worker"));
    final AnyMap workerData = jobRun.getMap("worker");
    assertTrue(workerData.containsKey(BulkbuilderConstants.BULK_BUILDER_WORKER_NAME));
    final AnyMap builderCounter = workerData.getMap(BulkbuilderConstants.BULK_BUILDER_WORKER_NAME);
    checkSlotCounters(BulkbuilderConstants.BULK_BUILDER_INSERTED_RECORDS_SLOT, 0, 0, builderCounter);
    checkSlotCounters(BulkbuilderConstants.BULK_BUILDER_DELETED_RECORDS_SLOT, 0, 0, builderCounter);
    return jobRun;
  }

  /**
   * check bulk builder counters for one slot.
   */
  protected void checkSlotCounters(final String slotName, final int expectedBulks, final int expectedRecords,
    final AnyMap counters) {
    final String prefix = "output." + slotName + ".";
    if (expectedBulks == 0) {
      assertFalse(counters.containsKey(prefix + "dataObjectCount"));
    } else {
      assertTrue(counters.get(prefix + "dataObjectCount").isLong());
      assertEquals(expectedBulks, counters.getLongValue(prefix + "dataObjectCount").intValue());
      assertTrue(counters.get(prefix + "recordCount").isLong());
      assertEquals(expectedRecords, counters.getLongValue(prefix + "recordCount").intValue());
      assertTrue(counters.get(prefix + "size").isLong());
      final int size = counters.getLongValue(prefix + "size").intValue();
      assertTrue("expected size>0, was: " + size, size > 0);
      assertTrue(counters.get(prefix + "duration").isDouble());
      final double duration = counters.getDoubleValue(prefix + "duration").doubleValue();
      assertTrue("expected duration>=0, was: " + duration, duration >= 0);
    }
  }

  /**
   * Wait until all finishing tasks are actually finished.
   */
  protected void waitForFinishingTasksProcessed() throws Exception {
    final long sleepTime = 500L;
    final long millisStarted = System.currentTimeMillis();
    while (true) {
      try {
        final Map<String, TaskCounter> taskCounters = _taskManager.getTaskCounters();
        for (final TaskCounter counter : taskCounters.values()) {
          if (TaskManager.FINISHING_TASKS_WORKER.equals(counter.getWorkerName())) {
            final int finishingTodo = counter.getTasksTodo();
            final int finishingInProgress = counter.getTasksInProgress();
            if (finishingInProgress == 0 && finishingTodo == 0) {
              return;
            }
          }
        }
        assertTrue("Waited too long for tasks to finish.",
          System.currentTimeMillis() - millisStarted <= WAITING_TIME);
        Thread.sleep(sleepTime);
      } catch (final Exception e) {
        e.printStackTrace();
        throw e;
      }
    }
  }

  /**
   * Wait until all workers are not running anymore.
   */
  protected void waitForRunningWorker() throws Exception {
    final long sleepTime = 500L;
    final long millisStarted = System.currentTimeMillis();
    final Set<String> activeWorkers = new HashSet<String>();
    while (true) {
      try {
        final Map<String, TaskCounter> taskCounters = _taskManager.getTaskCounters();
        for (final TaskCounter counter : taskCounters.values()) {
          final String workerName = counter.getWorkerName();
          final int finishingTodo = counter.getTasksTodo();
          final int finishingInProgress = counter.getTasksInProgress();
          if (finishingInProgress == 0 && finishingTodo == 0) {
            activeWorkers.remove(workerName);
          } else {
            activeWorkers.add(workerName);
          }
        }
        if (activeWorkers.isEmpty()) {
          return;
        }
        assertTrue("Waited too long for running worker.",
          System.currentTimeMillis() - millisStarted <= WAITING_TIME);
        Thread.sleep(sleepTime);
      } catch (final Exception e) {
        e.printStackTrace();
        throw e;
      }
    }
  }

  /**
   * add specified number of records, optionally commit afterwards, then check if expected number of bulks was created
   * and contains all records.
   */
  protected void addRecordsAndCheckBulks(final String jobName, final String jobId, final int noOfRecords,
    final boolean forceCommit, final int waitForFirstTask, final int expectedNoOfBulks) throws Exception {
    for (int i = 0; i < noOfRecords; i++) {
      _builder.addRecord(jobName, createRecord(i));
    }
    if (forceCommit) {
      _builder.commitJob(jobName);
    }
    checkBulks(jobName, jobId, 0, noOfRecords, waitForFirstTask, expectedNoOfBulks, false);
  }

  /**
   * add specified number of records, optionally commit afterwards, then check if expected number of bulks was created
   * and contains all records.
   */
  protected void deleteRecordsAndCheckBulks(final String jobName, final String jobId, final int noOfRecords,
    final boolean forceCommit, final int waitForFirstTask, final int expectedNoOfBulks) throws Exception {
    for (int i = 0; i < noOfRecords; i++) {
      _builder.deleteRecord(jobName, createRecord(i));
    }
    if (forceCommit) {
      _builder.commitJob(jobName);
    }
    checkBulks(jobName, jobId, 0, noOfRecords, waitForFirstTask, expectedNoOfBulks, true);
  }

  /**
   * check if expected number of bulks was created and contains all records.
   * 
   */
  protected void checkBulks(final String jobName, final String jobId, final int firstRecordId,
    final int noOfRecords, final int waitForFirstTask, final int expectedNoOfBulks, final boolean checkDeleteBulks)
    throws Exception {
    try {
      final String activeWorker = checkDeleteBulks ? WORKER_DELETESCONSUMER : WORKER_BULKCONSUMER;
      final String inactiveWorker = checkDeleteBulks ? WORKER_BULKCONSUMER : WORKER_DELETESCONSUMER;
      final String slotName = checkDeleteBulks ? "deletes" : "bulk";
      int taskCount = 0;
      int recordCount = 0;
      Task task = null;
      if (waitForFirstTask > 0) {
        task = waitForTask(activeWorker, waitForFirstTask);
      } else {
        waitForFinishingTasksProcessed();
        task = _taskManager.getTask(activeWorker, null);
      }
      while (task != null) {
        assertNoTask(inactiveWorker);
        taskCount++;
        final RecordInput reader = readBulk(task, slotName);
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              assertRecordId(record, recordCount + firstRecordId);
              recordCount++;
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
        task = _taskManager.getTask(activeWorker, null);
      }
      assertEquals(noOfRecords, recordCount);
      assertEquals(expectedNoOfBulks, taskCount);
      if (checkDeleteBulks) {
        checkCounters(jobName, jobId, 0, 0, expectedNoOfBulks, noOfRecords);
      } else {
        checkCounters(jobName, jobId, expectedNoOfBulks, noOfRecords, 0, 0);
      }
    } finally {
      try {
        if (_jobRunDataProvider.getJobRunInfo(jobName).getState() == JobState.RUNNING) {
          _jobRunEngine.finishJob(jobName, jobId);
        }
      } catch (final Exception ex) {
        ; // forget about it.
      }
    }
  }

  /**
   * create a single line JSON record with specified record id, terminated with a newline.
   */
  private byte[] createBulkRecord(final int recordId) throws Exception {
    return String.format("{\"_recordid\": \"%d\", \"content\": \"%s\", \"umlauts\": \"%s\" }\n", recordId, CONTENT,
      UMLAUTS).getBytes("utf-8");
  }

  /**
   * create a newline separated JSON-bulk with given size and record IDs starting with the given first Id.
   */
  protected byte[] createBulk(final int firstRecordId, final int bulkSize) throws Exception {
    final ByteArrayOutputStream bulk = new ByteArrayOutputStream();
    for (int i = 0; i < bulkSize; i++) {
      bulk.write(createBulkRecord(firstRecordId + i));
    }
    return bulk.toByteArray();
  }

  /**
   * Add a record via HTTP client.
   */
  protected Any addRecordWithClient(final String jobName, final Record record) throws Exception {
    return addRecordWithClient(jobName, record, false);
  }

  /**
   * Add a record via HTTP client, use chunked transfer optionally.
   */
  protected Any addRecordWithClient(final String jobName, final Record record, final boolean chunked)
    throws Exception {
    final PostMethod method = new PostMethod(BASE_URI + "job/" + jobName + "/record");
    method.setContentChunked(chunked);
    final RequestEntity requestEntity =
      new StringRequestEntity(_ipcRecordWriter.writeJsonObject(record), "application/json", "UTF-8");
    method.setRequestEntity(requestEntity);
    final int httpResponseCode = _httpClient.executeMethod(method);
    if (httpResponseCode >= HttpStatus.SC_BAD_REQUEST) {
      throw new HttpClientException(method.getResponseBodyAsString(), httpResponseCode);
    }
    return _ipcAnyReader.readJsonStream(method.getResponseBodyAsStream());
  }

  /**
   * Add a bulk via HTTP client.
   */
  protected Any addBulkStreamWithClient(final String jobName, final InputStream inputStream) throws Exception {
    final PostMethod method = new PostMethod(BASE_URI + "job/" + jobName + "/bulk");
    final RequestEntity requestEntity = new InputStreamRequestEntity(inputStream, "application/json");
    method.setRequestEntity(requestEntity);
    final int httpResponseCode = _httpClient.executeMethod(method);
    if (httpResponseCode >= HttpStatus.SC_BAD_REQUEST) {
      throw new HttpClientException(method.getResponseBodyAsString(), httpResponseCode);
    }
    return _ipcAnyReader.readJsonStream(method.getResponseBodyAsStream());
  }

  /**
   * Add a record and attachments via HTTP client, use chunked transfer optionally.
   */
  protected Any addRecordAndAttachFiles(final String jobName, final Record record,
    final Map<String, File> attachments, final boolean chunked) throws Exception {
    final PostMethod method = new PostMethod(BASE_URI + "job/" + jobName + "/record");
    method.setContentChunked(chunked);
    final HttpMethodParams params = new HttpMethodParams();
    final Part[] parts = new Part[attachments.size() + 1];
    parts[0] = new StringPart("metadata", _ipcRecordWriter.writeJsonObject(record), "UTF-8");
    int index = 1;
    for (final Map.Entry<String, File> attachment : attachments.entrySet()) {
      parts[index++] = new FilePart(attachment.getKey(), attachment.getValue());
    }
    final RequestEntity requestEntity = new MultipartRequestEntity(parts, params);
    method.setRequestEntity(requestEntity);
    final int httpResponseCode = _httpClient.executeMethod(method);
    if (httpResponseCode >= HttpStatus.SC_BAD_REQUEST) {
      throw new HttpClientException(method.getResponseBodyAsString(), httpResponseCode);
    }
    return _ipcAnyReader.readJsonStream(method.getResponseBodyAsStream());
  }

  /**
   * Performs a commit via HTTP client.
   * 
   * @param jobName
   *          The job name
   * @throws Exception
   *           An exception if something goes wrong
   */
  protected Any commitRecordWithClient(final String jobName) throws Exception {
    final PostMethod method = new PostMethod(BASE_URI + "job/" + jobName + "/record");
    final int httpResponseCode = _httpClient.executeMethod(method);
    if (httpResponseCode >= HttpStatus.SC_BAD_REQUEST) {
      throw new HttpClientException(method.getResponseBodyAsString(), httpResponseCode);
    }
    final InputStream resultStream = method.getResponseBodyAsStream();
    if (resultStream != null) {
      return _ipcAnyReader.readJsonStream(resultStream);
    }
    return null;
  }

  /**
   * Performs a commit via HTTP client.
   * 
   * @param jobName
   *          The job name
   * @throws Exception
   *           An exception if something goes wrong
   */
  protected Any commitBulkWithClient(final String jobName) throws Exception {
    final PostMethod method = new PostMethod(BASE_URI + "job/" + jobName + "/bulk");
    final int httpResponseCode = _httpClient.executeMethod(method);
    if (httpResponseCode >= HttpStatus.SC_BAD_REQUEST) {
      throw new HttpClientException(method.getResponseBodyAsString(), httpResponseCode);
    }
    return _ipcAnyReader.readJsonStream(method.getResponseBodyAsStream());
  }

  /**
   * Deletes a record via HTTP client.
   * 
   * @param jobName
   *          The job name
   * @param id
   *          The id of the record to delete
   * @throws Exception
   *           An exception if something goes wrong
   */
  protected Any deleteRecordWithClient(final String jobName, final int id) throws Exception {
    final DeleteMethod method = new DeleteMethod(BASE_URI + "job/" + jobName + "/record/?_recordid=" + id);
    final int httpResponseCode = _httpClient.executeMethod(method);
    if (httpResponseCode >= HttpStatus.SC_BAD_REQUEST) {
      throw new HttpClientException(method.getResponseBodyAsString(), httpResponseCode);
    }
    return _ipcAnyReader.readJsonStream(method.getResponseBodyAsStream());
  }

  /**
   * Deletes a record via HTTP client.
   * 
   * @param jobName
   *          The job name
   * @param id
   *          The id of the record to delete
   * @throws Exception
   *           An exception if something goes wrong
   */
  protected Any deleteBigRecordWithClient(final String jobName, final int id) throws Exception {
    final DeleteMethod method =
      new DeleteMethod(BASE_URI + "job/" + jobName + "/record/?_recordid=" + id + "&content="
        + URLEncoder.encode(CONTENT, "UTF-8"));
    final int httpResponseCode = _httpClient.executeMethod(method);
    if (httpResponseCode >= HttpStatus.SC_BAD_REQUEST) {
      throw new HttpClientException(method.getResponseBodyAsString(), httpResponseCode);
    }
    return _ipcAnyReader.readJsonStream(method.getResponseBodyAsStream());
  }
}
