/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.helper;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.bulkbuilder.BulkbuilderConstants;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.InvalidJobException;
import org.eclipse.smila.bulkbuilder.outputs.AppendableBulkOutput;
import org.eclipse.smila.bulkbuilder.outputs.BulkOutput;
import org.eclipse.smila.bulkbuilder.outputs.BulkType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.jobmanager.WorkflowRunInfo;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskworker.io.IODataObject;
import org.eclipse.smila.taskworker.output.AppendableOutput;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * basic BulkBuilder implementation.
 * 
 * Creates bulks from the pushed records on a time and size limit.
 * 
 */
public abstract class BulkbuilderBase implements BulkTrackerCallback {
  /** Number of ms per s. */
  private static final double MILLISECONDS_PER_SECOND = 1000.0;

  /** a reference to the object store service. */
  private final ObjectStoreService _objectStore;

  /** the bulk builder's task provider. */
  private final BulkbuilderTaskProvider _taskProvider;

  /** the micro bulk builder. */
  private final MicroBulkbuilder _microBulkbuilder;

  /** stores insert bulks for various job names. key = job name. */
  private final Map<String, BulkOutput> _activeInsertBulks = new HashMap<String, BulkOutput>();

  /** stores delete bulks for various job names. key = job name. */
  private final Map<String, BulkOutput> _activeDeleteBulks = new HashMap<String, BulkOutput>();

  /** local Logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * Creates a new BulkBuilder instance.
   * 
   * @param objectStore
   *          a reference to the objectStore the object store service.
   * @param taskProvider
   *          the task provider that generates tasks for the bulkbuilder
   * @param microBulkbuilder
   *          helper class for constructing in-memory micro bulk that can be appended to the objectstore bulk in one
   *          atomic append step.
   * @throws BlackboardAccessException
   *           cannot access blackboard.
   */
  public BulkbuilderBase(final ObjectStoreService objectStore, final BulkbuilderTaskProvider taskProvider,
    final MicroBulkbuilder microBulkbuilder) throws BlackboardAccessException {
    _objectStore = objectStore;
    _taskProvider = taskProvider;
    _microBulkbuilder = microBulkbuilder;
  }

  /**
   * Write record to add bulk.
   * 
   * @param jobName
   *          The job name
   * @param record
   *          The record to store
   * @throws BulkbuilderException
   *           error writing to object store
   */
  public synchronized WorkflowRunInfo addRecord(final String jobName, final Record record)
    throws BulkbuilderException {
    if (_log.isTraceEnabled()) {
      _log.trace("addRecord start");
    }
    try {
      final Task task = _taskProvider.getInitialTask(jobName);
      return writeRecordWithRetry(jobName, record, task, BulkType.ADD);
    } finally {
      if (_log.isTraceEnabled()) {
        _log.trace("addRecord end");
      }
    }
  }

  /**
   * Write record to delete bulk.
   * 
   * @param jobName
   *          The job name
   * @param record
   *          The record to store
   * @throws BulkbuilderException
   *           error writing to object store
   */
  public synchronized WorkflowRunInfo deleteRecord(final String jobName, final Record record)
    throws BulkbuilderException {
    if (_log.isTraceEnabled()) {
      _log.trace("deleteRecord start");
    }
    try {
      final Task task = _taskProvider.getInitialTask(jobName);
      return writeRecordWithRetry(jobName, record, task, BulkType.DEL);
    } finally {
      if (_log.isTraceEnabled()) {
        _log.trace("deleteRecord end");
      }
    }

  }

  /**
   * Commits active bulks for given job name.
   * 
   * @param jobName
   *          The job name
   * @throws BulkbuilderException
   *           An exception if something goes wrong
   */
  public synchronized WorkflowRunInfo commitBulk(final String jobName) throws BulkbuilderException {
    if (_log.isTraceEnabled()) {
      _log.trace("commitBulk start for job  '" + jobName + "'");
    }
    _taskProvider.checkJobActive(jobName);
    try {
      final BulkOutput addBulk = getActiveBulk(jobName, BulkType.ADD);
      final BulkOutput delBulk = getActiveBulk(jobName, BulkType.DEL);

      final Map<String, Number> counters = getResultCounters(jobName, addBulk, delBulk);
      final long startTime = System.nanoTime();
      if (addBulk != null) {
        commitBulk(addBulk);
      }
      if (delBulk != null) {
        commitBulk(delBulk);
      }
      addIoCloseDuration(counters, startTime);
      final ResultDescription taskResult;
      if (addBulk == null && delBulk == null) {
        taskResult = new ResultDescription(TaskCompletionStatus.OBSOLETE, null, null, counters);
      } else {
        taskResult = new ResultDescription(TaskCompletionStatus.SUCCESSFUL, null, null, counters);
      }
      final Task task = _taskProvider.finishTask(jobName, taskResult);
      if (_log.isTraceEnabled()) {
        _log.trace("commitBulk end for job '" + jobName + "'");
      }
      return createWorkflowRunInfo(task);
    } catch (final BulkbuilderException e) {
      _taskProvider.finishTask(jobName, new ResultDescription(TaskCompletionStatus.RECOVERABLE_ERROR, getClass()
        .getSimpleName(), e.toString(), null));
      throw e;
    }
  }

  /**
   * Add record to micro bulk.
   * 
   * @param jobName
   *          The job name
   * @param record
   *          The record to store
   * @param microBulkId
   *          The id of the micro bulk
   * @throws BulkbuilderException
   *           error writing to object store
   */
  public WorkflowRunInfo addToMicroBulk(final String jobName, final Record record, final String microBulkId)
    throws BulkbuilderException {
    if (_log.isTraceEnabled()) {
      _log.trace("addToMicroBulk start");
    }
    final Task task = _taskProvider.getInitialTask(jobName);
    final Map<String, List<BulkInfo>> outputBulks = task.getOutputBulks();
    if (outputBulks.containsKey(BulkbuilderConstants.BULK_BUILDER_INSERTED_RECORDS_SLOT)) {
      _microBulkbuilder.addToMicroBulk(microBulkId, record);
      return null;
    }
    throw new InvalidJobException("Operation not supported in job '" + jobName + "', slot '"
      + BulkbuilderConstants.BULK_BUILDER_INSERTED_RECORDS_SLOT + "' is not connected to a bucket.");
  }

  /**
   * Finishes the micro bulk.
   * 
   * @param jobName
   *          The job name
   * @param microBulkId
   *          The id of the micro bulk
   * @throws BulkbuilderException
   *           error writing to object store
   */
  public synchronized WorkflowRunInfo finishMicroBulk(final String jobName, final String microBulkId)
    throws BulkbuilderException {
    try {
      if (_log.isTraceEnabled()) {
        _log.trace("finishMicroBulk start " + microBulkId);
      }
      final Integer numberOfRecords = _microBulkbuilder.getNumberOfRecords(microBulkId);
      if (numberOfRecords <= 0) {
        // if nothing has been added to the bulk, existing current bulks must be committed
        return commitBulk(jobName);
      } else {
        final Task task = _taskProvider.getInitialTask(jobName);
        // if insertedRecords is not defined, the next line will throw an exception.
        final byte[] microBulk = _microBulkbuilder.finishMicroBulk(microBulkId);
        return writeMicroBulkWithRetry(jobName, microBulk, numberOfRecords, task);
      }
    } finally {
      if (_log.isTraceEnabled()) {
        _log.trace("finishMicroBulk end");
      }
    }
  }

  /**
   * Removes the micro bulk with the given id.
   * 
   * @param microBulkId
   *          The id of the micro bulk
   */
  public void removeMicroBulk(final String microBulkId) {
    _microBulkbuilder.finishMicroBulk(microBulkId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void checkBulks() throws BulkbuilderException {
    if (_log.isTraceEnabled()) {
      _log.trace("checkBulks start");
    }

    // first add bulks must be committed
    final Set<String> jobsToCommit = new HashSet<String>();

    for (final Entry<String, BulkOutput> curr : _activeInsertBulks.entrySet()) {
      final BulkOutput currBulk = curr.getValue();
      if (currBulk.isTimedOut()) {
        jobsToCommit.add(curr.getKey());
      }
    }

    for (final Entry<String, BulkOutput> curr : _activeDeleteBulks.entrySet()) {
      final BulkOutput currBulk = curr.getValue();
      if (currBulk.isTimedOut()) {
        jobsToCommit.add(curr.getKey());
      }
    }

    for (final String jobName : jobsToCommit) {
      commitBulkQuietly(jobName);
    }

    if (_log.isTraceEnabled()) {
      _log.trace("checkBulks end");
    }
  }

  /**
   * Shutdown builder and close all open bulks.
   */
  public synchronized void shutdown() {
    for (final BulkOutput curr : Collections.unmodifiableCollection(_activeInsertBulks.values())) {
      handleShutdownForBulk(curr);
    }
    _activeInsertBulks.clear();

    for (final BulkOutput curr : Collections.unmodifiableCollection(_activeDeleteBulks.values())) {
      handleShutdownForBulk(curr);
    }
    _activeDeleteBulks.clear();
  }

  /**
   * @return the timeout for committing bulks either from task parameter or from some configuration or default value.
   */
  protected abstract long getCommitTimeoutMillis(final AnyMap taskParameters);

  /**
   * @return the maximum bulk size either from task parameter or from some configuration or default value.
   */
  protected abstract long getBulkSizeLimit(final BulkType bulkType, final AnyMap taskParameters);

  /** get reference to object store service. */
  protected ObjectStoreService getObjectStore() {
    return _objectStore;
  }

  /** get reference to task provider helper. */
  protected BulkbuilderTaskProvider getTaskProvider() {
    return _taskProvider;
  }

  /** get reference to microbulk helper. */
  protected MicroBulkbuilder getMicroBulkbuilder() {
    return _microBulkbuilder;
  }

  /**
   * Get active bulk data for given job name and bulk type.
   * 
   * @param jobName
   *          The job name.
   * @param bulkType
   *          bulk type
   * @return BulkData object or <code>null</code> if none exists.
   */
  protected BulkOutput getActiveBulk(final String jobName, final BulkType bulkType) {
    if (bulkType == BulkType.DEL) {
      return _activeDeleteBulks.get(jobName);
    } else {
      return _activeInsertBulks.get(jobName);
    }
  }

  /**
   * set active bulk data for given job name and bulk type.
   * 
   * @param jobName
   *          The job name.
   * @param bulkData
   *          bulk data to set
   */
  protected void setActiveBulk(final String jobName, final BulkOutput bulkData) {
    final BulkType bulkType = bulkData.getBulkType();
    if (bulkType == BulkType.DEL) {
      _activeDeleteBulks.put(jobName, bulkData);
    } else {
      _activeInsertBulks.put(jobName, bulkData);
    }
  }

  /**
   * Adds record with retry if something goes wrong.
   * 
   * @param jobName
   *          The job name
   * @param record
   *          The record to store
   * @param task
   *          The task
   * @throws BulkbuilderException
   *           error writing to object store
   */
  protected WorkflowRunInfo writeRecordWithRetry(final String jobName, final Record record, final Task task,
    final BulkType bulkType) throws BulkbuilderException {
    try {
      return writeRecord(jobName, record, task, bulkType);
    } catch (final InvalidJobException ex) {
      throw ex;
    } catch (final BulkbuilderException ex) {
      final Task newTask = handleErrorOnRecordProcessing(jobName, ex);
      return writeRecord(jobName, record, newTask, bulkType);
    }
  }

  /** write record to bulk, commit job if configured limits are reached. */
  private WorkflowRunInfo writeRecord(final String jobName, final Record record, final Task task,
    final BulkType bulkType) throws BulkbuilderException {
    final BulkInfo bulkInfo = getBulkInfo(jobName, task.getOutputBulks(), bulkType);
    final BulkOutput currBulk = openBulk(jobName, bulkInfo, bulkType, task.getParameters());
    currBulk.addRecord(record);
    currBulk.setLastModificationTime(System.currentTimeMillis());
    if (currBulk.hasGrownBeyondLimit()) {
      commitBulk(jobName);
    }
    return createWorkflowRunInfo(task);
  }

  /**
   * Adds microbulk with retry if something goes wrong.
   * 
   * @param jobName
   *          The job name
   * @param microBulk
   *          The microbulk to store
   * @param numberOfRecords
   *          number of records in this bulk.
   * @param task
   *          The task
   * @throws BulkbuilderException
   *           error writing to object store
   */
  protected WorkflowRunInfo writeMicroBulkWithRetry(final String jobName, final byte[] microBulk,
    final Integer numberOfRecords, final Task task) throws BulkbuilderException {
    try {
      return writeMicroBulk(jobName, microBulk, numberOfRecords, task);
    } catch (final InvalidJobException ex) {
      throw ex;
    } catch (final BulkbuilderException ex) {
      final Task newTask = handleErrorOnRecordProcessing(jobName, ex);
      return writeMicroBulk(jobName, microBulk, numberOfRecords, newTask);
    }
  }

  /**
   * Add a micro bulk to a bulk for the given job. Create a new bulk if none is open.
   */
  private WorkflowRunInfo writeMicroBulk(final String jobName, final byte[] microBulk,
    final Integer numberOfRecords, final Task task) throws BulkbuilderException {
    final BulkInfo bulkInfo = getBulkInfo(jobName, task.getOutputBulks(), BulkType.ADD);
    final BulkOutput currBulk = openBulk(jobName, bulkInfo, BulkType.ADD, task.getParameters());
    if (_log.isDebugEnabled()) {
      _log.debug("Writing micro bulk to bulk " + currBulk.getCurrentBulkId() + " for job '" + jobName + "'.");
    }
    ((AppendableBulkOutput) currBulk).addMicroBulk(microBulk, numberOfRecords);
    currBulk.setLastModificationTime(System.currentTimeMillis());
    if (currBulk.hasGrownBeyondLimit()) {
      commitBulk(jobName);
    }
    return createWorkflowRunInfo(task);
  }

  /**
   * gets the bulk info from the output bulks for the output slots connected to this worker.
   * 
   * @param jobName
   *          job name
   * @param bulkType
   *          bulk type (add or delete)
   * @return bulk info for the operation
   * @throws BulkbuilderException
   *           task does not have the slot connected
   */
  protected BulkInfo getBulkInfo(final String jobName, final Map<String, List<BulkInfo>> outputBulks,
    final BulkType bulkType) throws BulkbuilderException {
    String slotName;
    List<BulkInfo> slotBulkInfos = null;
    if (bulkType == BulkType.DEL) {
      slotName = BulkbuilderConstants.BULK_BUILDER_DELETED_RECORDS_SLOT;
      slotBulkInfos = outputBulks.get(slotName);
      if (slotBulkInfos == null || slotBulkInfos.isEmpty()) {
        throw new InvalidJobException("Operation not supported in job '" + jobName + "', slot '"
          + BulkbuilderConstants.BULK_BUILDER_DELETED_RECORDS_SLOT + "' is not connected to a bucket.");
      }
    } else if (bulkType == BulkType.ADD) {
      slotName = BulkbuilderConstants.BULK_BUILDER_INSERTED_RECORDS_SLOT;
      slotBulkInfos = outputBulks.get(slotName);
      if (slotBulkInfos == null || slotBulkInfos.isEmpty()) {
        throw new InvalidJobException("Operation not supported in job '" + jobName + "', slot '"
          + BulkbuilderConstants.BULK_BUILDER_INSERTED_RECORDS_SLOT + "' is not connected to a bucket.");
      }
    } else {
      throw new InvalidJobException("Bulk type " + bulkType + " not supported in Bulkbuilder.");
    }
    final BulkInfo bulkInfo = slotBulkInfos.get(0);
    return bulkInfo;
  }

  /**
   * Open bulk. Create new one if no active bulk exists.
   * 
   * @return BulkData current bulk for given index name
   * @throws BulkbuilderException
   *           error accessing the object store.
   */
  private BulkOutput openBulk(final String jobName, final BulkInfo bulkInfo, final BulkType bulkType,
    final AnyMap taskParameters) throws BulkbuilderException {
    BulkOutput activeBulk = getActiveBulk(jobName, bulkType);
    if (activeBulk != null) {
      if (bulkInfo.getObjectName().equals(activeBulk.getCurrentBulkId())) {
        if (_log.isTraceEnabled()) {
          _log.trace("openBulk: active bulk found");
        }
        return activeBulk;
      } else {
        _log.warn("Active bulk does not match to given bulk info and has been released,"
          + " maybe that the respective job has been canceled.");
        releaseBulk(activeBulk);
      }
    }
    if (_log.isTraceEnabled()) {
      _log.trace("openBulk: no active bulk found - creating new bulk");
    }
    activeBulk = createBulk(jobName, bulkInfo, bulkType, taskParameters);
    setActiveBulk(jobName, activeBulk);
    return activeBulk;
  }

  /**
   * create an BulkOutput for the bulk.
   * 
   * @param jobName
   *          the name of the current job
   * @param bulkInfo
   *          the bulk info for the curent bulk
   * @param bulkType
   *          the bulk's type
   * @param taskParameters
   *          task parameters
   * @return the {@link BulkOutput} for the bulk.
   */
  protected BulkOutput createBulk(final String jobName, final BulkInfo bulkInfo, final BulkType bulkType,
    final AnyMap taskParameters) {
    final BulkOutput bulkData = new AppendableBulkOutput(jobName, bulkInfo.getObjectName(), bulkType);
    bulkData.setBulk(new AppendableOutput(bulkInfo, _objectStore));
    bulkData.setCommitTimeout(getCommitTimeoutMillis(taskParameters));
    bulkData.setBulkSizeLimit(getBulkSizeLimit(bulkType, taskParameters));
    return bulkData;
  }

  /**
   * Release active bulk.
   * 
   * @param currBulk
   *          current bulk for index.
   */
  private void releaseBulk(final BulkOutput currBulk) {
    final String jobName = currBulk.getJobName();
    if (_log.isDebugEnabled() && currBulk.getCurrentBulkId() != null) {
      _log.debug("Releasing bulk " + currBulk.getCurrentBulkId() + " for job name '" + jobName + "'.");
    }
    currBulk.setBulk(null);
    currBulk.setCurrentBulkId(null);
    switch (currBulk.getBulkType()) {
      case DEL:
        _activeDeleteBulks.remove(jobName);
        break;
      case ADD:
        _activeInsertBulks.remove(jobName);
        break;
      default:
        break;
    }
  }

  /**
   * commit given bulk.
   * 
   * @param currBulk
   *          current bulk
   * @throws BulkbuilderException
   *           error committing the bulk
   */
  private void commitBulk(final BulkOutput currBulk) throws BulkbuilderException {
    try {
      if (_log.isDebugEnabled()) {
        _log.debug("Committing bulk " + currBulk.getCurrentBulkId() + " for job '" + currBulk.getJobName() + "'");
      }
      currBulk.commit();
    } catch (final RuntimeException ex) {
      final String message = "Error submitting bulk " + currBulk.getCurrentBulkId();
      throw new BulkbuilderException(message, ex);
    } finally {
      releaseBulk(currBulk);
    }
  }

  /** try to commit and catch and log all exceptions. Fail task with fatal error if committing fails. */
  private void commitBulkQuietly(final String jobName) {
    try {
      commitBulk(jobName);
    } catch (final BulkbuilderException ex) {
      _log.warn("Failed to commit bulks for job '" + jobName + "'", ex);
      if (ex.isRecoverable()) {
        finishTaskWithError(jobName, TaskCompletionStatus.RECOVERABLE_ERROR, ex);
      } else {
        finishTaskWithError(jobName, TaskCompletionStatus.FATAL_ERROR, ex);
      }
    }
  }

  /**
   * called after error on adding data to bulks: try to commit the current task and get a new one so that the calling
   * method can retry adding data to a new bulk.
   */
  private Task handleErrorOnRecordProcessing(final String jobName, final BulkbuilderException ex)
    throws BulkbuilderException {
    _log.warn("Error adding/deleting record for job '" + jobName + "', retrying with new task.", ex);
    commitBulkQuietly(jobName);
    return _taskProvider.getInitialTask(jobName);
  }

  /** finish task with error and remove open bulks. */
  private void finishTaskWithError(final String jobName, final TaskCompletionStatus completionStatus,
    final Exception ex) {
    try {
      _taskProvider.finishTask(jobName,
        new ResultDescription(completionStatus, getClass().getSimpleName(), ex.toString(), null));
    } catch (final Exception e2) {
      _log.warn("Error during clean up.", e2);
    } finally {
      _activeInsertBulks.remove(jobName);
      _activeDeleteBulks.remove(jobName);
    }
  }

  /** @return new WorkflowRunInfo from properties in task. null, if task is null. */
  private WorkflowRunInfo createWorkflowRunInfo(final Task task) {
    if (task != null) {
      final Map<String, String> taskProperties = task.getProperties();
      return new WorkflowRunInfo(taskProperties.get(Task.PROPERTY_JOB_NAME),
        taskProperties.get(Task.PROPERTY_JOB_RUN_ID), taskProperties.get(Task.PROPERTY_WORKFLOW_RUN_ID));
    }
    return null;
  }

  /**
   * create taskmanager result description counters from performance counters.
   */
  protected Map<String, Number> getResultCounters(final String jobName, final BulkOutput... bulkDatas) {
    final Map<String, Number> counters = new HashMap<String, Number>();
    for (final BulkOutput bulk : bulkDatas) {
      if (bulk != null) {
        String prefix = null;
        if (bulk.getBulkType() == BulkType.ADD) {
          prefix = BulkbuilderConstants.BULK_BUILDER_INSERTED_RECORDS_SLOT;
        } else if (bulk.getBulkType() == BulkType.DEL) {
          prefix = BulkbuilderConstants.BULK_BUILDER_DELETED_RECORDS_SLOT;
        }
        addResultCounters(counters, bulk, prefix);
      }
    }
    return counters;
  }

  /**
   * add bulk counters to result description counters from performance counter.
   */
  protected void addResultCounters(final Map<String, Number> counters, final BulkOutput bulk, final String slotName) {
    final String durationPerformName = Counters.DURATION_PERFORM + ".output";
    final String prefix = "output." + slotName;
    if (bulk != null) {
      final IODataObject io = bulk.getOutput();
      final long durationOpen = io.getDurationOpen();
      if (durationOpen > 0) {
        Counters.addDuration(counters, Counters.DURATION_IODATA_OPEN, durationOpen);
      }
      final long durationPerform = io.getDurationPerform();
      if (durationPerform > 0) {
        Counters.addDuration(counters, durationPerformName, durationPerform);
        Counters.addDuration(counters, durationPerformName + "." + slotName, durationPerform);
      }
      Counters.addAll(counters, io.getCounter(), prefix);
      counters.put(prefix + ".duration", (System.currentTimeMillis() - bulk.getBulkStartTime())
        / MILLISECONDS_PER_SECOND);
    }
  }

  /** add duration.iodata counters. like in TaskContextImpl. */
  private void addIoCloseDuration(final Map<String, Number> counters, final long startTime) {
    Counters.addDuration(counters, Counters.DURATION_IODATA_CLOSE, System.nanoTime() - startTime);
    if (counters.containsKey(Counters.DURATION_IODATA_CLOSE)) {
      Counters.add(counters, Counters.DURATION_IODATA, counters.get(Counters.DURATION_IODATA_CLOSE).doubleValue());
    }
    if (counters.containsKey(Counters.DURATION_IODATA_OPEN)) {
      Counters.add(counters, Counters.DURATION_IODATA, counters.get(Counters.DURATION_IODATA_OPEN).doubleValue());
    }
  }

  /**
   * Handle shutdown for a bulk that is still active at shutdown time. This means we will log a warning and try to
   * commit the bulk.
   * 
   * @param bulkEntry
   */
  private void handleShutdownForBulk(final BulkOutput currBulk) {
    if (currBulk.getBulk() != null) {
      if (_log.isWarnEnabled()) {
        _log.warn("Deactivating service while bulk " + currBulk.getCurrentBulkId()
          + " is still open. Closing it without creating a task.");
      }
      try {
        currBulk.commit();
      } catch (final Exception ex) {
        _log.debug("Error committing bulk on shutdown", ex);
      }
    }
    releaseBulk(currBulk);
  }
}
