/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.helper;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;

/** Helper for creating and administrating micro bulks. */
public class MicroBulkbuilder {

  /** reserve this much memory for a microbulk. */
  private static final int MICROBULK_INIT_SIZE = (int) 1e6;

  /** for record<->BON serialization. */
  private final IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  /** key: micro bulk id, value: micro bulk. */
  private final Map<String, ByteArrayOutputStream> _microBulks =
    new ConcurrentHashMap<String, ByteArrayOutputStream>();

  /** key: micro bulk id, value: number of records. */
  private final Map<String, Integer> _recordCounts = new ConcurrentHashMap<String, Integer>();

  /** limits how many micro bulks can be processed in parallel. */
  private long _limitOfParallelBulks = -1; // default: unlimited

  /**
   * @param limitOfParallelBulks
   *          the maximum number of parallel micro bulks to process. (-1 = unlimited)
   */
  public MicroBulkbuilder(final long limitOfParallelBulks) {
    _limitOfParallelBulks = limitOfParallelBulks;
  }

  /**
   * add record to the micro bulk identified by the given id.
   * 
   * There's no need for synchronizing the whole method, we will not add records to a dedicated micro bulk in parallel.
   * So we only have to synch the creation of new micro bulks resp. the limit check.
   */
  public void addToMicroBulk(final String microBulkId, final Record record) throws BulkbuilderException {
    try {
      ByteArrayOutputStream microBulk = _microBulks.get(microBulkId);
      if (microBulk == null) {
        microBulk = initMicroBulk(microBulkId);
      }
      _ipcRecordWriter.writeBinaryStream(record, microBulk);
      addToRecordCountMap(microBulkId);
    } catch (final IOException e) {
      throw new BulkbuilderException("Error appending record to micro bulk", e);
    }
  }

  /**
   * This also removes the micro bulk and its counter entry.
   * 
   * @return the micro bulk that was finished with this call.
   */
  public byte[] finishMicroBulk(final String microBulkId) {
    _recordCounts.remove(microBulkId);
    final ByteArrayOutputStream microBulk = _microBulks.remove(microBulkId);
    if (microBulk == null) {
      return new byte[0];
    }
    return microBulk.toByteArray();
  }

  /**
   * Returns number of records for given micro bulk id.
   * 
   * @param microBulkId
   *          The micro bulk id
   */
  public Integer getNumberOfRecords(final String microBulkId) {
    final Integer value = _recordCounts.get(microBulkId);
    if (value == null) {
      return 0;
    }
    return value;
  }

  /**
   * initialize a new microbulk.
   * 
   * @throws BulkbuilderException
   *           if no more microbulks may be started currently.
   */
  private synchronized ByteArrayOutputStream initMicroBulk(final String microBulkId) throws BulkbuilderException {
    if (_limitOfParallelBulks > 0 && _microBulks.size() >= _limitOfParallelBulks) {
      throw new BulkbuilderException("No of parallel micro bulks exceeded configured limit "
        + _limitOfParallelBulks);
    }
    final ByteArrayOutputStream microBulk = new ByteArrayOutputStream(MICROBULK_INIT_SIZE);
    _microBulks.put(microBulkId, microBulk);
    _recordCounts.put(microBulkId, 0);
    return microBulk;
  }

  /**
   * Increases record counter for given micro bulk id.
   * 
   * @param microBulkId
   *          The micro bulk id
   */
  private void addToRecordCountMap(final String microBulkId) {
    final Integer value = _recordCounts.get(microBulkId);
    _recordCounts.put(microBulkId, value + 1);
  }
}
