/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.datamodel.ipc.test;

import java.io.ByteArrayInputStream;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.ipc.bon.BinaryToken;

/**
 * Test class for {@link IpcAnyReader} class.
 * 
 * @author aweber
 */
public class TestIpcAnyReader extends TestCase {

  private final IpcAnyReader _ipcAnyReader = new IpcAnyReader();

  /** JSON -> empty AnyMap. */
  public void testReadEmptyAnyMapFromJson() throws Exception {
    final Any expected = DataFactory.DEFAULT.createAnyMap();
    final String json = new String("{}");
    assertEquals(expected, _ipcAnyReader.readJsonObject(json));
  }

  /** JSON -> empty AnySeq. */
  public void testReadEmptyAnySeqFromJson() throws Exception {
    final Any expected = DataFactory.DEFAULT.createAnySeq();
    final String json = new String("[]");
    assertEquals(expected, _ipcAnyReader.readJsonObject(json));
  }

  /** JSON -> Value. */
  public void testReadSingleValueFromJson() throws Exception {
    final Any expected = DataFactory.DEFAULT.createStringValue("test");
    final String json = new String("\"test\""); // json parser seems to allow this, although this is no valid JSON
    assertEquals(expected, _ipcAnyReader.readJsonObject(json));
  }

  /** JSON -> Any, different tests. */
  public void testReadAnyFromJson() throws Exception {
    final ByteArrayInputStream mapInput =
      new ByteArrayInputStream("{\"key1\": \"value1\", \"key2\": \"value2\"}".getBytes());
    final Any map = _ipcAnyReader.readJsonStream(mapInput);
    assertTrue(map.isMap());
    assertEquals(2, map.size());
    assertEquals("value1", ((AnyMap) map).getStringValue("key1"));
    assertEquals("value2", ((AnyMap) map).getStringValue("key2"));

    final ByteArrayInputStream seqInput = new ByteArrayInputStream("[ \"value1\", \"value2\" ]".getBytes());
    final Any seq = _ipcAnyReader.readJsonStream(seqInput);
    assertTrue(seq.isSeq());
    assertEquals(2, seq.size());
    assertEquals("value1", ((AnySeq) seq).getStringValue(0));
    assertEquals("value2", ((AnySeq) seq).getStringValue(1));

    final ByteArrayInputStream strInput = new ByteArrayInputStream("\"value\"".getBytes());
    final Any strVal = _ipcAnyReader.readJsonStream(strInput);
    assertTrue(strVal.isString());
    assertEquals("value", ((Value) strVal).asString());

    final ByteArrayInputStream longInput = new ByteArrayInputStream("42".getBytes());
    final Any longVal = _ipcAnyReader.readJsonStream(longInput);
    assertTrue(longVal.isLong());
    assertEquals(42, ((Value) longVal).asLong().intValue());

    final ByteArrayInputStream doubleInput = new ByteArrayInputStream(Double.toString(Math.PI).getBytes());
    final Any doubleVal = _ipcAnyReader.readJsonStream(doubleInput);
    assertTrue(doubleVal.isDouble());
    assertEquals(Math.PI, ((Value) doubleVal).asDouble().doubleValue());

    final ByteArrayInputStream boolInput = new ByteArrayInputStream("true".getBytes());
    final Any boolVal = _ipcAnyReader.readJsonStream(boolInput);
    assertTrue(boolVal.isBoolean());
    assertTrue(((Value) boolVal).asBoolean());
  }

  /** BON -> AnyMap. */
  public void testReadEmptyAnyMapFromBon() throws Exception {
    final Any expected = DataFactory.DEFAULT.createAnyMap();
    final byte[] bon =
      new byte[] { BinaryToken.OBJECT_START.byteValue(), BinaryToken.MAPPING_START.byteValue(),
        BinaryToken.MAPPING_END.byteValue(), BinaryToken.OBJECT_END.byteValue() };
    assertEquals(expected, _ipcAnyReader.readBinaryObject(bon));
  }

  /** BON -> AnySeq. */
  public void testReadEmptyAnySeqFromBon() throws Exception {
    final Any expected = DataFactory.DEFAULT.createAnySeq();
    final byte[] bon =
      new byte[] { BinaryToken.OBJECT_START.byteValue(), BinaryToken.SEQUENCE_START.byteValue(),
        BinaryToken.SEQUENCE_END.byteValue(), BinaryToken.OBJECT_END.byteValue() };
    assertEquals(expected, _ipcAnyReader.readBinaryObject(bon));
  }

  /** BON -> Value. */
  public void testReadSingleValueFromBon() throws Exception {
    final Any expected = DataFactory.DEFAULT.createBooleanValue(false);
    final byte[] bon =
      new byte[] { BinaryToken.OBJECT_START.byteValue(), BinaryToken.SCALAR_BOOL_FALSE.byteValue(),
        BinaryToken.OBJECT_END.byteValue() };
    assertEquals(expected, _ipcAnyReader.readBinaryObject(bon));
  }
}
