/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.datamodel.util.test;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.util.AnyUtil;

/**
 * Test cases for {@link AnyUtil}.
 */
public class TestAnyUtil extends TestCase {

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
  }

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.util.AnyUtil#anyToNative(org.eclipse.smila.datamodel.Any)}.
   */
  @SuppressWarnings("unchecked")
  public void testAnyToNative() {
    final Date date = new Date();
    final AnySeq anySeq = DataFactory.DEFAULT.createAnySeq();
    anySeq.add(1);
    anySeq.add("string");
    anySeq.add(DataFactory.DEFAULT.createBooleanValue(true));
    anySeq.add(DataFactory.DEFAULT.createDateTimeValue(date));
    final AnyMap anyMap = DataFactory.DEFAULT.createAnyMap();
    anyMap.put("seq", anySeq);
    final AnyMap anyMap2 = DataFactory.DEFAULT.createAnyMap();
    anyMap2.put("string", "string");
    anyMap2.put("bool", true);
    anyMap.put("map", anyMap2);

    final Map<String, Object> map = (Map<String, Object>) AnyUtil.anyToNative(anyMap);
    assertEquals("string", ((Map<String, Object>) map.get("map")).get("string"));
    assertEquals(Boolean.TRUE, ((Map<String, Object>) map.get("map")).get("bool"));
    final List<Object> list = (List<Object>) map.get("seq");
    // it's a long...
    assertEquals(1L, list.get(0));
    assertEquals("string", list.get(1));
    assertEquals(Boolean.TRUE, list.get(2));
    assertEquals(date, list.get(3));
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.util.AnyUtil#objectToAny(java.lang.Object)}.
   */
  public void testObjectToAny() {
    final Date date = new Date();
    final Map<String, Object> map = new HashMap<String, Object>();
    map.put("1", 1L);
    map.put("date", date);
    final AnyMap anyMap = (AnyMap) AnyUtil.objectToAny(map);
    assertEquals(Long.valueOf(1), anyMap.getLongValue("1"));
    assertEquals(date, anyMap.getDateTimeValue("date"));
    final List<Object> list = new ArrayList<Object>();
    list.add(Boolean.TRUE);
    list.add("string");
    final AnySeq seq = (AnySeq) AnyUtil.objectToAny(list);
    assertEquals(Boolean.TRUE, seq.get(0).asValue().asBoolean());
    assertEquals("string", seq.get(1).asValue().asString());
    assertEquals(date, AnyUtil.objectToAny(date).asValue().asDateTime());
  }

  /**
   * Test method for
   * {@link org.eclipse.smila.datamodel.util.AnyUtil#saveGet(org.eclipse.smila.datamodel.Any, java.lang.String[])}.
   */
  public void testSaveGet() {
    final AnyMap anyMap = DataFactory.DEFAULT.createAnyMap();
    final AnyMap aMap = DataFactory.DEFAULT.createAnyMap();
    final AnyMap bMap = DataFactory.DEFAULT.createAnyMap();
    bMap.put("c", "value");
    aMap.put("b", bMap);
    anyMap.put("a", aMap);
    assertEquals("value", AnyUtil.saveGet(anyMap, new String[] { "a", "b", "c" }).asValue().asString());
    assertEquals("undef", AnyUtil.saveGet(anyMap, new String[] { "a", "b", "d" }).asValue().asString());
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.util.AnyUtil#exceptionToAny(java.lang.Throwable)}.
   */
  public void testExceptionToAny() {
    final Exception inner = new NullPointerException("bla");
    final Exception outer = new IllegalArgumentException("blub", inner);
    final AnyMap map = AnyUtil.exceptionToAny(outer);
    assertEquals(IllegalArgumentException.class.getName(), map.getStringValue("type"));
    assertEquals("blub", map.getStringValue("message"));
    assertEquals(NullPointerException.class.getName(), map.getMap("causedBy").getStringValue("type"));
    assertEquals("bla", map.getMap("causedBy").getStringValue("message"));
  }

}
