package org.eclipse.smila.datamodel.util.test;

import java.util.ArrayList;
import java.util.Arrays;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.util.RecordMerger;

/**
 * Test cases for {@link RecordMerger}.
 */
public class TestRecordMerger extends TestCase {

  /** test record merging with overwriting existing attributes. */
  public void testMergeRecords_overwriteAttributes() {
    final Record r1 = DataFactory.DEFAULT.createRecord("r1");
    r1.getMetadata().put("attrib1", "v1");
    r1.getMetadata().put("attrib2", "v2");
    final AnyMap map1 = DataFactory.DEFAULT.createAnyMap();
    map1.put("key1", "value1");
    r1.getMetadata().put("attribMap", map1);
    r1.setAttachment("attach1", new byte[0]);

    final Record r2 = DataFactory.DEFAULT.createRecord("r2");
    r2.getMetadata().put("attrib2", "v2_new");
    r2.getMetadata().put("attrib3", "v3");
    final AnyMap map1_new = DataFactory.DEFAULT.createAnyMap();
    map1_new.put("key2", "value2");
    r2.getMetadata().put("attribMap", map1_new);
    r2.setAttachment("attach2", new byte[0]);

    RecordMerger.mergeRecords(r1, r2, false); // false -> overwrite

    // check record r1 (to which r2 is merged)
    assertEquals("r1", r1.getId()); // record id is kept!
    assertEquals("v1", r1.getMetadata().get("attrib1").toString());
    assertEquals("v2_new", r1.getMetadata().get("attrib2").toString());
    assertEquals("v3", r1.getMetadata().get("attrib3").toString());
    assertEquals(Arrays.asList("key2"), new ArrayList<String>(r1.getMetadata().getMap("attribMap").keySet()));
    assertTrue(r1.hasAttachment("attach1"));
    assertTrue(r1.hasAttachment("attach2"));

    // check record r2 which should be unchanged
    assertEquals("r2", r2.getId());
    assertFalse(r2.getMetadata().containsKey("attrib1"));
    assertEquals("v2_new", r2.getMetadata().get("attrib2").toString());
    assertEquals("v3", r2.getMetadata().get("attrib3").toString());
    assertEquals(Arrays.asList("key2"), new ArrayList<String>(r1.getMetadata().getMap("attribMap").keySet()));
    assertFalse(r2.hasAttachment("attach1"));
    assertTrue(r2.hasAttachment("attach2"));
  }

  /** test record merging with merging existing attributes. */
  public void testMergeRecords_mergeAttributes() {
    final Record r1 = DataFactory.DEFAULT.createRecord("r1");
    r1.getMetadata().put("attrib1", "v1");
    r1.getMetadata().put("attrib2", "v2");
    final AnySeq v3 = DataFactory.DEFAULT.createAnySeq();
    v3.add("v3_a");
    v3.add("v3_b");
    r1.getMetadata().put("attrib3", v3);
    r1.getMetadata().put("attrib4", "v4");
    final AnyMap map1 = DataFactory.DEFAULT.createAnyMap();
    map1.put("key1", "value1");
    r1.getMetadata().put("attribMap", map1);
    r1.setAttachment("attach1", "attachment".getBytes());

    final Record r2 = DataFactory.DEFAULT.createRecord("r2");
    r2.getMetadata().put("attrib2", "v2_new");
    final AnySeq v3_new = DataFactory.DEFAULT.createAnySeq();
    v3_new.add("v3_c");
    v3_new.add("v3_d");
    r2.getMetadata().put("attrib3", v3_new);
    final AnySeq v4_new = DataFactory.DEFAULT.createAnySeq();
    v4_new.add("v4_a");
    v4_new.add("v4_b");
    final AnyMap map1_new = DataFactory.DEFAULT.createAnyMap();
    map1_new.put("key2", "value2");
    r2.getMetadata().put("attribMap", map1_new);
    r2.getMetadata().put("attrib4", v4_new);
    r2.setAttachment("attach1", "attachment_new".getBytes());

    RecordMerger.mergeRecords(r1, r2, true); // true -> merge

    // check record r1 (to which r2 is merged)
    assertEquals("r1", r1.getId()); // record id is kept!
    assertEquals("v1", r1.getMetadata().get("attrib1").toString());
    final AnySeq v2_merged = DataFactory.DEFAULT.createAnySeq();
    v2_merged.add("v2");
    v2_merged.add("v2_new");
    assertEquals(v2_merged, r1.getMetadata().get("attrib2"));
    final AnySeq v3_merged = DataFactory.DEFAULT.createAnySeq();
    v3_merged.add("v3_a");
    v3_merged.add("v3_b");
    v3_merged.add("v3_c");
    v3_merged.add("v3_d");
    assertEquals(v3_merged, r1.getMetadata().get("attrib3"));
    final AnySeq v4_merged = DataFactory.DEFAULT.createAnySeq();
    v4_merged.add("v4");
    v4_merged.add("v4_a");
    v4_merged.add("v4_b");
    assertEquals(v3_merged, r1.getMetadata().get("attrib3"));
    assertEquals(Arrays.asList("key1", "key2"),
      new ArrayList<String>(r1.getMetadata().getMap("attribMap").keySet()));
    // attachments will always be overwritten:
    assertEquals("attachment_new", new String(r1.getAttachmentAsBytes("attach1")));

    // check record r2 which should be unchanged
    assertEquals("r2", r2.getId());
    assertFalse(r2.getMetadata().containsKey("attrib1"));
    assertEquals("v2_new", r2.getMetadata().get("attrib2").toString());
    assertEquals(v3_new, r2.getMetadata().get("attrib3"));
    assertEquals("attachment_new", new String(r1.getAttachmentAsBytes("attach1")));
  }

}
