/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Thomas Menzel (brox IT Solution GmbH) - initial creator
 *******************************************************************************/
package org.eclipse.smila.datamodel.util;

import static org.apache.commons.lang.StringUtils.join;

import java.util.ArrayList;
import java.util.NoSuchElementException;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Value;

/**
 * @author tmenzel
 * 
 */
public final class MetadataUtils {

  /** The logger. */
  private static final Log LOG = LogFactory.getLog(MetadataUtils.class);

  /**
   * controls the behavior of methods.
   */
  public enum Mode {

    // /** all missing elements but the last are created. */
    // CREATE_PARENTS,
    /** all missing elements including the last are created. */
    CREATE_ALL,
    /** nothing is created and a null is returned. */
    CREATE_NONE,
    /** if one of the path elements is missing an NoSuchElementException is raised. */
    FAIL

  }

  /**
   * prevent instantiation.
   */
  private MetadataUtils() {
  }

  /**
   * returns the requested Any object.
   * 
   * @param metadata
   *          the metadata, must be either a SEQ or MAP
   * @param mode
   *          the create parents
   * @param path
   *          the path. may be distinct elements or whole segments. elements are separated by a /. an element is used as
   *          an index into either a map or sequence. in the latter case it must evaluate to a number, otherwise a
   *          {@link IllegalArgumentException} is thrown
   */
  public static Any getAny(final Any metadata, final Mode mode, final String... path) {
    /* PERF: could be made faster with a tokenizer/iterator | TM @ Apr 27, 2011 */
    final ArrayList<String> pathElements = new ArrayList<String>(Math.max(path.length, 16));
    for (final String pathSegment : path) {
      final String[] split = StringUtils.split(pathSegment, '/');
      CollectionUtils.addAll(pathElements, split);
    } // for

    final Any any = getAny(metadata, mode, pathElements, 0);
    return any;

  }

  /**
   * Gets the map.
   * 
   * @param createAll
   *          false -> NONE, true -> ALL
   * @deprecated as of 1.1, use {@link #getMap(AnyMap, Mode, String...)}
   * 
   */
  public static AnyMap getMap(final AnyMap metadata, final boolean createAll, final String... path) {
    final Any any = getAny(metadata, createAll ? Mode.CREATE_ALL : Mode.CREATE_NONE, path);
    return any == null ? null : any.asMap();
  }

  /**
   * Gets the map.
   */
  public static AnyMap getMap(final AnyMap metadata, final Mode mode, final String... path) {
    final Any any = getAny(metadata, mode, path);
    return AnyUtil.asMap(any);
  }

  /**
   * Gets the Seq.
   */
  public static AnySeq getSeq(final AnyMap metadata, final Mode mode, final String... path) {
    final Any any = getAny(metadata, mode, path);
    return AnyUtil.asSeq(any);
  }

  public static Value getValue(final AnyMap metadata, final Mode mode, final String... path) {
    final Any any = getAny(metadata, mode, path);
    return (Value) (any == null ? null : any);
  }

  /**
   * 
   */
  private static boolean flags2bool(final Mode mode) {
    final boolean create;
    switch (mode) {
      case CREATE_NONE:
      case FAIL:
        create = false;
        // case PARENTS:
        // create
        break;
      case CREATE_ALL:
        create = true;
        break;
      default:
        throw new NotImplementedException("unknown create flag: " + mode);
    }
    return create;
  }

  /**
   * Gets the any.
   */
  private static Any getAny(final Any parent, final Mode mode, final ArrayList<String> pathElements, int index) {

    // recursion stop
    if (index >= pathElements.size() || parent == null) {
      return parent;
    }

    final String keyOrIndex = pathElements.get(index);
    final boolean create = flags2bool(mode);

    final Any child;
    if (parent.isMap()) {
      /*
       * BETTER: only maps are created on the fly. extend this for sequences in case the next element is a number| TM @
       * Apr 27, 2011
       */
      if (create) {
        child = parent.asMap().getMap(keyOrIndex, create);
      } else {
        child = parent.asMap().get(keyOrIndex);
      }
    } else if (parent.isSeq()) {
      child = parent.asSeq().get(Integer.parseInt(keyOrIndex));
    } else {
      final String subPath = getSubPath(pathElements, index);
      throw new InvalidValueTypeException("expected Any to be one of SEQ or MAP at: " + subPath);
    }

    if (child == null) {
      if (LOG.isTraceEnabled()) {
        LOG.trace("missing element: " + getSubPath(pathElements, index));
      } // if
      if (mode == Mode.FAIL) {
        throw new NoSuchElementException(join(pathElements, '/'));
      }
    }

    // recursion
    return getAny(child, mode, pathElements, ++index);

  }

  /**
   * Gets the sub path.
   */
  private static String getSubPath(final ArrayList<String> pathElements, final int index) {
    return StringUtils.join(pathElements.subList(0, index), '/');
  }

}
