/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.test.failover;

import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.util.Arrays;
import java.util.List;
import java.util.Vector;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpStatus;
import org.apache.http.NoHttpResponseException;
import org.eclipse.smila.http.client.HttpMethod;
import org.eclipse.smila.http.client.RestException;
import org.eclipse.smila.http.client.impl.failover.FailoverRestClient;
import org.eclipse.smila.http.client.test.TestDefaultRestClient;
import org.eclipse.smila.http.client.test.httpserver.MultiHttpServerService;
import org.eclipse.smila.http.server.HttpService;

/** unit tests for the SMILA failover REST client. */
public class TestFailoverRestClientWithFailover extends TestDefaultRestClient {

  /**
   * the ports for the http servers. Note: 8080 is already started by a service and cannot be closed
   */
  private final List<Integer> _portNumbers = Arrays.asList(8088, 8089, 8888);

  /** out multi http server service to start and stop http servers on various ports. */
  private MultiHttpServerService _multiHttpServerService;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    _client = new FailoverRestClient("localhost:8088", "localhost:8089", "localhost:8888");
    getService(HttpService.class);
    _multiHttpServerService = getService(MultiHttpServerService.class);
    // we start only the last server to have the client fail over to the last one.
    _multiHttpServerService.start(_portNumbers.get(_portNumbers.size() - 1));
    Thread.sleep(1000);
  }

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    _client.shutdown();
  }

  /** some additional tests for circular failover if all servers fail... */
  public void testCompleteFailOverForPost() throws Exception {
    // check two complete circles...
    for (int run = 0; run < 2; run++) {
      // first start all
      for (final Integer portNo : _portNumbers) {
        _multiHttpServerService.start(portNo);
      }
      for (int i = 0; i < _portNumbers.size(); i++) {
        _multiHttpServerService.stop(_portNumbers.get(i));
        // if we just shut down the last client, start the first one.
        if (i == _portNumbers.size() - 1) {
          _multiHttpServerService.start(_portNumbers.get(0));
        }
        testPost();
      }
    }
  }

  /** some additional tests for circular failover if only the current client fails... */
  public void testCircularFailOverForPost() throws Exception {
    // check two complete circles...
    for (int run = 0; run < 2; run++) {
      // first start all
      for (final Integer portNo : _portNumbers) {
        _multiHttpServerService.start(portNo);
      }
      for (int i = 0; i < _portNumbers.size(); i++) {
        _multiHttpServerService.stop(_portNumbers.get(i));
        // if we just shut down the last client, start the first one.
        if (i == _portNumbers.size() - 1) {
          _multiHttpServerService.start(_portNumbers.get(0));
        }
        testPost();
        // restart the last client.
        _multiHttpServerService.start(_portNumbers.get(i));
      }
    }
  }

  /** some additional tests for the case when all nodes fail and get restarted. */
  public void testAllShutDownAndRestartOne() throws Exception {
    // first stop all
    for (final Integer portNo : _portNumbers) {
      _multiHttpServerService.stop(portNo);
    }
    try {
      _client.post(_mockPath);
      fail("Exception expected.");
    } catch (final Exception e) {
      assertTrue("NoHttpResponseException expected but was " + e, e instanceof NoHttpResponseException);
    }
    // restart the last client.
    _multiHttpServerService.start(_portNumbers.get(_portNumbers.size() - 1));
    // and redo the test
    testPost();
  }

  /** check failover with parallel requests. */
  public void testMultiThreadingWithFailover() throws Exception {
    for (final Integer portNo : _portNumbers) {
      _multiHttpServerService.start(portNo);
    }
    final int noOfUsers = 4;
    final ExecutorService executor = Executors.newFixedThreadPool(noOfUsers);
    final Vector<Throwable> errors = new Vector<Throwable>();
    try {
      for (int i = 0; i < noOfUsers; i++) {
        executor.submit(new Callable<Void>() {
          @Override
          public Void call() throws Exception {
            try {
              while (!Thread.interrupted()) {
                testPost();
              }
            } catch (final RestException ex) {
              if (ex.getResponseCode() != HttpStatus.SC_NOT_FOUND) {
                // 404 are OK: the client did failover, but the server didn't answer correctly.
                // probably happens during restart of server.
                errors.add(ex);
              }
            } catch (final Throwable ex) {
              errors.add(ex);
            }
            return null;
          }
        });
      }
      for (final Integer portNo : _portNumbers) {
        _multiHttpServerService.stop(portNo);
        Thread.sleep(500);
        _multiHttpServerService.start(portNo);
        Thread.sleep(500);
      }
    } finally {
      executor.shutdownNow();
      executor.awaitTermination(5, TimeUnit.SECONDS);
    }
    assertTrue(errors.toString(), errors.isEmpty());
  }

  /** test retry of a stream request when errors occurs before sending the stream. */
  public void testFailoverWithStreamRequests() throws Exception {
    for (final Integer portNo : _portNumbers) {
      _multiHttpServerService.start(portNo);
    }
    final Integer failPort = _portNumbers.get(0);
    testPostRequestFromStream();
    assertTrue(_client.getHostAndPort().endsWith(failPort.toString()));
    _multiHttpServerService.stop(failPort);
    testPostRequestFromStream();
    assertFalse(_client.getHostAndPort().endsWith(failPort.toString()));
  }

  /** test retry of a stream request when errors occurs before sending the stream. */
  public void testFailoverFailsWithStreamRequestsAfterSending() throws Exception {
    for (final Integer portNo : _portNumbers) {
      _multiHttpServerService.start(portNo);
    }
    final Integer failPort = _portNumbers.get(0);
    try {
      // create a custom stream that stops the contacted server as soon as the client tries to read the stream.
      _client.invoke(HttpMethod.POST, _mockPath, new InputStream() {
        @Override
        public int read() throws IOException {
          try {
            _multiHttpServerService.stop(failPort);
          } catch (final Exception ex) {
            throw new IOException(ex);
          }
          return 'a';
        }
      }, null);
      fail("should not work");
    } catch (final ConnectException ex) {
      ex.printStackTrace();
    }
    // but failover should have been initiated.
    assertFalse(_client.getHostAndPort().endsWith(failPort.toString()));
  }

}
