/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.test.httphandler;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;

import org.eclipse.jetty.http.HttpException;
import org.eclipse.jetty.http.HttpStatus;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.http.server.json.JsonRequestHandler;

/** A simple handler for testing the REST client. */
public class MockHandler extends JsonRequestHandler {

  public static final String SUCCESSSTATUS = "successstatus";

  public static final String ERRORREASON = "errorreason";

  public static final String ERRORSTATUS = "errorstatus";

  public static final String NORESPONSE = "noResponse";

  public static final String BULKSIZE = "bulksize";

  private final IpcAnyWriter _jsonWriter = new IpcAnyWriter(false);

  /** accept all methods. */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return true;
  }

  @Override
  public Object process(final String method, final String requestUri, final Record request) throws Exception {
    final AnyMap response = FACTORY.createAnyMap();
    if (request != null) {
      if (request.getMetadata().containsKey(NORESPONSE)) {
        return null;
      }
      if (request.getMetadata().containsKey(ERRORSTATUS)) {
        final int requestedStatus = request.getMetadata().getLongValue(ERRORSTATUS).intValue();
        String reason = request.getMetadata().getStringValue(ERRORREASON);
        if (reason == null) {
          reason = "Just for fun.";
        }
        throw new HttpException(requestedStatus, reason);
      }
      response.putAll(request.getMetadata());
      if (request.hasAttachments()) {
        for (final Iterator<String> names = request.getAttachmentNames(); names.hasNext();) {
          final String attachmentName = names.next();
          final String attachmentValue = new String(request.getAttachmentAsBytes(attachmentName), "utf-8");
          response.put(attachmentName, attachmentValue);
        }
      }
    }
    response.put("method", method);
    response.put("uri", requestUri);
    return response;
  }

  @Override
  protected void writeResultObject(final OutputStream responseStream, final Object resultObject) throws IOException {
    final AnyMap result = (AnyMap) resultObject;
    if (result != null && result.containsKey(BULKSIZE)) {
      final int bulkSize = result.getLongValue(BULKSIZE).intValue();
      for (int i = 0; i < bulkSize; i++) {
        result.put("bulkposition", i);
        _jsonWriter.writeJsonStream(result, responseStream);
        responseStream.write('\n');
      }
    } else {
      super.writeResultObject(responseStream, resultObject);
    }
  }

  @Override
  protected int getSuccessStatus(final String requestMethod, final String requestUri, final Record request,
    final Object response) {
    if (request != null && request.getMetadata().containsKey(SUCCESSSTATUS)) {
      return request.getMetadata().getLongValue(SUCCESSSTATUS).intValue();
    }
    return HttpStatus.OK_200;
  }
}
