/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.test;

import java.io.File;
import java.io.InputStream;
import java.util.Collection;
import java.util.UUID;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.ImportingException;
import org.eclipse.smila.importing.crawler.file.FileCrawlerService;
import org.eclipse.smila.importing.util.PropertyNameMapper;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.DefaultTaskLogFactory;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.TaskLog;
import org.eclipse.smila.taskworker.internal.TaskContextImpl;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

public class TestFileCrawlerService extends DeclarativeServiceTestCase {

  /** dummy logger. */
  private final TaskLog _taskLog = new DefaultTaskLogFactory().getTaskLog(null);

  private final PropertyNameMapper _mapper;

  /** dummy TaskContext */
  private final TaskContext _taskContext;

  private File _rootFolder;

  public TestFileCrawlerService() {
    final AnyMap mapping = initMapping();
    final Task task = new Task(UUID.randomUUID().toString(), "dummy");
    task.getParameters().put(ImportingConstants.TASK_PARAM_MAPPING, mapping);
    _taskContext = new TaskContextImpl(task, _taskLog, null);
    _mapper = new PropertyNameMapper(mapping);
  }

  @Override
  public void setUp() throws Exception {
    _rootFolder = FileystemHelper.initTestFilesystem();
  }

  @Override
  public void tearDown() throws Exception {
    if (_rootFolder != null) {
      FileUtils.deleteDirectory(_rootFolder);
    }
  }

  public void testService() throws Exception {
    getFileCrawlerService();
  }

  public void testListFiles() throws Exception {
    final FileCrawlerService fileCrawler = getFileCrawlerService();
    final Collection<File> files = fileCrawler.listFiles(_rootFolder);
    assertNotNull(files);
    assertEquals(2, files.size());
    assertTrue(files.contains(new File(_rootFolder, "c")));
    assertTrue(files.contains(new File(_rootFolder, "d.e")));
  }

  public void testListDirectories() throws Exception {
    final FileCrawlerService fileCrawler = getFileCrawlerService();
    final Collection<File> dirs = fileCrawler.listDirectories(_rootFolder);
    assertNotNull(dirs);
    assertEquals(2, dirs.size());
    assertTrue(dirs.contains(new File(_rootFolder, "a")));
    assertTrue(dirs.contains(new File(_rootFolder, "b")));

    final Collection<File> subDirs = fileCrawler.listDirectories(new File(_rootFolder, "a"));
    assertNotNull(subDirs);
    assertEquals(2, subDirs.size());
    assertTrue(subDirs.contains(new File(_rootFolder, "a/aa")));
    assertTrue(subDirs.contains(new File(_rootFolder, "a/ab")));
  }

  public void testList() throws Exception {
    final FileCrawlerService fileCrawler = getFileCrawlerService();
    final Collection<File> filesAndFolders = fileCrawler.list(_rootFolder);
    assertNotNull(filesAndFolders);
    assertEquals(4, filesAndFolders.size());
  }

  public void testGetContent() throws Exception {
    final FileCrawlerService fileCrawler = getFileCrawlerService();
    final Record record = DataFactory.DEFAULT.createRecord();
    record.getMetadata().put(_mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0),
      new File(_rootFolder, "c").getAbsolutePath());
    final InputStream contentStream = fileCrawler.getContent(record, _taskContext);
    try {
      assertEquals(-1, contentStream.read());
    } finally {
      IOUtils.closeQuietly(contentStream);
    }
  }

  public void testDontGetContentOfDirectory() throws Exception {
    final FileCrawlerService fileCrawler = getFileCrawlerService();
    final Record record = DataFactory.DEFAULT.createRecord();
    record.getMetadata().put(_mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0),
      new File(_rootFolder, "a").getAbsolutePath());
    try {
      fileCrawler.getContent(record, _taskContext);
      fail("should not work");
    } catch (final ImportingException ex) {
      assertFalse(ex.isRecoverable());
    }
  }

  private FileCrawlerService getFileCrawlerService() throws Exception {
    final FileCrawlerService fileCrawler = getService(FileCrawlerService.class);
    assertNotNull(fileCrawler);
    return fileCrawler;
  }

  private AnyMap initMapping() {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    map.put(FileCrawlerService.PROPERTY_FILE_EXTENSION, "my-file-extension");
    map.put(FileCrawlerService.PROPERTY_FILE_FOLDER, "my-file-folder");
    map.put(FileCrawlerService.PROPERTY_FILE_LAST_MODIFIED, "my-file-last-modified");
    map.put(FileCrawlerService.PROPERTY_FILE_NAME, "my-file-name");
    map.put(FileCrawlerService.PROPERTY_FILE_PATH, "my-file-path");
    map.put(FileCrawlerService.PROPERTY_FILE_SIZE, "my-file-size");
    map.put(FileCrawlerService.ATTACHMENT_FILE_CONTENT, "my-file-content");
    return map;
  }
}
