/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.internal;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.ImportingException;
import org.eclipse.smila.importing.crawler.file.FileCrawlerService;
import org.eclipse.smila.importing.util.PropertyNameMapper;
import org.eclipse.smila.taskworker.TaskContext;

/**
 * Default implementation of a FileCrawlerService.
 * 
 * @author stuc07
 * 
 */
public class FileCrawlerServiceImpl implements FileCrawlerService {

  private final Set<String> _filePropertyNames = new HashSet<String>();

  public FileCrawlerServiceImpl() {
    _filePropertyNames.add(PROPERTY_FILE_NAME);
    _filePropertyNames.add(PROPERTY_FILE_PATH);
    _filePropertyNames.add(PROPERTY_FILE_FOLDER);
    _filePropertyNames.add(PROPERTY_FILE_SIZE);
    _filePropertyNames.add(PROPERTY_FILE_LAST_MODIFIED);
    _filePropertyNames.add(PROPERTY_FILE_EXTENSION);
    _filePropertyNames.add(ATTACHMENT_FILE_CONTENT);
  }

  @Override
  public Collection<File> listFiles(final File directory) {
    final File[] fileList = directory.listFiles((FileFilter) FileFileFilter.FILE);
    if (fileList != null) {
      return Arrays.asList(fileList);
    }
    return new ArrayList<File>();
  }

  @Override
  public Collection<File> listDirectories(final File directory) {
    final File[] fileList = directory.listFiles((FileFilter) DirectoryFileFilter.DIRECTORY);
    if (fileList != null) {
      return Arrays.asList(fileList);
    }
    return new ArrayList<File>();
  }

  @Override
  public Collection<File> list(final File directory) {
    final File[] fileList = directory.listFiles();
    if (fileList != null) {
      return Arrays.asList(fileList);
    }
    return new ArrayList<File>();
  }

  @Override
  public Record fileToRecord(final File file, final String dataSource, final boolean withContent)
    throws IOException {
    final FileToRecordConverter converter = new FileToRecordConverter(DataFactory.DEFAULT);
    return converter.fileToRecord(file, dataSource, withContent);
  }

  @Override
  public void addAttachment(final Record record, final String pathAttribute, final String attachmentName)
    throws IOException {
    final FileToRecordConverter converter = new FileToRecordConverter(DataFactory.DEFAULT);
    converter.addAttachment(record, pathAttribute, attachmentName);
  }

  @Override
  public Record dirToRecord(final File directory, final String dataSource) {
    final FileToRecordConverter converter = new FileToRecordConverter(DataFactory.DEFAULT);
    return converter.dirToRecord(directory, dataSource);
  }

  @Override
  public InputStream getContent(final Record record, final TaskContext taskContext) throws ImportingException {
    final PropertyNameMapper mapper = PropertyNameMapper.createFrom(taskContext);
    final String path =
      record.getMetadata().getStringValue(mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0));
    if (path == null) {
      throw new IllegalArgumentException("Record '" + record.getId() + "' does not contain attribute '"
        + mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0) + "'");
    }
    final File file = new File(path);
    if (!file.exists()) {
      throw new ImportingException("File '" + file + "' does not exist.");
    }
    if (!file.canRead()) {
      throw new ImportingException("File '" + file + "' is not readable.");
    }
    if (!file.isFile()) {
      throw new ImportingException("File '" + file + "' is not a file with content.");
    }
    try {
      return new FileInputStream(file);
    } catch (final FileNotFoundException ex) {
      throw new ImportingException("File '" + file + "' does not exist or is not readable.", ex, false);
    }
  }

  @Override
  public Set<String> getFilePropertyNames() {
    return _filePropertyNames;
  }

}
