package org.eclipse.smila.importing.crawler.jdbc.test;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.importing.crawler.jdbc.DbAccessService;
import org.eclipse.smila.importing.crawler.jdbc.SqlExecutor;
import org.eclipse.smila.importing.util.MessageCollector;

/** Test for {@link DbAccessService} class. */
public class TestDbAccessService extends JdbcCrawlerTestBase {

  private static final int NO_OF_ROWS = 5;

  private static final MessageCollector IGNORE = new MessageCollector.Ignore();

  @Override
  protected int getNumberOfRowsToCreate() {
    return NO_OF_ROWS;
  }

  /** tests OSGI service. */
  public void testService() throws Exception {
    final DbAccessService service = getService(DbAccessService.class);
    assertNotNull(service);
  }

  /** tests executeSQL() method. */
  public void testExecuteSql() throws Exception {
    final DbAccessService dbAccess = getService(DbAccessService.class);
    final String url = getConnectUrl();
    final Properties props = getProperties();
    final String sql = "SELECT * FROM " + DB_TABLE_NAME;

    final Collection<Record> result = dbAccess.executeSql(url, props, sql, Long.MAX_VALUE, IGNORE);
    assertEquals(NO_OF_ROWS, result.size());
  }

  /** tests executePrepared() method with single parameter for PreparedStatement. */
  public void testExecutePreparedWithSingleParam() throws Exception {
    final DbAccessService dbAccess = getService(DbAccessService.class);
    final String url = getConnectUrl();
    final Properties props = getProperties();
    final String testColumn = "INT_VAL";
    final String sql = "SELECT * FROM " + DB_TABLE_NAME + " WHERE " + testColumn + " = ?";

    try (SqlExecutor sqlExecutor = dbAccess.executePrepared(url, props, sql, Long.MAX_VALUE, IGNORE)) {
      final Record r1 = sqlExecutor.execute(Arrays.asList(DataFactory.DEFAULT.createLongValue(2)));
      assertEquals(new Long(2), r1.getMetadata().getLongValue(testColumn));

      final Record r2 = sqlExecutor.execute(Arrays.asList(DataFactory.DEFAULT.createLongValue(3)));
      assertEquals(new Long(3), r2.getMetadata().getLongValue(testColumn));
    }
  }

  /** tests executePrepared() method with multiple parameters for PreparedStatement. */
  public void testExecutePreparedWithMultipleParams() throws Exception {
    final DbAccessService dbAccess = getService(DbAccessService.class);
    final String url = getConnectUrl();
    final Properties props = getProperties();
    final String testColumn1 = "INT_VAL";
    final String testColumn2 = "BIGINT_VAL";
    final String sql =
      "SELECT * FROM " + DB_TABLE_NAME + " WHERE " + testColumn1 + "= ? AND " + testColumn2 + "= ?";

    try (SqlExecutor sqlExecutor = dbAccess.executePrepared(url, props, sql, Long.MAX_VALUE, IGNORE)) {
      final List<Value> params1 = new ArrayList<>();
      params1.add(DataFactory.DEFAULT.createLongValue(2));
      params1.add(DataFactory.DEFAULT.createLongValue(4));
      final Record r1 = sqlExecutor.execute(params1);
      assertEquals(new Long(2), r1.getMetadata().getLongValue(testColumn1));
      assertEquals(new Long(4), r1.getMetadata().getLongValue(testColumn2));

      final List<Value> params2 = new ArrayList<>();
      params2.add(DataFactory.DEFAULT.createLongValue(3));
      params2.add(DataFactory.DEFAULT.createLongValue(6));
      final Record r2 = sqlExecutor.execute(params2);
      assertEquals(new Long(3), r2.getMetadata().getLongValue(testColumn1));
      assertEquals(new Long(6), r2.getMetadata().getLongValue(testColumn2));
    }
  }

  public void testExecutePreparedWithNullParam() throws Exception {
    final DbAccessService dbAccess = getService(DbAccessService.class);
    final String url = getConnectUrl();
    final Properties props = getProperties();
    final String testColumn = "INT_VAL";
    final String sql = "SELECT * FROM " + DB_TABLE_NAME + " WHERE " + testColumn + " = ?";

    try (SqlExecutor sqlExecutor = dbAccess.executePrepared(url, props, sql, Long.MAX_VALUE, IGNORE)) {
      try {
        sqlExecutor.execute(Arrays.asList((Value) null));
        fail("should not work.");
      } catch (final IllegalArgumentException ex) {
        ; // expected exception.
      }
    }
  }

}
