package org.eclipse.smila.importing.crawler.jdbc;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Properties;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.util.MessageCollector;

public interface DbAccessService {

  /**
   * Executes single SQL statement.
   * 
   * @param url
   *          the database connection URL of the form jdbc:subprotocol:subname
   * @param dbProperties
   *          connection properties, e.g. "user" and "password"
   * @param sql
   *          the SQL statement to execute
   * @return the SQL result as collection of records, one row -> one record
   */
  Collection<Record> executeSql(final String url, final AnyMap dbProperties, final String sql,
    final long maxAttachmentSize, MessageCollector messages) throws SQLException, IOException;

  /**
   * @see DbAccessService.executeSql(String, AnyMap, String), we just use Properties instead of AnyMap here
   */
  Collection<Record> executeSql(final String url, final Properties dbProperties, final String sql,
    final long maxAttachmentSize, MessageCollector messages) throws SQLException, IOException;

  /**
   * For executing an SQL statement multiple times with different parameters. The given SQL statement is used as
   * PreparedStatement. An SqlExecutor is returned which allows multiple executions of the same statement with different
   * parameters.
   * 
   * @param url
   *          the database connection URL of the form jdbc:subprotocol:subname
   * @param dbProperties
   *          connection properties, e.g. "user" and "password"
   * @param sql
   *          the SQL statement to execute, used as PreparedStatement
   * @return An SqlExecutor for multiple execution of the given sql statement with different parameters.
   */
  SqlExecutor executePrepared(final String url, final AnyMap dbProperties, final String sql,
    final long maxAttachmentSize, MessageCollector messages) throws SQLException;

  /**
   * @see DbAccessService.executePrepared(String, AnyMap, String), we just use Properties instead of AnyMap here
   */
  SqlExecutor executePrepared(final String url, final Properties dbProperties, final String sql,
    final long maxAttachmentSize, MessageCollector messages) throws SQLException;

}
