package org.eclipse.smila.importing.crawler.jdbc;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.importing.util.CrawlingContext;
import org.eclipse.smila.importing.util.MessageCollector;
import org.eclipse.smila.taskworker.TaskContext;

/** contains all parameters needed to control the crawl process. */
public final class JdbcCrawlingContext extends CrawlingContext {

  private final long _maxRecordsPerBulk;

  private final long _maxAttachmentSize;

  private final String _dbUrl;

  private final AnyMap _dbPropertiess;

  private final String _crawlSql;

  private Any _idColumns;

  private final Any _deltaColumns;

  private final MessageCollector _messages;

  public JdbcCrawlingContext(final TaskContext taskContext) {
    super(taskContext, true);
    final AnyMap taskParameters = taskContext.getTaskParameters();
    _maxRecordsPerBulk =
      getNonNegativeParameter(taskParameters, JdbcCrawlerWorker.TASK_PARAM_MAX_RECORDS_PER_BULK,
        JdbcCrawlerWorker.MAX_RECORDS_PER_BULK_DEFAULT);
    _maxAttachmentSize =
      getNonNegativeParameter(taskParameters, JdbcCrawlerWorker.TASK_PARAM_MAX_ATTACHMENT_SIZE,
        JdbcCrawlerWorker.MAX_ATTACHMENT_SIZE_DEFAULT);
    if (_maxRecordsPerBulk <= 0) {
      throw new IllegalArgumentException("Task parameter " + JdbcCrawlerWorker.TASK_PARAM_MAX_RECORDS_PER_BULK
        + " must be greater than 0");
    }
    if (_maxAttachmentSize <= 0) {
      throw new IllegalArgumentException("Task parameter " + JdbcCrawlerWorker.TASK_PARAM_MAX_ATTACHMENT_SIZE
        + " must be greater than 0");
    }
    _dbUrl = taskParameters.getStringValue(JdbcCrawlerWorker.TASK_PARAM_DB_URL);
    if (_dbUrl == null || _dbUrl.trim().length() == 0) {
      throw new IllegalArgumentException("Parameter '" + JdbcCrawlerWorker.TASK_PARAM_DB_URL + "' of task "
        + taskContext.getTask().getTaskId() + " is null or empty");
    }
    _dbPropertiess = taskParameters.getMap(JdbcCrawlerWorker.TASK_PARAM_DB_PROPS);
    if (_dbPropertiess == null || _dbPropertiess.isEmpty()) {
      throw new IllegalArgumentException("Parameter '" + JdbcCrawlerWorker.TASK_PARAM_DB_PROPS + "' of task "
        + taskContext.getTask().getTaskId() + " is null or empty");
    }
    _crawlSql = taskParameters.getStringValue(JdbcCrawlerWorker.TASK_PARAM_CRAWL_SQL);
    if (_crawlSql == null || _crawlSql.trim().length() == 0) {
      throw new IllegalArgumentException("Parameter '" + JdbcCrawlerWorker.TASK_PARAM_CRAWL_SQL + "' of task "
        + taskContext.getTask().getTaskId() + " is null or empty");
    }
    _idColumns = taskParameters.get(JdbcCrawlerWorker.TASK_PARAM_ID_COLUMNS);
    if (_idColumns == null || _idColumns.isEmpty()) {
      throw new IllegalArgumentException("Parameter '" + JdbcCrawlerWorker.TASK_PARAM_ID_COLUMNS + "' of task "
        + taskContext.getTask().getTaskId() + " is null or empty");
    }
    _idColumns = normalizeColumnNames(_idColumns);
    _deltaColumns = normalizeColumnNames(taskParameters.get(JdbcCrawlerWorker.TASK_PARAM_DELTA_COLUMNS));
    _messages = new MessageCollector.TaskLogWarn(taskContext.getLog());
  }

  public long getMaxRecordsPerBulk() {
    return _maxRecordsPerBulk;
  }

  public long getMaxAttachmentSize() {
    return _maxAttachmentSize;
  }

  public String getDbUrl() {
    return _dbUrl;
  }

  public AnyMap getDbProperties() {
    return _dbPropertiess;
  }

  public String getCrawlSql() {
    return _crawlSql;
  }

  public Any getIdColumns() {
    return _idColumns;
  }

  public Any getDeltaColumns() {
    return _deltaColumns;
  }

  private Any normalizeColumnNames(final Any columnNames) {
    if (columnNames == null) {
      return null;
    }
    if (columnNames.isString()) {
      return DataFactory.DEFAULT.createStringValue(columnNames.asValue().asString().toUpperCase());
    }
    final AnySeq normalizedNames = DataFactory.DEFAULT.createAnySeq();
    for (final Any column : columnNames) {
      normalizedNames.add(column.asValue().asString().toUpperCase());
    }
    return normalizedNames;
  }

  public MessageCollector getMessages() {
    return _messages;
  };
}
