package org.eclipse.smila.importing.crawler.jdbc;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.util.RecordMerger;
import org.eclipse.smila.importing.crawler.jdbc.internal.ResultSetToAnyConverter;
import org.eclipse.smila.importing.util.MessageCollector;

/**
 * Class for executing multiple SQL requests by using one Connection and one PreparedStatement. Must be closed after
 * processing to clean up DB resources!
 */
public class SqlExecutor implements AutoCloseable {
  private final String _sql;

  private final Connection _con;

  private final PreparedStatement _stmt;

  private final long _maxAttachmentSize;

  private final MessageCollector _messages;

  /**
   * @param con
   *          a database connection
   * @param sql
   *          an SQL statement which will be used as PreparedStatement, so it may contain '?' parameters.
   */
  public SqlExecutor(final Connection con, final String sql, final long maxAttachmentSize,
    final MessageCollector messages) throws SQLException {
    _sql = sql;
    _con = con;
    _stmt = con.prepareStatement(sql);
    _maxAttachmentSize = maxAttachmentSize;
    _messages = messages;
  }

  public String getSql() {
    return _sql;
  }

  /**
   * @param parameters
   *          the parameters for the SQL (Prepared)statement given in the constructor.
   * @return one record. If SQL result contains multiple results, they are merged in one record.
   */
  public Record execute(final List<Value> parameters) throws SQLException, IOException {
    ResultSet rs = null;
    // PreparedStatement uses list of parameters
    for (int j = 0; j < parameters.size(); j++) {
      final Any param = parameters.get(j);
      if (param == null) {
        throw new IllegalArgumentException("NULL parameters are not supported.");
      }
      _stmt.setObject(j + 1, param.asValue().getObject()); // PreparedStatement params start with '1'!
    }
    rs = _stmt.executeQuery();
    final Collection<Record> result =
      ResultSetToAnyConverter.convertResultSetToAny(rs, _maxAttachmentSize, _messages);
    // if we have more than one result -> merge result to one record
    return mergeRecords(result);
  }

  /** close statement and db connection. */
  @Override
  public void close() {
    if (_stmt != null) {
      try {
        _stmt.close();
      } catch (final Exception e) {
      }
    }
    if (_con != null) {
      try {
        _con.close();
      } catch (final Exception e) {
      }
    }
  }

  @Override
  /** just another chance if close() was not called. (double close() on Statement and Connection is ignored. */
  protected void finalize() throws Throwable {
    close();
  }

  /** @return merged record for given records to merge. */
  private Record mergeRecords(final Collection<Record> recordsToMerge) {
    if (recordsToMerge.size() == 0) {
      return null;
    }
    if (recordsToMerge.size() == 1) {
      return recordsToMerge.iterator().next();
    }
    final Iterator<Record> it = recordsToMerge.iterator();
    final Record result = it.next();
    while (it.hasNext()) {
      final Record r = it.next();
      RecordMerger.mergeRecords(result, r, true);
    }
    return result;
  }
}
