package org.eclipse.smila.importing.crawler.jdbc.internal;

import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.importing.crawler.jdbc.DbAccessService;
import org.eclipse.smila.importing.crawler.jdbc.SqlExecutor;
import org.eclipse.smila.importing.util.MessageCollector;
import org.eclipse.smila.jdbc.JdbcProvider;

/** Implementation of {@link DbAccessService} via JDBC. */
public class JdbcDbAccessService implements DbAccessService {
  private final Log _log = LogFactory.getLog(getClass());

  private JdbcProvider _jdbcProvider;

  @Override
  public Collection<Record> executeSql(final String url, final AnyMap dbProperties, final String sql,
    final long maxAttachmentSize, final MessageCollector messages) throws SQLException, IOException {
    return executeSql(url, AnyUtil.anyToProperties(dbProperties), sql, maxAttachmentSize, messages);
  }

  @Override
  public Collection<Record> executeSql(final String url, final Properties dbProperties, final String sql,
    final long maxAttachmentSize, final MessageCollector messages) throws SQLException, IOException {
    Connection con = null;
    Statement stmt = null;
    ResultSet rs = null;
    try {
      con = _jdbcProvider.getConnection(url, dbProperties);
      stmt = con.createStatement();
      rs = stmt.executeQuery(sql);
      final Collection<Record> result =
        ResultSetToAnyConverter.convertResultSetToAny(rs, maxAttachmentSize, messages);
      return result;
    } finally {
      closeQuietly(con, stmt, rs);
    }
  }

  @Override
  public SqlExecutor executePrepared(final String url, final AnyMap dbProperties, final String sql,
    final long maxAttachmentSize, final MessageCollector messages) throws SQLException {
    return executePrepared(url, AnyUtil.anyToProperties(dbProperties), sql, maxAttachmentSize, messages);
  }

  @Override
  public SqlExecutor executePrepared(final String url, final Properties dbProperties, final String sql,
    final long maxAttachmentSize, final MessageCollector messages) throws SQLException {
    Connection con = null;
    try {
      con = _jdbcProvider.getConnection(url, dbProperties);
      final SqlExecutor sqlExecutor = new SqlExecutor(con, sql, maxAttachmentSize, messages);
      return sqlExecutor;
    } catch (final SQLException e) {
      closeQuietly(con, null, null);
      throw e;
    } catch (final Exception e) {
      closeQuietly(con, null, null);
      throw new SQLException(e);
    }
  }

  /** close ResultSet, Statement and Connection quietly. */
  private void closeQuietly(final Connection con, final Statement stmt, final ResultSet rs) {
    if (rs != null) {
      try {
        rs.close();
      } catch (final Exception e) {
        _log.warn("Error while closing ResultSet: ", e);
      }
    }
    if (stmt != null) {
      try {
        stmt.close();
      } catch (final Exception e) {
        _log.warn("Error while closing Statement: ", e);
      }
    }
    if (con != null) {
      try {
        con.close();
      } catch (final Exception e) {
        _log.warn("Error while closing Connection: ", e);
      }
    }
  }

  /** DS service reference injection method. */
  public void setJdbcProvider(final JdbcProvider jdbcProvider) {
    _jdbcProvider = jdbcProvider;
  }

  /** DS service reference removal method. */
  public void unsetJdbcProvider(final JdbcProvider jdbcProvider) {
    if (_jdbcProvider == jdbcProvider) {
      _jdbcProvider = null;
    }
  }

}
