/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.importing.crawler.web.test.httphandler;

import java.io.IOException;

import org.apache.commons.io.IOUtils;
import org.eclipse.jetty.http.HttpMethods;
import org.eclipse.jetty.http.HttpStatus;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.util.ARequestHandler;
import org.eclipse.smila.http.server.util.UnsupportedMethodException;

/** A simple handler for testing HTTP redirects */
public class RedirectHandler extends ARequestHandler {

  @Override
  public void handle(final HttpExchange exchange) {
    boolean closeConnection = false;
    String resultHtml = null;
    String redirectUrl = null;
    final String method = exchange.getRequestMethod();
    final String requestUri = exchange.getRequestURI();
    try {
      try {
        if (!HttpMethods.GET.equals(method)) {
          throw new UnsupportedMethodException("Unsupported http method '" + method + "'");
        }
        final String[] pages = exchange.getParameterValues("page");
        if (pages != null) {
          int page = Integer.parseInt(pages[0]);
          if (page > 0) {
            redirectUrl = requestUri + "?page=" + (page - 1);
          } else {
            resultHtml = createStaticContent();
          }
        } else {
          resultHtml = createStaticContent();
        }
      } catch (final Throwable error) {
        _log.warn("Error processing " + method + " " + requestUri, error);
        closeConnection = prepareResponseHeader(exchange, 500, false);
        return; // to skip writing of success result.
      }

      if (redirectUrl != null) {
        closeConnection = writeRedirectResult(exchange, redirectUrl);
      } else {
        closeConnection = writeSuccessResult(exchange, resultHtml);
      }
    } catch (final IOException ex) {
      _log.warn("Error writing the response. Maybe client has closed the connection.", ex);
      closeConnection = true;
    } finally {
      try {
        if (closeConnection) {
          IOUtils.closeQuietly(exchange.getResponseStream());
        } else {
          exchange.getResponseStream().flush();
        }
      } catch (final Exception ex) {
        ; // if there is an exception it should have occurred before and be logged and everything.
      }
    }
  }

  public boolean prepareResponseHeader(final HttpExchange exchange, final int statusCode,
    final boolean hasResponseBody) {
    boolean closeConnection = false;
    exchange.setResponseStatus(statusCode);
    final String connectionHeader = exchange.getRequestHeader("Connection");
    if (connectionHeader != null && connectionHeader.equals("close")) {
      closeConnection = true;
      exchange.setResponseHeader("Connection", "close");
    } else {
      exchange.setResponseHeader("Connection", "keep-alive");
      exchange.setResponseHeader("Keep-Alive", "timeout=0");
    }
    if (hasResponseBody && statusCode != HttpStatus.NO_CONTENT_204) {
      exchange.setResponseHeader("Content-Type", "text/html; charset=utf-8");
      exchange.setResponseHeader("Transfer-Encoding", "chunked");
    } else {
      exchange.setResponseHeader("Content-Length", "0");
    }
    return closeConnection;
  }

  private final boolean writeSuccessResult(final HttpExchange exchange, final String resultHtml) throws IOException {
    final int statusCode = HttpStatus.OK_200;
    final boolean closeConnection = prepareResponseHeader(exchange, statusCode, resultHtml != null);
    if (resultHtml != null) {
      exchange.getResponseStream().write(resultHtml.getBytes());
    }
    return closeConnection;
  }

  private final boolean writeRedirectResult(final HttpExchange exchange, final String redirectUrl)
    throws IOException {
    final int statusCode = HttpStatus.MOVED_PERMANENTLY_301;
    final boolean closeConnection = prepareResponseHeader(exchange, statusCode, false);
    exchange.setResponseHeader("Location", redirectUrl);
    return closeConnection;
  }

  public static final String createStaticContent() {
    final StringBuffer buffer = new StringBuffer();
    buffer.append("<html><body>");
    buffer
      .append("This page was generated for redirect testing. This is the final result page where every redirects should finally arrive.");
    buffer.append("<br>More static data can be found <a href=\"../../files/\">here</a>.");
    buffer.append("</body></html>");
    return buffer.toString();
  }
}
