package org.eclipse.smila.importing.test;

import java.util.List;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.importing.DeltaImportStrategy;
import org.eclipse.smila.importing.DeltaService;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.worker.UpdatePusherTaskGenerator;
import org.eclipse.smila.importing.worker.UpdatePusherWorker;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerator;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/** Unit tests for {@link UpdatePusherTaskGenerator}. */
public class TestUpdatePusherTaskGenerator extends DeclarativeServiceTestCase {
  private static final String WORKER = "updatePusher";

  /** the service under test. */
  private TaskGenerator _taskGenerator;

  /** use in test to simulate entries. */
  private DeltaService _deltaService;

  @Override
  protected void setUp() throws Exception {
    _taskGenerator = getService(TaskGenerator.class, "(generatorName=" + WORKER + ")");
    _deltaService = getService(DeltaService.class);
    _deltaService.clearAll();
  }

  /** test if correct taskgenerator was found. */
  public void testGeneratorClass() throws Exception {
    assertEquals(UpdatePusherTaskGenerator.class, _taskGenerator.getClass());
  }

  /** test creation of for empty source: no tasks. */
  public void testGenerateEmptyCompletionTasks() throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(ImportingConstants.TASK_PARAM_DATA_SOURCE, "s");
    final List<Task> tasks = _taskGenerator.createCompletionTasks(null, null, parameters, WORKER);
    assertTrue(tasks.isEmpty());
  }

  /** test creation of completion tasks: one per shard. */
  public void testGenerateCompletionTasks() throws Exception {
    _deltaService.markAsUpdated("s", "record1", "j1", "h1");
    _deltaService.markAsUpdated("s", "record2", "j1", "h1");
    _deltaService.markAsUpdated("s", "record3", "j1", "h1");
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(ImportingConstants.TASK_PARAM_DATA_SOURCE, "s");
    final List<Task> tasks = _taskGenerator.createCompletionTasks(null, null, parameters, WORKER);
    assertEquals(3, tasks.size());
    for (final Task task : tasks) {
      assertTrue(task.getProperties().containsKey(Task.PROPERTY_IS_COMPLETING_TASK));
      final String shard = task.getParameters().getStringValue(UpdatePusherWorker.DELETE_SHARD_PARAM);
      assertNotNull(shard);
      assertFalse(_deltaService.getUnvisitedEntries(shard, "j2").isEmpty());
    }
  }

  /** test if tasks are only created if delta-delete is enabled by deltaImportStrategy parameter. */
  public void testCompletionTasksDisabled() throws Exception {
    _deltaService.markAsUpdated("s", "record1", "j1", "h1");
    _deltaService.markAsUpdated("s", "record2", "j1", "h1");
    _deltaService.markAsUpdated("s", "record3", "j1", "h1");
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(ImportingConstants.TASK_PARAM_DATA_SOURCE, "s");
    parameters.put(DeltaImportStrategy.TASK_PARAM, "full");
    assertEquals(3, _taskGenerator.createCompletionTasks(null, null, parameters, WORKER).size());
    parameters.put(DeltaImportStrategy.TASK_PARAM, "disabled");
    assertTrue(_taskGenerator.createCompletionTasks(null, null, parameters, WORKER).isEmpty());
    parameters.put(DeltaImportStrategy.TASK_PARAM, "initial");
    assertTrue(_taskGenerator.createCompletionTasks(null, null, parameters, WORKER).isEmpty());
    parameters.put(DeltaImportStrategy.TASK_PARAM, "additive");
    assertTrue(_taskGenerator.createCompletionTasks(null, null, parameters, WORKER).isEmpty());
  }

}
