package org.eclipse.smila.importing.util.test;

import java.util.HashSet;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.util.PropertyNameMapper;
import org.eclipse.smila.taskworker.TaskContext;

/** Test for {@link PropertyNameMapper} class. */
public class TestPropertyNameMapper extends TestCase {

  PropertyNameMapper _mapper;

  Record _record;

  @Override
  public void setUp() {
    final AnyMap mapping = DataFactory.DEFAULT.createAnyMap();
    mapping.put("key1", "my-key1");
    mapping.put("key2", "my-key2");
    mapping.put("key3", "key3");
    mapping.put("key4", "my-key4");
    mapping.put("key5", "my-key5");
    mapping.put("attachment1", "my-attachment1");
    mapping.put("attachment2", "attachment2");

    _record = DataFactory.DEFAULT.createRecord("testRecord");
    _record.getMetadata().put("key1", "value1");
    _record.getMetadata().put("key2", "value2");
    _record.getMetadata().put("key3", "value3");
    _record.getMetadata().put("key4", "value4");
    _record.getMetadata().put("key5", "value5");
    _record.getMetadata().put("key6", "value6");
    _record.getMetadata().put("key7", "value7");
    _record.getMetadata().put("noKey", "value8");
    _record.setAttachment("attachment1", "bytes1".getBytes());
    _record.setAttachment("attachment2", "bytes2".getBytes());
    _record.setAttachment("attachment3", "bytes3".getBytes());
    _record.setAttachment("noKeyAttachment", "bytes4".getBytes());

    _mapper = new PropertyNameMapper(mapping);
  }

  public void testExceptions() {
    try {
      PropertyNameMapper.createFrom((TaskContext) null);
      fail("expected NullPointerException");
    } catch (final NullPointerException e) {
    }

    try {
      new PropertyNameMapper((AnyMap) null);
      fail("expected NullPointerException");
    } catch (final NullPointerException e) {
    }

    try {
      new PropertyNameMapper(DataFactory.DEFAULT.createAnyMap());
      fail("expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
    }
  }

  public void testMapping() {
    _mapper.mapNames(_record, null);

    assertFalse(_record.getMetadata().containsKey("key1")); // mapped to different name
    assertFalse(_record.getMetadata().containsKey("key2")); // mapped to different name
    assertFalse(_record.getMetadata().containsKey("key4")); // mapped to different name
    assertFalse(_record.getMetadata().containsKey("key5")); // mapped to different name
    assertFalse(_record.hasAttachment("attachment1")); // mapped to different name

    assertEquals("value1", _record.getMetadata().getStringValue("my-key1"));
    assertEquals("value2", _record.getMetadata().getStringValue("my-key2"));
    assertEquals("value3", _record.getMetadata().getStringValue("key3"));
    assertEquals("value4", _record.getMetadata().getStringValue("my-key4"));
    assertEquals("value5", _record.getMetadata().getStringValue("my-key5"));
    assertEquals("value6", _record.getMetadata().getStringValue("key6"));
    assertEquals("value7", _record.getMetadata().getStringValue("key7"));
    assertEquals("value8", _record.getMetadata().getStringValue("noKey")); // neither part of mapping nor of
                                                                           // propertyNames (does usually not happen in
                                                                           // crawler context)

    assertEquals("bytes1", new String(_record.getAttachmentAsBytes("my-attachment1")));
    assertEquals("bytes2", new String(_record.getAttachmentAsBytes("attachment2")));
    assertEquals("bytes3", new String(_record.getAttachmentAsBytes("attachment3")));
    assertEquals("bytes4", new String(_record.getAttachmentAsBytes("noKeyAttachment"))); // neither part of mapping nor
                                                                                         // of propertyNames (does
                                                                                         // usually not happen in
                                                                                         // crawler context)
  }

  public void testMappingWithPropertyNameFiltering() {
    final HashSet<String> propertyNames = new HashSet<String>();
    propertyNames.add("key1");
    propertyNames.add("key2");
    propertyNames.add("key3");
    propertyNames.add("key4");
    propertyNames.add("key5");
    propertyNames.add("key6");
    propertyNames.add("key7");
    propertyNames.add("attachment1");
    propertyNames.add("attachment2");
    propertyNames.add("attachment3");
    _mapper.mapNames(_record, propertyNames);

    assertFalse(_record.getMetadata().containsKey("key1")); // mapped to different name
    assertFalse(_record.getMetadata().containsKey("key2")); // mapped to different name
    assertFalse(_record.getMetadata().containsKey("key4")); // mapped to different name
    assertFalse(_record.getMetadata().containsKey("key5")); // mapped to different name
    assertFalse(_record.getMetadata().containsKey("key6")); // not part of mapping
    assertFalse(_record.getMetadata().containsKey("key7")); // not part of mapping
    assertFalse(_record.hasAttachment("attachment1")); // mapped to different name
    assertFalse(_record.hasAttachment("attachment3")); // not part of mapping

    assertEquals("value1", _record.getMetadata().getStringValue("my-key1"));
    assertEquals("value2", _record.getMetadata().getStringValue("my-key2"));
    assertEquals("value3", _record.getMetadata().getStringValue("key3"));
    assertEquals("value4", _record.getMetadata().getStringValue("my-key4"));
    assertEquals("value5", _record.getMetadata().getStringValue("my-key5"));
    assertEquals("value8", _record.getMetadata().getStringValue("noKey")); // neither part of mapping nor of
                                                                           // propertyNames (does usually not happen in
                                                                           // crawler context)

    assertEquals("bytes1", new String(_record.getAttachmentAsBytes("my-attachment1")));
    assertEquals("bytes2", new String(_record.getAttachmentAsBytes("attachment2")));
    assertEquals("bytes4", new String(_record.getAttachmentAsBytes("noKeyAttachment"))); // neither part of mapping nor
                                                                                         // of propertyNames (does
                                                                                         // usually not happen in
                                                                                         // crawler context)
  }

  public void testMultiMapping() {

    final AnyMap mapping = DataFactory.DEFAULT.createAnyMap();
    final AnySeq multiMappingForKey1 = DataFactory.DEFAULT.createAnySeq();
    multiMappingForKey1.add("my-key1.1");
    multiMappingForKey1.add("my-key1.2");
    multiMappingForKey1.add("my-key1.3");
    mapping.put("key1", multiMappingForKey1);
    mapping.put("key2", "my-key2");
    mapping.put("key3", "key3");
    final AnySeq multiMappingForKey4 = DataFactory.DEFAULT.createAnySeq();
    multiMappingForKey4.add("my-key4");
    multiMappingForKey4.add("key4"); // also map it to its own attribute name
    mapping.put("key4", multiMappingForKey4);
    mapping.put("key5", "my-key5");
    mapping.put("attachment1", "my-attachment1");
    mapping.put("attachment2", "attachment2");

    final PropertyNameMapper mapper = new PropertyNameMapper(mapping);
    mapper.mapNames(_record, null);

    assertFalse(_record.getMetadata().containsKey("key1")); // mapped to different name
    assertFalse(_record.getMetadata().containsKey("key2")); // mapped to different name
    assertTrue(_record.getMetadata().containsKey("key4")); // mapped to different name, but also mapped to original name
    assertFalse(_record.getMetadata().containsKey("key5")); // mapped to different name
    assertFalse(_record.hasAttachment("attachment1")); // mapped to different name

    assertEquals("value1", _record.getMetadata().getStringValue("my-key1.1"));
    assertEquals("value1", _record.getMetadata().getStringValue("my-key1.2"));
    assertEquals("value1", _record.getMetadata().getStringValue("my-key1.3"));
    assertEquals("value2", _record.getMetadata().getStringValue("my-key2"));
    assertEquals("value3", _record.getMetadata().getStringValue("key3"));
    assertEquals("value4", _record.getMetadata().getStringValue("my-key4"));
    assertEquals("value5", _record.getMetadata().getStringValue("my-key5"));
    assertEquals("value6", _record.getMetadata().getStringValue("key6"));
    assertEquals("value7", _record.getMetadata().getStringValue("key7"));
    assertEquals("value8", _record.getMetadata().getStringValue("noKey")); // neither part of mapping nor of
                                                                           // propertyNames (does usually not happen in
                                                                           // crawler context)

    assertEquals("bytes1", new String(_record.getAttachmentAsBytes("my-attachment1")));
    assertEquals("bytes2", new String(_record.getAttachmentAsBytes("attachment2")));
    assertEquals("bytes3", new String(_record.getAttachmentAsBytes("attachment3")));
    assertEquals("bytes4", new String(_record.getAttachmentAsBytes("noKeyAttachment"))); // neither part of mapping nor
                                                                                         // of propertyNames (does
                                                                                         // usually not happen in
                                                                                         // crawler context)
  }
}
