package org.eclipse.smila.importing.util.test;

import junit.framework.TestCase;

import org.eclipse.smila.importing.util.RegexPatternMatcher;

/** Test for {@link RegexPatternMatcher} class. */
public class TestRegexPatternMatcher extends TestCase {

  /** tests that pattern matches if no includes and excludes are given. */
  public void testNoIncludesAndNoExcludes() {
    final RegexPatternMatcher matcher = new RegexPatternMatcher();
    assertTrue(matcher.matches("test"));
    assertTrue(matcher.isIncluded("bla")); // no patterns - everything is included
    assertFalse(matcher.isExcluded("4711")); // no patterns - nothing is excluded
  }

  /** test with only exclude patterns specified. */
  public void testExcludesOnly() {
    final RegexPatternMatcher matcher = new RegexPatternMatcher();
    matcher.addExcludePattern(".*/?.*\\.txt");
    matcher.addExcludePattern(".*/?.*\\.doc");

    assertFalse(matcher.matches("test.doc"));
    assertFalse(matcher.matches("C:/dir1/dir2/test.doc"));
    assertFalse(matcher.matches("test.txt"));
    assertFalse(matcher.matches("/dir1/dir2/test.txt"));

    assertTrue(matcher.matches("test.pdf"));
    assertTrue(matcher.matches("C:/dir1/dir2/test.pdf"));
    
    assertTrue(matcher.isIncluded("bla")); // no include patterns - everything is included
    assertTrue(matcher.isExcluded("test.txt"));
    assertFalse(matcher.isExcluded("test.pdf"));
  }

  /** test with only include patterns specified. */
  public void testIncludesOnly() {        
    final RegexPatternMatcher matcher = new RegexPatternMatcher();    
    
    matcher.addIncludePattern(".*/?.*\\.txt");
    matcher.addIncludePattern(".*/?.*\\.doc");

    assertTrue(matcher.matches("test.doc"));
    assertTrue(matcher.matches("C:/dir1/dir2/test.doc"));
    assertTrue(matcher.matches("test.txt"));
    assertTrue(matcher.matches("/dir1/dir2/test.txt"));

    assertFalse(matcher.matches("test.pdf"));
    assertFalse(matcher.matches("C:/dir1/dir2/test.pdf"));
        
    assertFalse(matcher.isIncluded("bla"));
    assertTrue(matcher.isIncluded("test.txt"));    
    assertFalse(matcher.isExcluded("test.pdf")); // no exclude patterns given
  }

  /** test with includes and excludes that don't match -> no match. */
  public void testNonMatchingIncludesAndExcludes() {
    final RegexPatternMatcher matcher = new RegexPatternMatcher();
    matcher.addIncludePattern(".*/?.*\\.txt");
    matcher.addExcludePattern(".*/?.*\\.doc");
    assertFalse(matcher.matches("test.pdf"));
  }

  /** test with includes and excludes that match both -> no match. */
  public void testMatchingIncludesAndExcludes() {
    final RegexPatternMatcher matcher = new RegexPatternMatcher();
    matcher.addIncludePattern("\\.*doc");
    matcher.addExcludePattern(".*/?.*\\.doc");
    assertFalse(matcher.matches("test.doc"));
  }

  /** test with includes that match and excludes that don't match -> match. */
  public void testMatchingIncludesAndNonMatchingExcludes() {
    final RegexPatternMatcher matcher = new RegexPatternMatcher();
    matcher.addIncludePattern(".*/?.*\\.txt");
    matcher.addExcludePattern(".*/?.*\\.doc");
    assertTrue(matcher.matches("test.txt"));
  }

  /** test with includes that don't match and excludes that match -> no match. */
  public void testNonMatchingIncludesAndMatchingExcludes() {
    final RegexPatternMatcher matcher = new RegexPatternMatcher();
    matcher.addIncludePattern(".*/?.*\\.txt");
    matcher.addExcludePattern(".*/?.*\\.doc");
    assertFalse(matcher.matches("test.doc"));
  }

  /** tests with in- and exclude pattern. */
  public void testMultipleInAndExcludes() {
    final RegexPatternMatcher matcher = new RegexPatternMatcher();
    matcher.addIncludePattern(".*/?.*\\.txt");
    matcher.addExcludePattern(".*/?.*\\.pdf");
    matcher.addIncludePattern(".*/?.*\\.doc");
    matcher.addExcludePattern(".*/?.*\\.xls");

    assertTrue(matcher.matches("test.doc"));
    assertFalse(matcher.matches("test.pdf"));
    assertTrue(matcher.matches("test.doc"));
    assertFalse(matcher.matches("test.xls"));
    assertFalse(matcher.matches("test.???"));
  }

  /** tests with overlapping in- and exclude pattern. */
  public void testOverlappingInAndExcludes() {
    final RegexPatternMatcher matcher = new RegexPatternMatcher();
    matcher.addIncludePattern(".*\\.txt");
    matcher.addExcludePattern("a.*\\.txt");

    matcher.addIncludePattern("b.*\\.pdf");
    matcher.addExcludePattern(".*\\.pdf");

    assertTrue(matcher.matches("bbb.txt"));
    assertFalse(matcher.matches("aaa.txt"));

    assertFalse(matcher.matches("bbb.pdf")); // exclude dominates!
    assertFalse(matcher.matches("aaa.pdf"));
  }

}
