/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.ipc.bon;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;

/**
 * Helper class to convert between byte arrays and scalar values.
 * 
 * @author aweber
 */
public final class ValueTrafo {

  /**
   * default encoding for strings: utf-8.
   */
  private static final String DEFAULT_STRING_ENCODING = "utf-8";

  /**
   * Number of bits for one byte.
   */
  private static final int BYTE_LENGTH = 8;

  /**
   * Byte having all bits set.
   */
  private static final int BYTE_FF = 0xff;

  /**
   * Maximum length for a byte array to convert to long.
   */
  private static final int MAX_LONG_BYTE_ARRAY_LENGTH = 8;

  /**
   * Maximum length for a byte array to convert to int.
   */
  private static final int MAX_INT_BYTE_ARRAY_LENGTH = 4;

  /**
   * prevent instance creation.
   */
  private ValueTrafo() {
    // prevent instance creation
  }

  /**
   * Converts given byte array to positive long value.
   * 
   * @param b
   *          the byte array with length <= 8
   * @return long value
   */
  public static long byte2long(final byte[] b) {
    // assumes Big Endian byte order
    // assumes result is unsigned
    long result = 0;
    long currentByte;
    final int len = b.length; // length = number of bytes
    if (len > MAX_LONG_BYTE_ARRAY_LENGTH) {
      throw new IllegalArgumentException("byte array length must be <= " + MAX_LONG_BYTE_ARRAY_LENGTH + "; length="
        + len);
    }
    for (int i = len - 1; i >= 0; i--) {
      // start with lowest byte (Big Endian -> lowest byte at the right)
      currentByte = b[i] & BYTE_FF; // '& 0xff' removes "false sign bit" for values > 127
      result = result + (currentByte << (BYTE_LENGTH * (len - 1 - i))); // shift one byte left;
    }
    return result;
  }

  /**
   * Converts given byte array to positive int value assuming result < 2^31.
   * 
   * @param b
   *          the byte array with length <= 4
   * @return int value.
   */
  public static int byte2int(final byte[] b) {
    // assumes Big Endian byte order
    // assumes result is unsigned
    int result = 0;
    int currentByte;
    final int len = b.length; // length = number of bytes
    if (len > MAX_INT_BYTE_ARRAY_LENGTH) {
      throw new IllegalArgumentException("byte array length must be <= " + MAX_INT_BYTE_ARRAY_LENGTH + "; length="
        + len);
    }
    for (int i = len - 1; i >= 0; i--) {
      // start with lowest byte (Big Endian -> lowest byte at the right)
      currentByte = b[i] & BYTE_FF; // '& 0xff' removes "false sign bit" for values > 127
      result = result + (currentByte << (BYTE_LENGTH * (len - 1 - i))); // shift one byte left;
    }
    return result;
  }

  /**
   * Converts positive long value to byte array in Big Endian byte order, assuming the length of the resulting byte
   * array has been calculated before.
   * 
   * @param v
   *          long value.
   * @param byteLength
   *          length of resulting byte array
   * @return byte array representing the long value
   */
  public static byte[] long2byte(final long v, final int byteLength) {
    if (v < 0) {
      throw new IllegalArgumentException("value must be > 0");
    }
    final byte[] b = getBytes(v, byteLength);
    return b;
  }

  /**
   * Converts positive long value to byte array in Big Endian byte order. The byte array length is the minimum needed to
   * store the value.
   * 
   * @param v
   *          long value.
   * @return byte array representing the long value
   */
  public static byte[] long2byte(final long v) {
    final int size = getBytesLength(v);
    return long2byte(v, size);
  }

  /**
   * Convert long to byte[] with given length in Big Endian byte order.
   * 
   * @param v
   *          long value.
   * @param size
   *          length of byte array.
   * @return byte array representing the long value
   */
  private static byte[] getBytes(long v, final int size) {
    final byte[] b = new byte[size];
    // fill bytes:
    for (int i = size - 1; i >= 0; i--) {
      // start with lowest byte (Big Endian -> lowest byte at the "right of the array")
      b[i] = (byte) (v & BYTE_FF); // eliminate all bits but current byte
      v >>= BYTE_LENGTH; // shift right -> next byte
    }
    return b;
  }

  /**
   * Get minimum number of bytes to store the long value.
   * 
   * @param v
   *          long value.
   * @return minimum length of byte array.
   */
  public static int getBytesLength(final long v) {
    long tmp = v;
    int size = 0;
    do {
      size++;
      tmp >>= BYTE_LENGTH;
    } while (tmp != 0);
    return size;
  }

  /**
   * Reads 8 byte from the byte array and converts them to a double value.
   * 
   * @param b
   *          byte array
   * @return double value. Throws java.nio.BufferUnderflowException, if byte array length < 8.
   */
  public static double byte2double(final byte[] b) {
    // assumes Big Endian byte order
    return ByteBuffer.wrap(b).getDouble();
  }

  /**
   * Converts the given byte array to a string by using the default encoding.
   * 
   * @param b
   *          byte array.
   * @return string value
   */
  public static String byte2string(final byte[] b) {
    return new String(b, Charset.forName(DEFAULT_STRING_ENCODING));
  }

  /**
   * Converts the given double value to a byte array.
   * 
   * @param d
   *          double value.
   * @return 1 token byte + 8 bytes in Big Endian byte order (IEEE format)
   */
  public static byte[] double2byte(final double d) {
    // writes Big Endian byte order
    final ByteBuffer bb = ByteBuffer.allocate(BinaryToken.SCALAR_DOUBLE.valueLength());
    bb.putDouble(d); // double -> 8 bytes length
    return bb.array(); // safe, because buffer is created via allocate()
  }

  /**
   * Converts the given string to a byte array using the default encoding (utf-8).
   * 
   * @param s
   *          string value.
   * @return byte representation of string
   */
  public static byte[] string2byte(final String s) {
    return s.getBytes(Charset.forName(DEFAULT_STRING_ENCODING));
  }
}
