/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH.  
 * All rights reserved. This program and the accompanying materials are made available
 * under the terms of the Eclipse Public License v1.0 which accompanies this distribution, 
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.ipc.util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.commons.io.output.NullOutputStream;
import org.eclipse.smila.ipc.IpcStreamReader;
import org.eclipse.smila.ipc.IpcToken;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.ipc.bon.BinaryStreamReader;
import org.eclipse.smila.ipc.bon.BinaryStreamWriter;
import org.eclipse.smila.ipc.json.JsonFactory;
import org.eclipse.smila.ipc.json.JsonStreamReader;

/**
 * Transforms IPC input stream (e.g. JSON) to binary (BON) output stream.
 * 
 * The incoming stream can be validated by a given Validator implementation.
 * 
 * @author aweber
 */
public class Ipc2BonTrafo {

  /** The validator to use in validation. */
  private IpcValidator _validator;

  /** Speciefies if validation should be done. */
  private final boolean _validate;

  /** the factory to create JSON reader/writer objects. */
  private final JsonFactory _jsonFactory = new JsonFactory();

  /** the factory to create BON reader/writer objects. */
  private final BinaryFactory _bonFactory = new BinaryFactory();

  /**
   * constructor without Validator.
   */
  public Ipc2BonTrafo() {
    _validate = false;
  }

  /**
   * @param validator
   *          used to validate the input JSON.
   */
  public Ipc2BonTrafo(final IpcValidator validator) {
    _validator = validator;
    _validate = true;
  }

  /**
   * Writes an JSON input stream to a binary (BON) output stream.
   * 
   * @param input
   *          the (JSON) input stream
   * @param output
   *          the (BON) output stream
   * @throws IOException
   *           conversion error
   */
  public void json2bon(final InputStream input, final OutputStream output) throws IOException {
    final JsonStreamReader reader = _jsonFactory.newStreamReader(input);
    final BinaryStreamWriter writer = _bonFactory.newStreamWriter(output);
    try {
      ipc2bon(reader, writer);
    } finally {
      writer.closeWithoutStream();
      reader.closeWithoutStream();
    }
  }

  /**
   * Validates a BON object.
   * 
   * @param input
   *          the BON input stream
   * @throws IOException
   *           validation error
   */
  public void validateBon(final byte[] input) throws IOException {
    final BinaryStreamReader reader = _bonFactory.newStreamReader(new ByteArrayInputStream(input));
    final BinaryStreamWriter writer = _bonFactory.newStreamWriter(new NullOutputStream());
    try {
      ipc2bon(reader, writer);
    } finally {
      writer.close();
      reader.close();
    }
  }

  /**
   * Writes an ipc input stream (e.g. JSON) to a binary (BON) output stream.
   * 
   * @param ipcReader
   *          contains the input stream
   * @param bonWriter
   *          contains the output stream
   * @throws IOException
   *           conversion error
   */
  public void ipc2bon(final IpcStreamReader ipcReader, final BinaryStreamWriter bonWriter) throws IOException {
    IpcToken token = null;
    do {
      token = ipcReader.nextToken();
      if (_validate) {
        _validator.validate(token, ipcReader.currentStringValue());
      }
      writeToken(ipcReader, bonWriter, token);
    } while (token != IpcToken.OBJECT_END);
  }

  /**
   * Writes an ipc token to a binary (BON) output stream.
   * 
   * @param ipcReader
   *          contains the input stream
   * @param bonWriter
   *          contains the output stream
   * @param token
   *          the token to write
   * @throws IOException
   *           conversion error
   */
  private void writeToken(final IpcStreamReader ipcReader, final BinaryStreamWriter bonWriter, final IpcToken token)
    throws IOException {
    switch (token) {
      case MAPPING_START:
        bonWriter.writeMappingStart();
        break;
      case MAPPING_END:
        bonWriter.writeMappingEnd();
        break;
      case SEQUENCE_START:
        bonWriter.writeSequenceStart();
        break;
      case SEQUENCE_END:
        bonWriter.writeSequenceEnd();
        break;
      case SCALAR_STRING:
        bonWriter.writeScalarString(ipcReader.currentStringValue());
        break;
      case SCALAR_INT:
        bonWriter.writeScalarLong(ipcReader.currentLongValue());
        break;
      case SCALAR_BOOL:
        bonWriter.writeScalarBoolean(ipcReader.currentBoolValue());
        break;
      case SCALAR_DOUBLE:
        bonWriter.writeScalarDouble(ipcReader.currentDoubleValue());
        break;
      case OBJECT_START:
        bonWriter.writeObjectStart();
        break;
      case OBJECT_END:
        bonWriter.writeObjectEnd();
        break;
      default:
        throw new IOException("Unexpected token " + token);
    }
  }
}
