/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.persistence.zk.test;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.ValueFormatHelper;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.JobRunDefinitions;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.definitions.WorkflowAction;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorageException;
import org.eclipse.smila.jobmanager.persistence.zk.RunStorageZk;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test for RunStorage service.
 */
public class TestRunStorageZk extends DeclarativeServiceTestCase {

  /** RunStorage service to test. */
  private RunStorageZk _runStorage;

  /**
   * clean up test data.
   * 
   * @throws Exception
   *           error
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _runStorage = (RunStorageZk) getService(RunStorage.class);
    _runStorage.clear();
  }

  /**
   * Create simple job run definitions to store for a job run.
   */
  private JobRunDefinitions createJobRunDef(final String jobName) throws Exception {
    final JobRunDefinitions runDef = new JobRunDefinitions();
    final AnyMap jobDef = DataFactory.DEFAULT.createAnyMap();
    jobDef.put(JobDefinition.KEY_NAME, jobName);
    jobDef.put(JobDefinition.KEY_WORKFLOW, "Workflow");
    final AnyMap wfDef = DataFactory.DEFAULT.createAnyMap();
    wfDef.put(WorkflowDefinition.KEY_NAME, "Workflow");
    final AnyMap wfAction = DataFactory.DEFAULT.createAnyMap();
    wfAction.put(WorkflowAction.KEY_WORKER, "Worker");
    wfDef.put(WorkflowDefinition.KEY_START_ACTION, wfAction);
    wfDef.put("additionalParameter", "additional parameter");
    wfDef.put("additionalParameter2", "another additional parameter");

    runDef.setJobDefinition(new JobDefinition(jobDef));
    runDef.setWorkflowDefinition(new WorkflowDefinition(wfDef));
    runDef.addBucketDefinition(new BucketDefinition("Bucket1", "recordBulks"));
    runDef.addBucketDefinition(new BucketDefinition("Bucket2", "recordBulks"));
    return runDef;
  }

  /**
   * Test if service was successfully started and registered.
   * 
   * @throws Exception
   *           no service found.
   */
  public void testService() throws Exception {
    final RunStorage service = getService(RunStorage.class);
    assertNotNull(service);
    assertTrue(service instanceof RunStorageZk);
  }

  /**
   * Tests start/finish job run, get job run data, get/set job state.
   * 
   * @throws Exception
   *           error
   */
  public void testBasicJobRun() throws Exception {
    final String jobName = "myJob";
    final String jobRunId = "4711";
    final Collection<String> jobNameList = new ArrayList<String>();
    jobNameList.add(jobName);
    assertTrue(_runStorage.getCurrentJobs().isEmpty());
    _runStorage.startJobRun(jobName, jobRunId, JobRunMode.STANDARD, createJobRunDef(jobName));
    assertEquals(JobState.PREPARING, _runStorage.getJobState(jobName));
    assertEquals(jobNameList, _runStorage.getCurrentJobs());
    _runStorage.setJobState(jobName, jobRunId, JobState.RUNNING);
    assertEquals(JobState.RUNNING, _runStorage.getJobState(jobName));
    assertEquals(jobRunId, _runStorage.getJobRunId(jobName));
    assertEquals(jobNameList, _runStorage.getCurrentJobs());

    final AnyMap jobRunData = _runStorage.getJobRunData(jobName, false);
    final AnyMap workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertTrue(jobRunData.containsKey(JobManagerConstants.DATA_JOB_ID));
    assertEquals(jobRunId, jobRunData.getStringValue(JobManagerConstants.DATA_JOB_ID));
    assertTrue(jobRunData.containsKey(JobManagerConstants.DATA_JOB_STATE));
    assertEquals(JobState.RUNNING.name(), jobRunData.getStringValue(JobManagerConstants.DATA_JOB_STATE));
    assertTrue(workflowRunData.containsKey(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertTrue(workflowRunData.containsKey(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertTrue(jobRunData.containsKey(JobManagerConstants.DATA_JOB_RUN_MODE));
    assertEquals("STANDARD", jobRunData.getStringValue(JobManagerConstants.DATA_JOB_RUN_MODE));

    _runStorage.finishJobRun(jobName, jobRunId);
    assertEquals(JobState.FINISHING, _runStorage.getJobState(jobName));
    assertEquals(jobNameList, _runStorage.getCurrentJobs());
    _runStorage.setJobState(jobName, jobRunId, JobState.SUCCEEDED);
    assertEquals(JobState.SUCCEEDED, _runStorage.getJobState(jobName));
    assertEquals(jobNameList, _runStorage.getCurrentJobs());
    _runStorage.deleteJobRun(jobName, jobRunId);
    assertTrue(_runStorage.getCurrentJobs().isEmpty());
  }

  /**
   * Tests start workflow run, start/delete tasks.
   * 
   * @throws Exception
   *           error
   */
  public void testBasicWorkflowRun() throws Exception {
    final String jobName = "myJob";
    final String jobRunId = "4711";

    // create counters for testing
    final Map<String, Number> counters = new HashMap<String, Number>();
    counters.put("counter-1", 3);
    counters.put("counter-2", 2.0);

    _runStorage.startJobRun(jobName, jobRunId, JobRunMode.STANDARD, createJobRunDef(jobName));
    AnyMap jobRunData = _runStorage.getJobRunData(jobName, false);
    AnyMap workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));

    final String workflowRunId = _runStorage.startWorkflowRun(jobName, jobRunId);
    assertEquals(workflowRunId, "1");
    jobRunData = _runStorage.getJobRunData(jobName, false);
    workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertEquals(String.valueOf(1),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertEquals(String.valueOf(1),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));

    workflowRunData = _runStorage.getWorkflowRunData(jobName, workflowRunId);
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    assertFalse(_runStorage.hasTasks(jobName, jobRunId, workflowRunId));
    _runStorage.startTask(jobName, jobRunId, workflowRunId, "step-1", "task-1");
    assertTrue(_runStorage.hasTasks(jobName, jobRunId, workflowRunId));
    _runStorage.startTask(jobName, jobRunId, workflowRunId, "step-1", "task-2");
    _runStorage.startTask(jobName, jobRunId, workflowRunId, "step-2", "task-3");
    _runStorage.startTask(jobName, jobRunId, workflowRunId, "step-3", "task-4");
    _runStorage.startTask(jobName, jobRunId, workflowRunId, "step-3", "task-5");
    assertTrue(_runStorage.hasTasks(jobName, jobRunId, workflowRunId));

    workflowRunData = _runStorage.getWorkflowRunData(jobName, workflowRunId);
    assertEquals(String.valueOf(5),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    final String startTime = ValueFormatHelper.INSTANCE.formatDateTime(new Date());
    Thread.sleep(1000);
    final Map<String, String> properties = new HashMap<String, String>();
    properties.put(Task.PROPERTY_START_TIME, startTime);
    properties.put(Task.PROPERTY_END_TIME, ValueFormatHelper.INSTANCE.formatDateTime(new Date()));
    _runStorage.finishTask(jobName, jobRunId, workflowRunId, "step-1", "task-1", counters, properties);
    assertTrue(_runStorage.hasTasks(jobName, jobRunId, workflowRunId));

    workflowRunData = _runStorage.getWorkflowRunData(jobName, workflowRunId);
    assertEquals(String.valueOf(4),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    try {
      properties.put(Task.PROPERTY_START_TIME, ValueFormatHelper.INSTANCE.formatDateTime(new Date()));
      properties.put(Task.PROPERTY_END_TIME, ValueFormatHelper.INSTANCE.formatDateTime(new Date()));
      _runStorage.finishTask(jobName, jobRunId, workflowRunId, "step-1", "task-XXX", counters, properties);
      fail("expected exception due to wrong task");
    } catch (final RunStorageException e) {
      ; // expected
    }

    // add second workflow run
    final String workflowRunId2 = _runStorage.startWorkflowRun(jobName, jobRunId);
    assertEquals("2", workflowRunId2);
    jobRunData = _runStorage.getJobRunData(jobName, false);
    workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertEquals(String.valueOf(2),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));

    assertFalse(_runStorage.hasTasks(jobName, jobRunId, workflowRunId2));
    _runStorage.startTask(jobName, jobRunId, workflowRunId2, "step-1", "task-1");
    _runStorage.startTask(jobName, jobRunId, workflowRunId2, "step-1", "task-2");
    assertTrue(_runStorage.hasTasks(jobName, jobRunId, workflowRunId2));

    workflowRunData = _runStorage.getWorkflowRunData(jobName, workflowRunId2);
    assertEquals(String.valueOf(2),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    // remove all tasks of workflow run 1
    properties.put(Task.PROPERTY_START_TIME, ValueFormatHelper.INSTANCE.formatDateTime(new Date()));
    Thread.sleep(1000);
    final String endTime = ValueFormatHelper.INSTANCE.formatDateTime(new Date());
    properties.put(Task.PROPERTY_END_TIME, endTime);
    _runStorage.finishTask(jobName, jobRunId, workflowRunId, "step-1", "task-2", counters, properties);
    properties.clear();
    properties.put(Task.PROPERTY_END_TIME, endTime);
    _runStorage.finishTask(jobName, jobRunId, workflowRunId, "step-2", "task-3", counters, properties);
    _runStorage.startTask(jobName, jobRunId, workflowRunId, "step-3", "task-6");
    _runStorage.finishTask(jobName, jobRunId, workflowRunId, "step-3", "task-4", counters, null);
    _runStorage.finishTask(jobName, jobRunId, workflowRunId, "step-3", "task-5", counters, null);
    _runStorage.finishTask(jobName, jobRunId, workflowRunId, "step-3", "task-6", counters, null);
    assertFalse(_runStorage.hasTasks(jobName, jobRunId, workflowRunId));

    workflowRunData = _runStorage.getWorkflowRunData(jobName, workflowRunId);
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    // test counters (see counter map above)
    final AnyMap step1Data = _runStorage.getWorkerData(jobName, "step-1");
    assertEquals(step1Data.getLongValue("counter-1"), Long.valueOf(6)); // we finished 2 tasks for this worker
    assertEquals(step1Data.getDoubleValue("counter-2"), 4.0); // we finished 2 tasks for this worker
    assertEquals(step1Data.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS), Long.valueOf(2));
    // start time is the first start time, end time is the last one
    assertEquals(startTime, step1Data.getStringValue(Task.PROPERTY_START_TIME));
    assertEquals(endTime, step1Data.getStringValue(Task.PROPERTY_END_TIME));
    final AnyMap step2Data = _runStorage.getWorkerData(jobName, "step-2");
    assertEquals(step2Data.getLongValue("counter-1"), Long.valueOf(3)); // we finished 1 task for this worker
    assertEquals(step2Data.getDoubleValue("counter-2"), 2.0); // we finished 1 task for this worker
    assertEquals(step2Data.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS), Long.valueOf(1));
    // we did not set an start time, only end time
    assertNull(step2Data.getStringValue(Task.PROPERTY_START_TIME));
    assertEquals(endTime, step2Data.getStringValue(Task.PROPERTY_END_TIME));
    final AnyMap step3Data = _runStorage.getWorkerData(jobName, "step-3");
    assertEquals(step3Data.getLongValue("counter-1"), Long.valueOf(9)); // we finished 3 tasks for this worker
    assertEquals(step3Data.getDoubleValue("counter-2"), 6.0); // we finished 3 tasks for this worker
    assertEquals(step3Data.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS), Long.valueOf(3));
    assertNull(step3Data.getStringValue(Task.PROPERTY_START_TIME));
    assertNull(step3Data.getStringValue(Task.PROPERTY_END_TIME));

    _runStorage.successfulWorkflowRun(jobName, jobRunId, workflowRunId);
    jobRunData = _runStorage.getJobRunData(jobName, false);
    workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);

    assertEquals(String.valueOf(1),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertEquals(String.valueOf(1),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));
    _runStorage.successfulWorkflowRun(jobName, jobRunId, workflowRunId2);
    jobRunData = _runStorage.getJobRunData(jobName, false);
    workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertEquals(String.valueOf(2),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));
  }

  /**
   * Tests failed workflow run.
   * 
   * @throws Exception
   *           error
   */
  public void testFailedWorkflowRun() throws Exception {
    final String jobName = "myJob";
    final String jobRunId = "4711";

    _runStorage.startJobRun(jobName, jobRunId, JobRunMode.STANDARD, createJobRunDef(jobName));
    AnyMap jobRunData = _runStorage.getJobRunData(jobName, false);
    AnyMap workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));

    final String workflowRunId1 = _runStorage.startWorkflowRun(jobName, jobRunId);
    final String workflowRunId2 = _runStorage.startWorkflowRun(jobName, jobRunId);
    final String workflowRunId3 = _runStorage.startWorkflowRun(jobName, jobRunId);

    _runStorage.failedWorkflowRun(jobName, jobRunId, workflowRunId1);
    _runStorage.successfulWorkflowRun(jobName, jobRunId, workflowRunId2);
    _runStorage.failedWorkflowRun(jobName, jobRunId, workflowRunId3);

    jobRunData = _runStorage.getJobRunData(jobName, false);
    workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertEquals(String.valueOf(0),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS));
    assertEquals(String.valueOf(3),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS));
    assertEquals(String.valueOf(1),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    assertEquals(String.valueOf(2),
      workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));
  }

  /**
   * Tests add/get transient bulks.
   * 
   * @throws Exception
   *           error
   */
  public void testTransientBulks() throws Exception {
    final String jobName = "myJob";
    final String jobRunId = "4711";

    _runStorage.startJobRun(jobName, jobRunId, JobRunMode.STANDARD, createJobRunDef(jobName));
    final String workflowRunId1 = _runStorage.startWorkflowRun(jobName, jobRunId);
    assertEquals(workflowRunId1, "1");
    final String workflowRunId2 = _runStorage.startWorkflowRun(jobName, jobRunId);
    assertEquals(workflowRunId2, "2");

    AnyMap workflowRunData1 = _runStorage.getWorkflowRunData(jobName, workflowRunId1);
    assertEquals(String.valueOf(0),
      workflowRunData1.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS));
    assertEquals(String.valueOf(0),
      workflowRunData1.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));
    AnyMap workflowRunData2 = _runStorage.getWorkflowRunData(jobName, workflowRunId2);
    assertEquals(String.valueOf(0),
      workflowRunData1.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS));
    assertEquals(String.valueOf(0),
      workflowRunData1.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).isEmpty());
    _runStorage.addTransientBulk(jobName, jobRunId, workflowRunId1, "bulk-1");
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).contains("bulk-1"));
    workflowRunData1 = _runStorage.getWorkflowRunData(jobName, workflowRunId1);
    assertEquals(String.valueOf(1),
      workflowRunData1.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    _runStorage.addTransientBulk(jobName, jobRunId, workflowRunId1, "bulk-2");
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).contains("bulk-1"));
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).contains("bulk-2"));
    workflowRunData1 = _runStorage.getWorkflowRunData(jobName, workflowRunId1);
    assertEquals(String.valueOf(2),
      workflowRunData1.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));
    workflowRunData2 = _runStorage.getWorkflowRunData(jobName, workflowRunId2);
    assertEquals(String.valueOf(0),
      workflowRunData2.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId2).isEmpty());
    _runStorage.addTransientBulk(jobName, jobRunId, workflowRunId2, "bulk-A");
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).contains("bulk-1"));
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).contains("bulk-2"));
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId2).contains("bulk-A"));
    _runStorage.addTransientBulk(jobName, jobRunId, workflowRunId2, "bulk-B");
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId2).contains("bulk-B"));
    workflowRunData1 = _runStorage.getWorkflowRunData(jobName, workflowRunId1);
    assertEquals(String.valueOf(2),
      workflowRunData1.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));
    workflowRunData2 = _runStorage.getWorkflowRunData(jobName, workflowRunId2);
    assertEquals(String.valueOf(2),
      workflowRunData2.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));

    _runStorage.addTransientBulk(jobName, jobRunId, workflowRunId1, "bulk-3");
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).contains("bulk-1"));
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).contains("bulk-2"));
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId1).contains("bulk-3"));
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId2).contains("bulk-A"));
    assertTrue(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId2).contains("bulk-B"));
    assertEquals(_runStorage.getTransientBulks(jobName, jobRunId, workflowRunId2).size(), 2);
    workflowRunData1 = _runStorage.getWorkflowRunData(jobName, workflowRunId1);
    assertEquals(String.valueOf(3),
      workflowRunData1.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));
    workflowRunData2 = _runStorage.getWorkflowRunData(jobName, workflowRunId2);
    assertEquals(String.valueOf(2),
      workflowRunData2.getStringValue(JobManagerConstants.DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS));
  }

  /**
   * test triggered jobs.
   */
  public void testTriggerBuckets() throws Exception {
    final String bucketId = "bucket[index=test]";
    assertTrue(_runStorage.getTriggeredJobs(bucketId).isEmpty());
    _runStorage.addJobTrigger(bucketId, "job1");
    _runStorage.addJobTrigger(bucketId, "job2");
    _runStorage.addJobTrigger(bucketId, "job3");
    Collection<String> jobNames = _runStorage.getTriggeredJobs(bucketId);
    assertEquals(3, jobNames.size());
    assertTrue(jobNames.contains("job1"));
    assertTrue(jobNames.contains("job2"));
    assertTrue(jobNames.contains("job3"));
    assertTrue(_runStorage.getTriggeredJobs("nobucket").isEmpty());
    _runStorage.removeJobTrigger(bucketId, "job1");
    _runStorage.removeJobTrigger(bucketId, "job3");
    jobNames = _runStorage.getTriggeredJobs(bucketId);
    assertEquals(1, jobNames.size());
    assertTrue(jobNames.contains("job2"));
    assertTrue(_runStorage.getTriggeredJobs("nobucket").isEmpty());
    _runStorage.removeJobTrigger(bucketId, "job2");
    assertTrue(_runStorage.getTriggeredJobs(bucketId).isEmpty());
    assertTrue(_runStorage.getTriggeredJobs("nobucket").isEmpty());
  }
}
