/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.io.IOException;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.exceptions.InvalidConfigException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.test.worker.TestComplexParametersWorker;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskworker.input.RecordInput;

/**
 * Test JobManager.finishTask() method.
 */
public class TestComplexParametersInJob extends JobManagerTestBase {

  /** worker name for initial worker (startAction worker). */
  private static final String INPUT_WORKER = "inputWorker";

  /** workflow name. */
  private static final String WORKFLOW_NAME = "testComplexParametersWorkflow";

  /** job name. */
  private static final String JOB_NAME = "testComplexParametersJob";

  /** store name. */
  private static final String STORE_NAME = "test";

  /** The object store service. */
  private ObjectStoreService _defStorage;

  /** the jobId (to be able to finish it when test is finished. */
  private String _jobId;

  /** ref. to the taskmanager. */
  private TaskManager _taskManager;

  /** to create BON from record. */
  private final IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  /** the job parameters. */
  private final AnyMap _parameters = DataFactory.DEFAULT.createAnyMap();

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _defStorage = getService(ObjectStoreService.class);
    for (final String storeName : _objectStoreService.getStoreNames()) {
      if ("jobmanager".equals(storeName)) {
        _objectStoreService.clearStore(storeName);
      } else {
        _objectStoreService.removeStore(storeName);
      }
    }
    _defStorage.ensureStore(STORE_NAME);

    _taskManager = getService(TaskManager.class);

    // default job
    final AnyMap jobAny = AccessAny.FACTORY.createAnyMap();
    jobAny.put("name", JOB_NAME);
    final AnyMap complexParametersWorkerMap = DataFactory.DEFAULT.createAnyMap();
    final AnyMap mapInSeq = DataFactory.DEFAULT.createAnyMap();
    final AnySeq seq = DataFactory.DEFAULT.createAnySeq();
    mapInSeq.put("a", "${aString}");
    mapInSeq.put("l", "${longMax}");
    mapInSeq.put("b", "${boolTrue}");
    mapInSeq.add("d", DataFactory.DEFAULT.createStringValue("${doubleMin}"));
    mapInSeq.add("d", DataFactory.DEFAULT.createStringValue("${doubleMax}"));
    seq.add(mapInSeq);
    seq.add("${aDate}");
    complexParametersWorkerMap.put("seq", seq);
    final AnyMap mapInMap = DataFactory.DEFAULT.createAnyMap();
    mapInMap.put("dateTime", "${aDateTime}");
    complexParametersWorkerMap.put("mapInMap", mapInMap);
    complexParametersWorkerMap.put("mapBloodyMap", DataFactory.DEFAULT.cloneAnyMap(mapInMap));
    final AnyMap crapMap = DataFactory.DEFAULT.createAnyMap();
    crapMap.put("l", "${longMax}");
    complexParametersWorkerMap.put("crapMap", crapMap);
    complexParametersWorkerMap.put("d", "${double2}");
    complexParametersWorkerMap.put("dMin", "${doubleMin}");
    final AnyMap yetAnotherMap = DataFactory.DEFAULT.createAnyMap();
    yetAnotherMap.put("key", "value");
    complexParametersWorkerMap.put("yetAnotherMap", yetAnotherMap);
    _parameters.put("map", complexParametersWorkerMap);
    _parameters.put("doubleMin", Double.MIN_VALUE);
    _parameters.put("doubleMax", Double.MAX_VALUE);
    _parameters.put("double2", 2.0);
    _parameters.put("longMax", Long.MAX_VALUE);
    _parameters.put("boolTrue", true);
    _parameters.put("aString", "a");
    _parameters.put("aDateTime", TestComplexParametersWorker.EXPECTED_MAP.getMap("mapInMap").getValue("dateTime"));
    _parameters.put("aDate", TestComplexParametersWorker.EXPECTED_MAP.getSeq("seq").getValue(1));
    _parameters.put("aDate", DataFactory.DEFAULT.createDateValue(new Date()));
    _parameters.put("tempStore", STORE_NAME);
    _parameters.put("store", STORE_NAME);
    jobAny.put("parameters", _parameters);
    jobAny.put("workflow", WORKFLOW_NAME);
    _defPersistence.addJob(new JobDefinition(jobAny));

    _jobId = _jobRunEngine.startJob(JOB_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
    // job has already been finished.
    // but if something went wrong, we should maybe try to cancel it:
    try {
      _jobRunEngine.cancelJob(JOB_NAME, _jobId);
    } catch (final JobManagerException e) {
      ; // ignore.
    }
    _defPersistence.removeJob(JOB_NAME);
  }

  /**
   * tests evaluation and access on complex worker parameters in job processing.
   * 
   * @throws Exception
   *           something went wrong.
   */
  public void testEvaluatedComplexParametersForWorker() throws Exception {
    final Task currentTask = _jobTaskProcessor.getInitialTask(INPUT_WORKER, JOB_NAME);
    assertNotNull(currentTask);
    final ResultDescription resultDescription =
      new ResultDescription(TaskCompletionStatus.SUCCESSFUL, null, null, null);
    createOutputObjects(currentTask);
    _taskManager.finishTask(INPUT_WORKER, currentTask.getTaskId(), resultDescription);
    _jobRunEngine.finishJob(JOB_NAME, _jobId);
    waitForJobRunCompleted(JOB_NAME, _jobId, 120000);
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(STORE_NAME, "finalBucket");
    assertFalse(storeObjects.isEmpty());
    int recordsFound = 0;
    for (final StoreObject object : storeObjects) {
      final RecordInput rinput =
        new RecordInput(new BulkInfo("outbulkBucket", STORE_NAME, object.getId()), _objectStoreService);
      try {
        Record record = null;
        do {
          record = rinput.getRecord();
          if (record != null) {
            ++recordsFound;
            final AnyMap addedOutputAttribute =
              record.getMetadata().getMap(TestComplexParametersWorker.OUTPUT_ATTRIBUTE);
            assertNotNull(addedOutputAttribute);
            // we cannot use equal since the conversion of expressions will leave strings in the maps,
            // although it could be accessed with the proper source type due to implicit conversion.
            assertEquals(TestComplexParametersWorker.EXPECTED_MAP.toString(), addedOutputAttribute.toString());
          }
        } while (record != null);
      } finally {
        rinput.close();
      }
    }
    assertTrue(recordsFound > 0);
  }

  /**
   * tests evaluation and access on complex worker parameters in job processing.
   * 
   * @throws Exception
   *           something went wrong.
   */
  public void testMissingComplexParametersForWorker() throws Exception {
    final String jobName = "testMissingComplexParametersForWorker";
    final AnyMap parameters = DataFactory.DEFAULT.cloneAnyMap(_parameters);
    parameters.remove("doubleMin");
    final AnyMap jobAny = AccessAny.FACTORY.createAnyMap();
    jobAny.put("name", jobName);
    jobAny.put("parameters", parameters);
    jobAny.put("workflow", WORKFLOW_NAME);
    try {
      _defPersistence.addJob(new JobDefinition(jobAny));
      fail("JobManagerException expected.");
    } catch (final InvalidConfigException e) {
      assertTrue(e.getCause() instanceof InvalidValueTypeException);
    }
  }

  /**
   * tests evaluation and access on complex worker parameters in job processing.
   * 
   * @throws Exception
   *           something went wrong.
   */
  public void testWrongComplexParametersForWorker() throws Exception {
    final String jobName = "testWrongComplexParametersForWorker";
    final AnyMap parameters = DataFactory.DEFAULT.cloneAnyMap(_parameters);
    parameters.getMap("map").getMap("mapInMap").put("dateTime", "123");
    final AnyMap jobAny = AccessAny.FACTORY.createAnyMap();
    jobAny.put("name", jobName);
    jobAny.put("parameters", parameters);
    jobAny.put("workflow", WORKFLOW_NAME);
    try {
      _defPersistence.addJob(new JobDefinition(jobAny));
      fail("JobManagerException expected.");
    } catch (final InvalidConfigException e) {
      assertTrue(e.getCause() instanceof InvalidValueTypeException);
    }
  }

  /**
   * tests evaluation and access on complex worker parameters in job processing.
   * 
   * @throws Exception
   *           something went wrong.
   */
  public void testWrongComplexParametersWithVariableNameForWorker() throws Exception {
    final String jobName = "testWrongComplexParametersWithVariableNameForWorker";
    final AnyMap parameters = DataFactory.DEFAULT.cloneAnyMap(_parameters);
    parameters.getMap("map").getMap("mapBloodyMap").put("dateTime", "123");
    final AnyMap jobAny = AccessAny.FACTORY.createAnyMap();
    jobAny.put("name", jobName);
    jobAny.put("parameters", parameters);
    jobAny.put("workflow", WORKFLOW_NAME);
    try {
      _defPersistence.addJob(new JobDefinition(jobAny));
      fail("JobManagerException expected.");
    } catch (final InvalidConfigException e) {
      assertTrue(e.getCause() instanceof InvalidValueTypeException);
    }
  }

  /**
   * Writes an pseudo bulk.
   * 
   * @param currentTask
   *          the current task, for which the output buckets are to be filled.
   * @throws ObjectStoreException
   *           creation of dummy data failed.
   * @throws IOException
   *           cannot convert record to bon.
   */
  private void createOutputObjects(final Task currentTask) throws ObjectStoreException, IOException {
    int i = 0;
    for (final List<BulkInfo> bulkInfoList : currentTask.getOutputBulks().values()) {
      for (final BulkInfo bulkInfo : bulkInfoList) {
        final Record record = DataFactory.DEFAULT.createRecord(String.valueOf(i++));
        record.getMetadata().put("dummy", "dummy value");
        _defStorage.putObject(bulkInfo.getStoreName(), bulkInfo.getObjectName(),
          _ipcRecordWriter.writeBinaryObject(record));
      }
    }
  }
}
