/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.common.definitions.ParameterDefinition;
import org.eclipse.smila.common.definitions.ValueExpression;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition.Mode;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Input;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Output;
import org.eclipse.smila.jobmanager.definitions.WorkflowAction;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.jobmanager.persistence.PermanentStorage;

/**
 * Test for reading of preconfigured JobManager definitions.
 */
public class TestDefinitionPersistence extends JobManagerTestBase {

  /** PermanentStorage service under test. */
  private PermanentStorage _permStorage;

  /** {@inheritDoc} get permanent storage service. */
  @Override
  protected void setUp() throws Exception {
    _permStorage = getService(PermanentStorage.class);
    assertNotNull(_permStorage);
    super.setUp();
  }

  /**
   * Reads data object type definitions from json file and checks if all values are set.
   * 
   * @throws Exception
   *           exception if something goes wrong during reading data object type definition.
   */
  public void testReadPredefinedDataObjectTypeDefinitions() throws Exception {

    // no available data object type definition with that name
    assertNull(_defPersistence.getDataObjectType("NA"));

    // check if the expected object is available
    assertNotNull(_defPersistence.getDataObjectType("recordBulks"));
    final DataObjectTypeDefinition recordBulkDot = _defPersistence.getDataObjectType("recordBulks");
    // persistent
    assertEquals(new ValueExpression("${store}"), recordBulkDot.getStore(Mode.PERSISTENT));
    assertEquals(new ValueExpression("${_bucketName}/${_uuid}"), recordBulkDot.getObject(Mode.PERSISTENT));
    // transient
    assertEquals(new ValueExpression("${tempStore}"), recordBulkDot.getStore(Mode.TRANSIENT));
    assertEquals(new ValueExpression("${_bucketName}/${_uuid}"), recordBulkDot.getObject(Mode.TRANSIENT));
    // read only
    assertEquals(true, recordBulkDot.isReadOnly());
  }

  /**
   * Reads data object type definitions from json file and checks if all values are set.
   * 
   * @throws Exception
   *           exception if something goes wrong during reading data object type definition.
   */
  public void testReadPredefinedBuckets() throws Exception {
    assertTrue(_defPersistence.getBuckets().size() > 0);

    final BucketDefinition indexBucket = _defPersistence.getBucket("finalBucket");
    assertEquals("finalBucket", indexBucket.getName());
    assertEquals("recordBulks", indexBucket.getDataObjectType());
    assertEquals(true, indexBucket.isReadOnly());

    assertNull(_defPersistence.getBucket("imnothere"));
  }

  /**
   * test access to job run data in Definition Storage.
   * 
   * @throws Exception
   *           test fails
   */
  public void testJobRunDataManagement() throws Exception {
    final String jobName = "jobRunDataTest";
    assertTrue(_permStorage.getJobRunIds(jobName).isEmpty());
    final String jobRunId1 = "jobrun1";
    final AnyMap jobRunData1 = DataFactory.DEFAULT.createAnyMap();
    jobRunData1.put(JobManagerConstants.DATA_JOB_ID, jobRunId1);
    jobRunData1.put(JobManagerConstants.DATA_JOB_STATE, JobState.FAILED.name());
    final String jobRunId2 = "jobrun2";
    final AnyMap jobRunData2 = DataFactory.DEFAULT.createAnyMap();
    jobRunData2.put(JobManagerConstants.DATA_JOB_ID, jobRunId2);
    jobRunData2.put(JobManagerConstants.DATA_JOB_STATE, JobState.SUCCEEDED.name());
    _permStorage.storeJobRun(jobName, jobRunId1, jobRunData1);
    _permStorage.storeJobRun(jobName, jobRunId2, jobRunData2);
    assertTrue(_permStorage.containsJobRun(jobName, jobRunId1));
    assertTrue(_permStorage.containsJobRun(jobName, jobRunId2));
    assertFalse(_permStorage.containsJobRun(jobName, jobName));
    Iterator<String> storedIds = _permStorage.getJobRunIds(jobName).iterator();
    assertTrue(storedIds.hasNext());
    assertEquals(jobRunId1, storedIds.next());
    assertTrue(storedIds.hasNext());
    assertEquals(jobRunId2, storedIds.next());
    assertFalse(storedIds.hasNext());
    final AnyMap readData1 = _permStorage.getJobRunData(jobName, jobRunId1);
    assertEquals(jobRunId1, readData1.getStringValue(JobManagerConstants.DATA_JOB_ID));
    assertEquals(JobState.FAILED.name(), readData1.getStringValue(JobManagerConstants.DATA_JOB_STATE));
    _permStorage.deleteJobRunData(jobName, jobRunId1);
    assertFalse(_permStorage.containsJobRun(jobName, jobRunId1));
    assertTrue(_permStorage.containsJobRun(jobName, jobRunId2));
    storedIds = _permStorage.getJobRunIds(jobName).iterator();
    assertTrue(storedIds.hasNext());
    assertEquals(jobRunId2, storedIds.next());
    assertFalse(storedIds.hasNext());
    final AnyMap readData2 = _permStorage.getJobRunData(jobName, jobRunId2);
    assertEquals(jobRunId2, readData2.getStringValue(JobManagerConstants.DATA_JOB_ID));
    assertEquals(JobState.SUCCEEDED.name(), readData2.getStringValue(JobManagerConstants.DATA_JOB_STATE));
    _permStorage.deleteJobRunData(jobName, jobRunId2);
    assertFalse(_permStorage.containsJobRun(jobName, jobRunId1));
    assertFalse(_permStorage.containsJobRun(jobName, jobRunId2));
    assertTrue(_permStorage.getJobRunIds(jobName).isEmpty());
    _permStorage.deleteJobRunData(jobName, jobRunId2);
  }

  /**
   * reads a predefined worker and checks its attributes.
   */
  public void testReadPredefinedWorker() {
    final String workerName = "testOptionalParametersWorker";
    final String outputName = "output";
    final WorkerDefinition wDef = _defPersistence.getWorker(workerName);
    assertTrue(wDef.isReadOnly());
    assertEquals(workerName, wDef.getName());
    assertValues(wDef.getModes(), WorkerDefinition.Mode.BULKSOURCE, WorkerDefinition.Mode.AUTOCOMMIT);
    final Collection<Input> inputs = wDef.getInput();
    assertEquals(0, inputs.size());
    final Collection<Output> outputs = wDef.getOutput();
    assertEquals(1, outputs.size());
    assertEquals(wDef.getOutput(outputName), outputs.iterator().next());
    assertEquals(outputName, wDef.getOutput(outputName).getName());
    final List<ParameterDefinition> params = wDef.getParameters();
    assertEquals(2, params.size());
    for (final ParameterDefinition param : params) {
      if (param.getName().equals("workerParameter")) {
        assertFalse(param.isOptional());
      } else if (param.getName().equals("optionalParameter")) {
        assertTrue(param.isOptional());
      } else {
        fail("unexpected parameter: " + param);
      }
    }

    // check ranges.
    final WorkerDefinition pDef = _defPersistence.getWorker("testParameterRanges");
    List<ParameterDefinition> par = pDef.getParametersbyRange(null);
    assertEquals(1, par.size());
    assertEquals("noRange", par.get(0).getName());
    par = pDef.getParametersbyRange("jobName");
    assertEquals(1, par.size());
    assertEquals("jobName", par.get(0).getName());
    par = pDef.getParametersbyRange("doubleRange");
    assertEquals(1, par.size());
    assertEquals("someRange", par.get(0).getName());
    par = pDef.getParametersbyRange("noRange");
    assertTrue(par.isEmpty());
  }

  public void testWorkerWithComplexParametersToAnyWithDetails() {
    final String workerName = "workerWithComplexParameters";
    final WorkerDefinition wDef = _defPersistence.getWorker(workerName);
    final AnySeq parameters = wDef.toAny(true).getSeq("parameters");
    checkParameter(parameters.getMap(0), 3, "stringParam", null, true, null, false, false, true);
    checkParameter(parameters.getMap(1), 3, "booleanParam", "boolean", null, null, false, false, true);
    final AnyMap enumParam =
      checkParameter(parameters.getMap(2), 6, "enumParam", "string", true, null, false, true, true);
    assertEquals("val1", enumParam.getStringValue("default"));
    assertTrue(enumParam.get("values").isSeq());
    final AnyMap mapParam =
      checkParameter(parameters.getMap(3), 4, "mapParam", "map", null, null, true, false, true);
    final AnySeq entries = mapParam.getSeq("entries");
    checkParameter(entries.getMap(0), 2, "key1", "string", null, null, false, false, false);
    checkParameter(entries.getMap(1), 4, "key2", "string", null, true, false, false, true);
    checkParameter(entries.getMap(2), 6, "key3", "map", true, false, true, false, true);
    checkParameter(parameters.getMap(4), 4, "sequenceOfStringsParam", "string", null, true, false, false, true);
    checkParameter(parameters.getMap(5), 3, "<something>", "string", null, null, false, false, true);
    checkParameter(parameters.getMap(6), 4, "anyParam", "any", true, null, false, false, true);
  }

  public void testWorkerWithComplexParametersToAnyWithoutDetails() {
    final String workerName = "workerWithComplexParameters";
    final WorkerDefinition wDef = _defPersistence.getWorker(workerName);
    final AnySeq parameters = wDef.toAny(false).getSeq("parameters");
    checkParameter(parameters.getMap(0), 2, "stringParam", null, true, null, false, false, false);
    checkParameter(parameters.getMap(1), 2, "booleanParam", "boolean", null, null, false, false, false);
    final AnyMap enumParam =
      checkParameter(parameters.getMap(2), 4, "enumParam", "string", true, null, false, true, false);
    assertFalse(enumParam.containsKey("default"));
    assertTrue(enumParam.get("values").isSeq());
    final AnyMap mapParam =
      checkParameter(parameters.getMap(3), 3, "mapParam", "map", null, null, true, false, false);
    final AnySeq entries = mapParam.getSeq("entries");
    checkParameter(entries.getMap(0), 2, "key1", "string", null, null, false, false, false);
    checkParameter(entries.getMap(1), 3, "key2", "string", null, true, false, false, false);
    checkParameter(entries.getMap(2), 5, "key3", "map", true, false, true, false, false);
    checkParameter(parameters.getMap(4), 3, "sequenceOfStringsParam", "string", null, true, false, false, false);
    checkParameter(parameters.getMap(5), 2, "<something>", "string", null, null, false, false, false);
    checkParameter(parameters.getMap(6), 3, "anyParam", "any", true, null, false, false, false);
  }

  private AnyMap checkParameter(final AnyMap param, final int size, final String name, final String type,
    final Boolean optional, final Boolean multiple, final boolean hasEntries, final boolean hasValues,
    final boolean hasDescription) {
    assertEquals(size, param.size());
    assertEquals(name, param.getStringValue("name"));
    assertEquals(type, param.getStringValue("type"));
    assertEquals(optional, param.getBooleanValue("optional"));
    assertEquals(multiple, param.getBooleanValue("multi"));
    assertEquals(hasEntries, param.containsKey("entries"));
    assertEquals(hasValues, param.containsKey("values"));
    assertEquals(hasDescription, param.containsKey("description"));
    return param;
  }

  /**
   * reads a predefined workflow and checks some of its attributes.
   * 
   * @throws PersistenceException
   */
  public void testReadPredefinedWorkflow() throws PersistenceException {
    final String worklowName = "optionalParamWorkflow";
    final WorkflowDefinition wDef = _defPersistence.getWorkflow(worklowName);
    assertTrue(wDef.isReadOnly());
    assertEquals(worklowName, wDef.getName());
    assertNull(wDef.getActions());
    final WorkflowAction startAction = wDef.getStartAction();
    assertNotNull(startAction);
    assertEquals("testOptionalParametersWorker", startAction.getWorker());
    assertNull(startAction.getInput());
    final Map<String, String> outputMap = startAction.getOutput();
    assertEquals(1, outputMap.size());
    assertEquals("triggeredBucket", outputMap.get("output"));
  }

  /**
   * reads a predefined job and checks some of its attributes.
   */
  public void testReadPredefinedJobs() throws PersistenceException {
    final String jobName = "testDummyJob";
    final JobDefinition jDef = _defPersistence.getJob(jobName);
    assertTrue(jDef.isReadOnly());
    assertEquals(jobName, jDef.getName());
    assertEquals("testWorkflow", jDef.getWorkflow());
    assertEquals(3, jDef.getParameters().size());
    assertEquals("temp", jDef.getParameters().getStringValue("tempStore"));
    assertEquals("teststore", jDef.getParameters().getStringValue("store"));
    assertEquals("dummy", jDef.getParameters().getStringValue("workerParameter"));

    final String jobName2 = "testDummyJob2";
    final JobDefinition jDef2 = _defPersistence.getJob(jobName2);
    assertTrue(jDef2.isReadOnly());
    assertEquals(jobName2, jDef2.getName());
    assertEquals("testWorkflow", jDef2.getWorkflow());
    assertEquals(3, jDef2.getParameters().size());
    assertEquals("temp2", jDef2.getParameters().getStringValue("tempStore"));
    assertEquals("teststore2", jDef2.getParameters().getStringValue("store"));
    assertEquals("dummy2", jDef2.getParameters().getStringValue("workerParameter"));
  }
}
